//
//  BSAltImgPreviewer.m
//  Bardiche
//
//  Created by Tsutomu Sawada on 05/12/16.
//  @modified 06/03/14 - 1.0.1
//  @modified 07/11/16 - 1.0.2
//  @modified 09/02/22 - 1.0.3
//  @modified 11/07/29 - 1.0.4
//  @modified 14/01/07 - 1.0.5
//  Copyright 2005-2014 BathyScaphe Project. All rights reserved.
//  encoding="UTF-8"
//

#import "BSAltImgPreviewer.h"

@interface BSAltImgPreviewer(Private)
- (NSBundle *)applicationSupportBundle;

- (AppDefaults *)preferences;
- (void)setPreferences:(AppDefaults *)prefs;

- (NSString *)createPreviewFilesFolder;

- (NSString *)previewFolderPath;
- (void)setPreviewFolderPath:(NSString *)newPath;

- (BOOL)previewLinkWithWebBrowser:(NSArray *)urlsArray;
@end


@implementation BSAltImgPreviewer
- (id)initWithPreferences:(AppDefaults *)prefs
{
    if (self = [super init]) {
        [self setPreferences:prefs];

        [[NSNotificationCenter defaultCenter]
            addObserver:self
               selector:@selector(applicationWillTerminate:)
                   name:NSApplicationWillTerminateNotification
                 object:NSApp];

        [self setPreviewFolderPath:[self createPreviewFilesFolder]];
    }

    return self;
}

- (void)dealloc
{
    [self setPreviewFolderPath:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [self setPreferences:nil];
    [super dealloc];
}

- (BOOL)previewLink:(NSURL *)url
{
    return [self previewLinkWithWebBrowser:[NSArray arrayWithObject:url]];
}

- (BOOL)previewLinks:(NSArray *)urls
{
    return [self previewLinkWithWebBrowser:urls];
}

- (BOOL)validateLink:(NSURL *)url
{
    return YES;
}

- (IBAction)resetPreviewer:(id)sender
{
    [self setPreviewFolderPath:nil];
    [self setPreviewFolderPath:[self createPreviewFilesFolder]];
}

- (void)applicationWillTerminate:(NSNotification *)notification
{
    [self setPreviewFolderPath:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}
@end


@implementation BSAltImgPreviewer(Private)
- (NSBundle *)applicationSupportBundle
{
    static NSString *path = nil;
    if (!path) {
        NSArray *paths = NSSearchPathForDirectoriesInDomains(NSApplicationSupportDirectory, NSUserDomainMask, YES);
        NSString *appName = [[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleExecutable"];
        if (paths && ([paths count] > 0)) {
            path = [[[paths objectAtIndex:0] stringByAppendingPathComponent:appName] retain];
        }
    }
    return path ? [NSBundle bundleWithPath:path] : nil;
}

- (AppDefaults *)preferences
{
    return m_preferences;
}

- (void)setPreferences:(AppDefaults *)prefs
{
    [prefs retain];
    [m_preferences release];
    m_preferences = prefs;
}

- (NSString *)createPreviewFilesFolder
{
    NSString *path;
    NSString *tmpDir = [NSTemporaryDirectory() stringByAppendingPathComponent:@"Bardiche-XXXXXX"];

    char *cTmpDir = strdup([tmpDir fileSystemRepresentation]);

    mkdtemp(cTmpDir);
    path = [[NSFileManager defaultManager] stringWithFileSystemRepresentation:cTmpDir length:strlen(cTmpDir)];

    free(cTmpDir);

    return path;
}

- (NSString *)previewFolderPath
{
    return m_tmpFolderPath;
}

- (void)setPreviewFolderPath:(NSString *)newPath
{
    if (!newPath && m_tmpFolderPath) {
        [[NSFileManager defaultManager] removeItemAtPath:m_tmpFolderPath error:nil];
    }

    [newPath retain];
    [m_tmpFolderPath release];
    m_tmpFolderPath = newPath;
}

- (NSString *)previewSourcePathForName:(NSString *)resourceName type:(NSString *)aType
{
    NSString *path;

    path = [[self applicationSupportBundle] pathForResource:resourceName ofType:aType inDirectory:PREVIEW_DIR];
    if (path) {
        return path;
    }

    return [[NSBundle bundleForClass:[self class]] pathForResource:resourceName ofType:aType inDirectory:PREVIEW_DIR];
}

- (NSURL *)createPreviewFileFromSource:(NSString *)source previewURL:(NSURL *)url
{
    static NSUInteger countNumber = 0;

    NSString *filePath;
    NSString *htmlSource;
    BOOL writeSuccess;
    NSError *error;

    htmlSource = [source stringByReplacingOccurrencesOfString:PREVIEW_URL_KEY withString:[url absoluteString]];

    // create or replace temporary file
    countNumber++;
    filePath = [[self previewFolderPath] stringByAppendingPathComponent:[NSString stringWithFormat:@"%@%lu", PREVIEW_SRC, (unsigned long)countNumber]];
    filePath = [filePath stringByAppendingPathExtension:PREVIEW_TYPE];

    writeSuccess = [htmlSource writeToFile:filePath atomically:YES encoding:PREVIEW_ENC error:&error];
    if (writeSuccess) {
        return [NSURL fileURLWithPath:filePath];
    } else {
        if (error) {
            NSLog(@"** Bardiche ** Can't write html source file to %@. reason:%@ (%@ %ld)", filePath, [error localizedDescription], [error domain], (long)[error code]);
        }
    }

    return nil;
}

- (BOOL)launchWebBrowser:(NSArray *)urlsArray inBackground:(BOOL)flag
{
    NSURL *browserURL = [[NSWorkspace sharedWorkspace] URLForApplicationToOpenURL:[NSURL URLWithString:@"http://"]];
    NSString *bundleIdentifier = [[NSBundle bundleWithURL:browserURL] bundleIdentifier];
    
    NSWorkspaceLaunchOptions options = flag ? (NSWorkspaceLaunchDefault|NSWorkspaceLaunchWithoutActivation) : NSWorkspaceLaunchDefault;
    
    return [[NSWorkspace sharedWorkspace] openURLs:urlsArray withAppBundleIdentifier:bundleIdentifier options:options additionalEventParamDescriptor:nil launchIdentifiers:nil];
}

- (BOOL)previewLinkWithWebBrowser:(NSArray *)urlsArray
{
    NSString *templatePath;
    NSError *error;
    NSString *source;
    NSString *previewDirectoryPath;
    NSMutableArray *array;
    NSURL *fileURL;

    templatePath = [self previewSourcePathForName:PREVIEW_SRC type:PREVIEW_TYPE];
    if (!templatePath) {
        return NO;
    }

    source = [NSString stringWithContentsOfFile:templatePath encoding:PREVIEW_ENC error:&error];
    if (!source) {
        if (error) {
            NSLog(@"** Bardiche ** Can't read html source file at %@. reason:%@ (%@ %ld)", templatePath, [error localizedDescription], [error domain], (long)[error code]);
        }
        return NO;
    }

    // replace keyword to absolute path
    previewDirectoryPath = [templatePath stringByDeletingLastPathComponent];
    source = [source stringByReplacingOccurrencesOfString:PREVIEW_DIR_KEY withString:previewDirectoryPath];

    array = [NSMutableArray arrayWithCapacity:[urlsArray count]];
    for (NSURL *url in urlsArray) {
        fileURL = [self createPreviewFileFromSource:source previewURL:url];
        if (fileURL) {
            [array addObject:fileURL];
        }
    }
    if ([array count] == 0) {
        return NO;
    }
    return [self launchWebBrowser:array inBackground:[[self preferences] openInBg]];
}
@end
