/*
Copyright 2009 senju@users.sourceforge.jp

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 */
package net.excentrics.bandWidth;

import java.util.LinkedList;
import java.util.ListIterator;

/**
 * 帯域制御コントロール用のクラス
 * 
 */
class BandWidthController {
	private static final long defaultCycleInMils = 100; // デフォルトのサイクル時間。ミリ秒で設定。
	/** 現在の転送速度を調べる時に何個のsizeList内の要素を使うか。 **/
	private static final int speedCount = 70;

	/** 基準となるinterval **/
	private long cycle;
	/** 1サイクル毎に何バイト送出するべきか */
	private int bytePerCycle = 0;
	/** bit per second **/
	private double bandWidth = 0;

	/** 送出サイズの履歴 **/
	private final LinkedList<Integer> sizeList;
	private final LinkedList<Long> timeList;

	public BandWidthController() {
		this.timeList = new LinkedList<Long>();
		this.sizeList = new LinkedList<Integer>();
		this.cycle = defaultCycleInMils;
		setBytePerCycle();
	}

	/**
	 * 帯域制限に必要な情報を計算し返す。
	 * 
	 * @return 以下の2つの値を設定したSizeAndIntervalを返す。
	 *         <ul>
	 *         <li>書き込まれるべきサイズ(バイト)。 書き込むサイズに制限が必要ない場合は-1を返す。
	 *         <li>また、次の書き込みまでの待機時間をミリ秒で返す。
	 *         </ul>
	 */
	private SizeAndInterval calculationSizeForWrite(int bufferSize) {
		// 帯域制限を行わない場合
		if (this.bandWidth == 0) {
			return new SizeAndInterval(-1, 0);
		}
		// 初回呼び出しの場合
		if (this.timeList.isEmpty()) {
			return new SizeAndInterval(this.bytePerCycle, this.cycle);
		}

		final double currentBps = getCurrentBps();

		// 現在の速度がbandWidthより出ている場合は書き込まない。
		if (currentBps > this.bandWidth) {
			return new SizeAndInterval(0, this.cycle);
		}
		return new SizeAndInterval(this.bytePerCycle, this.cycle);

	}

	/**
	 * 現在の転送速度を求める(送出開始以降すべてではなく、過去数秒の記録による)
	 * 
	 * @return 現在の転送速度
	 */
	private double getCurrentBps() {
		// speedCount個のサイズの合計を求め、現在の速度を求める。
		final int firstIndex = Math.max(this.sizeList.size() - speedCount, 0);
		final ListIterator<Integer> li = this.sizeList.listIterator(firstIndex);
		int sizeInByte = 0;
		while (li.hasNext()) {
			final Integer s = li.next();
			sizeInByte += s.intValue();
		}
		final long startTime = this.timeList.get(firstIndex);
		final long currentTime = System.currentTimeMillis();
		long intervalInMillis = currentTime - startTime;
		// 最低1ミリ秒以上の待機時間を確保(時計の設定変更等の対応)
		intervalInMillis = Math.max(1, intervalInMillis);
		final double intervalInSecond = intervalInMillis / 1000.0;
		final double currentBps = sizeInByte * 8 / intervalInSecond;
		return currentBps;
	}

	/**
	 * 送出前に呼び出すメソッド。送出すべきサイズとその後何ミリ秒待機すべきかを返す
	 * 
	 * @param bufferSize
	 *            使用しているバッファのサイズ
	 * @return 書き込むべきサイズ(byte)、および次回呼び出しまでの間隔(ミリ秒)を返す
	 */
	public SizeAndInterval preWrite(int bufferSize) {
		final SizeAndInterval si = calculationSizeForWrite(bufferSize);
		this.timeList.add(System.currentTimeMillis());

		return si;
	}

	/**
	 * 送出を行ったあとに呼び出すべきメソッド
	 * 
	 * @param writtenSize
	 *            実際に書き込まれたサイズ
	 * @throws AsymmetricalCallException
	 *             preWriteを呼び出す前にpostWriteが呼ばれた時
	 */
	public void postWrite(int writtenSize) throws AsymmetricalCallException {
		this.sizeList.add(writtenSize);
		removeListItems();
		if (this.sizeList.size() != this.timeList.size()) {
			// TODO: 文字列のを設定ファイルに
			throw new AsymmetricalCallException(
					"preWrite and postWrite called asymmetric.");
		}
	}

	/**
	 * 時刻とサイズのリストを1部削除する
	 */
	private void removeListItems() {
		// TODO:magic number
		if (this.timeList.size() > 100) {
			// 100個を越えたら20個削除する(毎回1つずつ消していくと遅いので)
			for (int i = 0; i < 20; i++) {
				this.timeList.remove();
				this.sizeList.remove();
			}
		}
	}

	/**
	 * 1サイクルあたり何バイト書き出すべきか算出する
	 */
	private void setBytePerCycle() {
		// bit per second -> bit per milsec -> bpms * cycle -> byte per mils
		this.bytePerCycle = (int) (this.bandWidth / 1000 * this.cycle / 8);
	}

	/**
	 * 設定されている1サイクルあたりの時間を返す
	 * 
	 * @return サイクル時間(ミリ秒)
	 */
	public long getCycle() {
		return this.cycle;
	}

	/**
	 * 1サイクルあたりの時間を設定する。また、新しい値で1サイクルあたりの送出バイト数を計算しなおす。
	 * 
	 * @param cycle
	 *            サイクル時間(ミリ秒)
	 */
	public void setCycle(long cycle) {
		this.cycle = cycle;
		setBytePerCycle();
	}

	/**
	 * 現在設定されている制限帯域を返す
	 * 
	 * @return 帯域(bps)
	 */
	public double getBandWidth() {
		return this.bandWidth;
	}

	/**
	 * @param bandWidth
	 *            帯域をbps(bit per second) で与える。0が与えられた場合帯域制限を行わない。
	 */
	public void setBandWidth(double bandWidth) {
		this.bandWidth = bandWidth;
		this.sizeList.clear();
		this.timeList.clear();
		setBytePerCycle();
	}

}
