/*
**      V60 + 68k + 4xTGP
*/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "machine/eeprom.h"
#include "system16.h"
#include "vidhrdw/segaic24.h"
#include "cpu/m68000/m68k.h"

WRITE16_HANDLER( model1_paletteram_w );
VIDEO_START(model1);
VIDEO_UPDATE(model1);
VIDEO_UPDATE(model1a);
VIDEO_EOF(model1);
extern UINT16 *model1_display_list0, *model1_display_list1;
extern UINT16 *model1_color_xlat;
READ16_HANDLER( model1_listctl_r );
WRITE16_HANDLER( model1_listctl_w );

READ16_HANDLER( model1_tgp_copro_r );
WRITE16_HANDLER( model1_tgp_copro_w );
READ16_HANDLER( model1_tgp_copro_adr_r );
WRITE16_HANDLER( model1_tgp_copro_adr_w );
READ16_HANDLER( model1_tgp_copro_ram_r );
WRITE16_HANDLER( model1_tgp_copro_ram_w );

static int model1_sound_irq;

void model1_tgp_reset(int swa);

static READ16_HANDLER( io_r )
{
	if(offset < 0x8)
		return readinputport(offset);
	if(offset < 0x10) {
		offset -= 0x8;
		if(offset < 3)
			return readinputport(offset+8) | 0xff00;
		return 0xff;
	}

	logerror("IOR: %02x\n", offset);
	return 0xffff;
}

static READ16_HANDLER( fifoin_status_r )
{
	return 0xffff;
}

static WRITE16_HANDLER( bank_w )
{
	if(ACCESSING_LSB) {
		switch(data & 0xf) {
		case 0x1: // 100000-1fffff data roms banking
			cpu_setbank(1, memory_region(REGION_CPU1) + 0x1000000 + 0x100000*((data >> 4) & 0xf));
			logerror("BANK %x\n", 0x1000000 + 0x100000*((data >> 4) & 0xf));
			break;
		case 0x2: // 200000-2fffff data roms banking (unused, all known games have only one bank)
			break;
		case 0xf: // f00000-ffffff program roms banking (unused, all known games have only one bank)
			break;
		}
	}
}


static int last_irq;

static void irq_raise(int level)
{
	//	logerror("irq: raising %d\n", level);
	//	irq_status |= (1 << level);
	last_irq = level;
	cpu_set_irq_line(0, 0, HOLD_LINE);
}

static int irq_callback(int irqline)
{
	return last_irq;
}
// vf
// 1 = fe3ed4
// 3 = fe3f5c
// other = fe3ec8 / fe3ebc

// vr
// 1 = fe02bc
// other = f302a4 / fe02b0

// swa
// 1 = ff504
// 3 = ff54c
// other = ff568/ff574

static void irq_init(void)
{
	cpu_set_irq_line(0, 0, CLEAR_LINE);
	cpu_set_irq_callback(0, irq_callback);
}

extern void tgp_tick(void);
static INTERRUPT_GEN(model1_interrupt)
{
	if (cpu_getiloops())
	{
		irq_raise(1);
		tgp_tick();
	}
	else
	{
		irq_raise(model1_sound_irq);
	}
}

static MACHINE_INIT(model1)
{
	cpu_setbank(1, memory_region(REGION_CPU1) + 0x1000000);
	irq_init();
 	model1_tgp_reset(!strcmp(Machine->gamedrv->name, "swa") || !strcmp(Machine->gamedrv->name, "wingwar") || !strcmp(Machine->gamedrv->name, "wingwara"));
	if (!strcmp(Machine->gamedrv->name, "swa"))
	{
		model1_sound_irq = 0;
	}
	else
	{
		model1_sound_irq = 3;
	}
}

static READ16_HANDLER( network_ctl_r )
{
	if(offset)
		return 0x40;
	else
		return 0x00;
}

static WRITE16_HANDLER( network_ctl_w )
{
}

static WRITE16_HANDLER(md1_w)
{
	extern int model1_dump;
	COMBINE_DATA(model1_display_list1+offset);
	if(0 && offset)
		return;
	if(1 && model1_dump)
		logerror("TGP: md1_w %x, %04x @ %04x (%x)\n", offset, data, mem_mask, activecpu_get_pc());
}

static WRITE16_HANDLER(md0_w)
{
	extern int model1_dump;
	COMBINE_DATA(model1_display_list0+offset);
	if(0 && offset)
		return;
	if(1 && model1_dump)
		logerror("TGP: md0_w %x, %04x @ %04x (%x)\n", offset, data, mem_mask, activecpu_get_pc());
}

static WRITE16_HANDLER(p_w)
{
	UINT16 old = paletteram16[offset];
	paletteram16_xBBBBBGGGGGRRRRR_word_w(offset, data, mem_mask);
	if(0 && paletteram16[offset] != old)
		logerror("XVIDEO: p_w %x, %04x @ %04x (%x)\n", offset, data, mem_mask, activecpu_get_pc());
}

static UINT16 *mr;
static WRITE16_HANDLER(mr_w)
{
	COMBINE_DATA(mr+offset);
	if(0 && offset == 0x1138/2)
		logerror("MR.w %x, %04x @ %04x (%x)\n", offset*2+0x500000, data, mem_mask, activecpu_get_pc());
}

static UINT16 *mr2;
static WRITE16_HANDLER(mr2_w)
{
	COMBINE_DATA(mr2+offset);
#if 0
	if(0 && offset == 0x6e8/2) {
		logerror("MR.w %x, %04x @ %04x (%x)\n", offset*2+0x400000, data, mem_mask, activecpu_get_pc());
	}
	if(offset/2 == 0x3680/4)
		logerror("MW f80[r25], %04x%04x (%x)\n", mr2[0x3680/2+1], mr2[0x3680/2], activecpu_get_pc());
	if(offset/2 == 0x06ca/4)
		logerror("MW fca[r19], %04x%04x (%x)\n", mr2[0x06ca/2+1], mr2[0x06ca/2], activecpu_get_pc());
	if(offset/2 == 0x1eca/4)
		logerror("MW fca[r22], %04x%04x (%x)\n", mr2[0x1eca/2+1], mr2[0x1eca/2], activecpu_get_pc());
#endif

	// wingwar scene position, pc=e1ce -> d735
	if(offset/2 == 0x1f08/4)
		logerror("MW  8[r10], %f (%x)\n", *(float *)(mr2+0x1f08/2), activecpu_get_pc());
	if(offset/2 == 0x1f0c/4)
		logerror("MW  c[r10], %f (%x)\n", *(float *)(mr2+0x1f0c/2), activecpu_get_pc());
	if(offset/2 == 0x1f10/4)
		logerror("MW 10[r10], %f (%x)\n", *(float *)(mr2+0x1f10/2), activecpu_get_pc());
}

static int to_68k;

static READ16_HANDLER( snd_68k_ready_r )
{
	int sr = cpunum_get_reg(1, M68K_REG_SR);

	if ((sr & 0x0700) > 0x0100)
	{
		cpu_spinuntil_time(TIME_IN_USEC(40));
		return 0;	// not ready yet, interrupts disabled
	}
	
	return 0xff;
}

static WRITE16_HANDLER( snd_latch_to_68k_w )
{
	while (!snd_68k_ready_r(0, 0))
	{
		cpu_spinuntil_time(TIME_IN_USEC(40));
	}

	to_68k = data;
	
	cpu_set_irq_line(1, 2, HOLD_LINE);
	// give the 68k time to reply
	cpu_spinuntil_time(TIME_IN_USEC(40));
}

static MEMORY_READ16_START( readmem )
	{ 0x000000, 0x0fffff, MRA16_ROM },
	{ 0x100000, 0x1fffff, MRA16_BANK1 },
	{ 0x200000, 0x2fffff, MRA16_ROM },

	{ 0x400000, 0x40ffff, MRA16_RAM },
	{ 0x500000, 0x53ffff, MRA16_RAM },

	{ 0x600000, 0x60ffff, MRA16_RAM },
	{ 0x610000, 0x61ffff, MRA16_RAM },
	{ 0x680000, 0x680003, model1_listctl_r },

	{ 0x700000, 0x70ffff, sys24_tile_r },
	{ 0x780000, 0x7fffff, sys24_char_r },

	{ 0x900000, 0x903fff, MRA16_RAM },
	{ 0x910000, 0x91bfff, MRA16_RAM },            // Network ?

	{ 0xc00000, 0xc0003f, io_r },

	{ 0xc00040, 0xc00043, network_ctl_r },
    { 0xc00200, 0xc002ff, MRA16_RAM },             // ??

	{ 0xc40002, 0xc40003, snd_68k_ready_r },

	{ 0xd00000, 0xd00001, model1_tgp_copro_adr_r },
	{ 0xdc0000, 0xdc0003, fifoin_status_r },

	{ 0xfc0000, 0xffffff, MRA16_ROM },
MEMORY_END

static MEMORY_WRITE16_START( writemem )
	{ 0x000000, 0x0fffff, MWA16_ROM },
	{ 0x100000, 0x1fffff, MWA16_BANK1 },
	{ 0x200000, 0x2fffff, MWA16_ROM },

	{ 0x400000, 0x40ffff, mr2_w, &mr2 },
	{ 0x500000, 0x53ffff, mr_w, &mr },

	{ 0x600000, 0x60ffff, md0_w, &model1_display_list0 },
	{ 0x610000, 0x61ffff, md1_w, &model1_display_list1 },
	{ 0x680000, 0x680003, model1_listctl_w },

	{ 0x700000, 0x70ffff, sys24_tile_w },
	{ 0x720000, 0x720001, MWA16_NOP },		// Unknown, always 0
	{ 0x740000, 0x740001, MWA16_NOP },		// Horizontal synchronization register
	{ 0x760000, 0x760001, MWA16_NOP },		// Vertical synchronization register
	{ 0x770000, 0x770001, MWA16_NOP },		// Video synchronization switch
	{ 0x780000, 0x7fffff, sys24_char_w },

	{ 0x900000, 0x903fff, p_w, &paletteram16 },
	{ 0x910000, 0x91bfff, MWA16_RAM, &model1_color_xlat},

	{ 0xc00000, 0xc0003f, MWA16_NOP },

	{ 0xc00040, 0xc00043, network_ctl_w },
    { 0xc00200, 0xc002ff, MWA16_RAM, (data16_t **)&generic_nvram, &generic_nvram_size},             // ??

	{ 0xc40000, 0xc40001, snd_latch_to_68k_w },

	{ 0xd00000, 0xd00001, model1_tgp_copro_adr_w },
	{ 0xd20000, 0xd20003, model1_tgp_copro_ram_w },
	{ 0xd80000, 0xd80003, model1_tgp_copro_w },
	{ 0xd80010, 0xd80013, model1_tgp_copro_w },

	{ 0xe00000, 0xe00001, MWA16_NOP },    // Watchdog?  IRQ ack? Always 0x20, usually on irq
	{ 0xe00004, 0xe00005, bank_w },
	{ 0xe0000c, 0xe0000f, MWA16_NOP },

	{ 0xfc0000, 0xffffff, MWA16_ROM },
MEMORY_END

static PORT_READ16_START( readport )
	{ 0xd20000, 0xd20003, model1_tgp_copro_ram_r },
	{ 0xd80000, 0xd80003, model1_tgp_copro_r },
PORT_END

static READ16_HANDLER( m1_snd_68k_latch_r )
{
	return to_68k;
}

static READ16_HANDLER( m1_snd_v60_ready_r )
{
	return 1;
}

static READ16_HANDLER( m1_snd_mpcm0_r )
{
	return MultiPCM_reg_0_r(0);
}

static WRITE16_HANDLER( m1_snd_mpcm0_w )
{
	MultiPCM_reg_0_w(offset, data);
}

static WRITE16_HANDLER( m1_snd_mpcm0_bnk_w )
{
	MultiPCM_bank_0_w(0, data);
}

static READ16_HANDLER( m1_snd_mpcm1_r )
{
	return MultiPCM_reg_1_r(0);
}

static WRITE16_HANDLER( m1_snd_mpcm1_w )
{
	MultiPCM_reg_1_w(offset, data);
}

static WRITE16_HANDLER( m1_snd_mpcm1_bnk_w )
{
	MultiPCM_bank_1_w(0, data);
}

static READ16_HANDLER( m1_snd_ym_r )
{
	return YM2612_status_port_0_A_r(0);
}

static WRITE16_HANDLER( m1_snd_ym_w )
{
	switch (offset)
	{
		case 0:
			YM2612_control_port_0_A_w(0, data);
			break;

		case 1:
			YM2612_data_port_0_A_w(0, data);
			break;

		case 2:
			YM2612_control_port_0_B_w(0, data);
			break;

		case 3:
			YM2612_data_port_0_B_w(0, data);
			break;
	}
}

static WRITE16_HANDLER( m1_snd_68k_latch1_w )
{
}

static WRITE16_HANDLER( m1_snd_68k_latch2_w )
{
}

static MEMORY_READ16_START( model1_snd_readmem )
	{ 0x000000, 0x0bffff, MRA16_ROM },
	{ 0xc20000, 0xc20001, m1_snd_68k_latch_r },
	{ 0xc20002, 0xc20003, m1_snd_v60_ready_r },
	{ 0xc40000, 0xc40007, m1_snd_mpcm0_r },
	{ 0xc60000, 0xc60007, m1_snd_mpcm1_r },
	{ 0xd00000, 0xd00007, m1_snd_ym_r },
	{ 0xf00000, 0xf0ffff, MRA16_RAM },
MEMORY_END

static MEMORY_WRITE16_START( model1_snd_writemem )
	{ 0x000000, 0x0bffff, MWA16_ROM },
	{ 0xc20000, 0xc20001, m1_snd_68k_latch1_w },
	{ 0xc20002, 0xc20003, m1_snd_68k_latch2_w },
	{ 0xc40000, 0xc40007, m1_snd_mpcm0_w },
	{ 0xc40012, 0xc40013, MWA16_NOP },
	{ 0xc50000, 0xc50001, m1_snd_mpcm0_bnk_w },
	{ 0xc60000, 0xc60007, m1_snd_mpcm1_w },
	{ 0xc70000, 0xc70001, m1_snd_mpcm1_bnk_w },
	{ 0xd00000, 0xd00007, m1_snd_ym_w },
	{ 0xf00000, 0xf0ffff, MWA16_RAM },
MEMORY_END

static struct MultiPCM_interface m1_multipcm_interface =
{
	2,
	{ 8000000, 8000000 },
	{ MULTIPCM_MODE_MODEL1, MULTIPCM_MODE_MODEL1 },
	{ (1024*1024), (1024*1024) },
	{ REGION_SOUND1, REGION_SOUND2 },
	{ YM3012_VOL(100, MIXER_PAN_LEFT, 100, MIXER_PAN_RIGHT), YM3012_VOL(100, MIXER_PAN_LEFT, 100, MIXER_PAN_RIGHT) }
};

static struct YM2612interface m1_ym3438_interface =
{
	1,
	8000000,
	{ 60,60 },
	{ 0 },	{ 0 },	{ 0 },	{ 0 }
};

INPUT_PORTS_START( vf )
	PORT_START  /* Unused analog port 0 */
	PORT_START  /* Unused analog port 1 */
	PORT_START  /* Unused analog port 2 */
	PORT_START  /* Unused analog port 3 */
	PORT_START  /* Unused analog port 4 */
	PORT_START  /* Unused analog port 5 */
	PORT_START  /* Unused analog port 6 */
	PORT_START  /* Unused analog port 7 */

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
INPUT_PORTS_END

INPUT_PORTS_START( vr )
	PORT_START	/* Steering */
	PORT_ANALOG( 0xff, 0x80, IPT_PADDLE| IPF_CENTER, 100, 3, 0, 0xff )

	PORT_START	/* Accel / Decel */
	PORT_ANALOG( 0xff, 0x30, IPT_PEDAL, 100, 16, 1, 0xff )

	PORT_START	/* Brake */
	PORT_ANALOG( 0xff, 0x30, IPT_PEDAL | IPF_PLAYER2, 100, 16, 1, 0xff )

	PORT_START  /* Unused analog port 3 */
	PORT_START  /* Unused analog port 4 */
	PORT_START  /* Unused analog port 5 */
	PORT_START  /* Unused analog port 6 */
	PORT_START  /* Unused analog port 7 */

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( wingwar )
	PORT_START	/* X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 100, 4, 0x00, 0xff )

	PORT_START	/* Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y| IPF_REVERSE, 100, 4, 0x00, 0xff )

	PORT_START	/* Throttle */
	PORT_ANALOG( 0xff, 0x00, IPT_PEDAL, 100, 16, 1, 0xff )

	PORT_START  /* Unused analog port 3 */
	PORT_START  /* Unused analog port 4 */
	PORT_START  /* Unused analog port 5 */
	PORT_START  /* Unused analog port 6 */
	PORT_START  /* Unused analog port 7 */

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON7 | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( swa )
	PORT_START	/* X */
	PORT_ANALOG( 0xff, 127, IPT_AD_STICK_X|IPF_REVERSE, 100, 4, 27, 227 )

	PORT_START	/* Y */
	PORT_ANALOG( 0xff, 127, IPT_AD_STICK_Y|IPF_REVERSE, 100, 4, 27, 227 )

	PORT_START	/* Throttle */
	PORT_ANALOG( 0xff, 228, IPT_PEDAL|IPF_REVERSE, 100, 16, 28, 228 )

	PORT_START  /* Unused analog port 3 */

	PORT_START	/* X */
	PORT_ANALOG( 0xff, 127, IPT_AD_STICK_X | IPF_REVERSE | IPF_PLAYER2, 100, 4, 27, 227 )

	PORT_START	/* Y */
	PORT_ANALOG( 0xff, 127, IPT_AD_STICK_Y | IPF_PLAYER2, 100, 4, 27, 227 )

	PORT_START  /* Unused analog port 6 */
	PORT_START  /* Unused analog port 7 */

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0xe0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

ROM_START( vf )
	ROM_REGION( 0x1400000, REGION_CPU1, 0 ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16082.14", 0x200000, 0x80000, 0xb23f22ee )
	ROM_LOAD16_BYTE( "epr-16083.15", 0x200001, 0x80000, 0xd12c77f8 )

	ROM_LOAD( "epr-16080.4", 0xfc0000, 0x20000, 0x3662E1A5 )
	ROM_LOAD( "epr-16081.5", 0xfe0000, 0x20000, 0x6DEC06CE )

	ROM_LOAD16_BYTE( "mpr-16084.6", 0x1000000, 0x80000, 0x483f453b )
	ROM_LOAD16_BYTE( "mpr-16085.7", 0x1000001, 0x80000, 0x5fa01277 )
	ROM_LOAD16_BYTE( "mpr-16086.8", 0x1100000, 0x80000, 0xdeac47a1 )
	ROM_LOAD16_BYTE( "mpr-16087.9", 0x1100001, 0x80000, 0x7a64daac )
	ROM_LOAD16_BYTE( "mpr-16088.10", 0x1200000, 0x80000, 0xfcda2d1e )
	ROM_LOAD16_BYTE( "mpr-16089.11", 0x1200001, 0x80000, 0x39befbe0 )
	ROM_LOAD16_BYTE( "mpr-16090.12", 0x1300000, 0x80000, 0x90c76831 )
	ROM_LOAD16_BYTE( "mpr-16091.13", 0x1300001, 0x80000, 0x53115448 )

	ROM_REGION( 0xc0000, REGION_CPU2, 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-16120.7", 0x00000, 0x20000, 0x2bff8378 )
	ROM_LOAD16_WORD_SWAP( "epr-16121.8", 0x20000, 0x20000, 0xff6723f9 )
	ROM_RELOAD( 0x80000, 0x20000)

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* Samples */
	ROM_LOAD( "mpr-16122.32", 0x000000, 0x200000, 0x568bc64e )
	ROM_LOAD( "mpr-16123.33", 0x200000, 0x200000, 0x15d78844 )

	ROM_REGION( 0x400000, REGION_SOUND2, 0 ) /* Samples */
	ROM_LOAD( "mpr-16124.4", 0x000000, 0x200000, 0x45520ba1 )
	ROM_LOAD( "mpr-16125.5", 0x200000, 0x200000, 0x9b4998b6 )

	ROM_REGION32_LE( 0x1000000, REGION_USER1, 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16096.26", 0x000000, 0x200000, 0xa92b0bf3 )
	ROM_LOAD32_WORD( "mpr-16097.27", 0x000002, 0x200000, 0x0232955a )
	ROM_LOAD32_WORD( "mpr-16098.28", 0x400000, 0x200000, 0xcf2e1b84 )
	ROM_LOAD32_WORD( "mpr-16099.29", 0x400002, 0x200000, 0x20e46854 )
	ROM_LOAD32_WORD( "mpr-16100.30", 0x800000, 0x200000, 0xe13e983d )
	ROM_LOAD32_WORD( "mpr-16101.31", 0x800002, 0x200000, 0x0dbed94d )
	ROM_LOAD32_WORD( "mpr-16102.32", 0xc00000, 0x200000, 0x4cb41fb6 )
	ROM_LOAD32_WORD( "mpr-16103.33", 0xc00002, 0x200000, 0x526d1c76 )
ROM_END

ROM_START( vr )
	ROM_REGION( 0x1400000, REGION_CPU1, 0 ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-14882.14", 0x200000, 0x80000, 0x547D75AD )
	ROM_LOAD16_BYTE( "epr-14883.15", 0x200001, 0x80000, 0x6BFAD8B1 )

	ROM_LOAD( "epr-14878a.4", 0xfc0000, 0x20000, 0x6D69E695 )
	ROM_LOAD( "epr-14879a.5", 0xfe0000, 0x20000, 0xD45AF9DD )

	ROM_LOAD16_BYTE( "mpr-14880.6",  0x1000000, 0x80000, 0xADC7C208 )
	ROM_LOAD16_BYTE( "mpr-14881.7",  0x1000001, 0x80000, 0xE5AB89DF )
	ROM_LOAD16_BYTE( "mpr-14884.8",  0x1100000, 0x80000, 0x6CF9C026 )
	ROM_LOAD16_BYTE( "mpr-14885.9",  0x1100001, 0x80000, 0xF65C9262 )
	ROM_LOAD16_BYTE( "mpr-14886.10", 0x1200000, 0x80000, 0x92868734 )
	ROM_LOAD16_BYTE( "mpr-14887.11", 0x1200001, 0x80000, 0x10C7C636 )
	ROM_LOAD16_BYTE( "mpr-14888.12", 0x1300000, 0x80000, 0x04BFDC5B )
	ROM_LOAD16_BYTE( "mpr-14889.13", 0x1300001, 0x80000, 0xC49F0486 )

	ROM_REGION( 0xc0000, REGION_CPU2, 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-14870a.7", 0x00000, 0x20000, 0x919d9b75 )

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* Samples */
	ROM_LOAD( "mpr-14873.32", 0x000000, 0x200000, 0xb1965190 )

	ROM_REGION( 0x400000, REGION_SOUND2, 0 ) /* Samples */
	ROM_LOAD( "mpr-14876.4", 0x000000, 0x200000, 0xba6b2327 )

	ROM_REGION32_LE( 0x1000000, REGION_USER1, 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-14890.26", 0x000000, 0x200000, 0xdcbe006b )
	ROM_LOAD32_WORD( "mpr-14891.27", 0x000002, 0x200000, 0x25832b38 )
	ROM_LOAD32_WORD( "mpr-14892.28", 0x400000, 0x200000, 0x5136f3ba )
	ROM_LOAD32_WORD( "mpr-14893.29", 0x400002, 0x200000, 0x1c531ada )
	ROM_LOAD32_WORD( "mpr-14894.30", 0x800000, 0x200000, 0x830a71bc )
	ROM_LOAD32_WORD( "mpr-14895.31", 0x800002, 0x200000, 0xaf027ac5 )
	ROM_LOAD32_WORD( "mpr-14896.32", 0xc00000, 0x200000, 0x382091dc )
	ROM_LOAD32_WORD( "mpr-14879.33", 0xc00002, 0x200000, 0x74873195 )

	ROM_REGION32_LE( 0x200000, REGION_USER2, 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-14898.39", 0x000000, 0x80000, 0x61da2bb6 )
	ROM_LOAD32_BYTE( "mpr-14899.40", 0x000001, 0x80000, 0x2cd58bee )
	ROM_LOAD32_BYTE( "mpr-14900.41", 0x000002, 0x80000, 0xaa7c017d )
	ROM_LOAD32_BYTE( "mpr-14901.42", 0x000003, 0x80000, 0x175b7a9a )
ROM_END

ROM_START( swa )
	ROM_REGION( 0x1000000, REGION_CPU1, 0 ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16468.14", 0x200000, 0x80000, 0x681d03c0 )
	ROM_LOAD16_BYTE( "epr-16469.15", 0x200001, 0x80000, 0x6f281f7c )

	ROM_LOAD( "epr-16467.5", 0xf80000, 0x80000, 0x605068f5 )
	ROM_RELOAD(          0x000000, 0x80000 )
	ROM_RELOAD(          0x080000, 0x80000 )

	ROM_REGION( 0xc0000, REGION_CPU2, 0 )  /* 68K code */
        ROM_LOAD16_WORD_SWAP( "epr16470.bin", 0x000000, 0x020000, 0x7da18cf7 )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* Samples */
        ROM_LOAD( "mpr16486.bin", 0x000000, 0x200000, 0x7df50533 )
        ROM_LOAD( "mpr16487.bin", 0x200000, 0x200000, 0x31b28dfa )

	ROM_REGION( 0x400000, REGION_SOUND2, 0 ) /* Samples */
        ROM_LOAD( "mpr16484.bin", 0x000000, 0x200000, 0x9d4c334d )
        ROM_LOAD( "mpr16485.bin", 0x200000, 0x200000, 0x95aadcad )

	ROM_REGION( 0x20000, REGION_CPU3, 0 ) /* Z80 DSB code */
        ROM_LOAD( "epr16471.bin", 0x000000, 0x020000, 0xf4ee84a4 )

	ROM_REGION( 0x400000, REGION_SOUND3, 0 ) /* DSB MPEG data */
        ROM_LOAD( "mpr16514.bin", 0x000000, 0x200000, 0x3175b0be )
        ROM_LOAD( "mpr16515.bin", 0x000000, 0x200000, 0x3114d748 )

	ROM_REGION32_LE( 0xc00000, REGION_USER1, 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16476.26", 0x000000, 0x200000, 0xd48609ae )
// original dump (which one is right?)
//	ROM_LOAD32_WORD( "mpr-16477.27", 0x000002, 0x200000, 0xb979b082 )
// new dump
        ROM_LOAD32_WORD( "mpr16477.bin", 0x000002, 0x200000, 0x971ff194 )
	ROM_LOAD32_WORD( "mpr-16478.28", 0x400000, 0x200000, 0x80c780f7 )
	ROM_LOAD32_WORD( "mpr-16479.29", 0x400002, 0x200000, 0xe43183b3 )
	ROM_LOAD32_WORD( "mpr-16480.30", 0x800000, 0x200000, 0x3185547a )
	ROM_LOAD32_WORD( "mpr-16481.31", 0x800002, 0x200000, 0xce8d76fe )

	ROM_REGION32_LE( 0x200000, REGION_USER2, 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16472.39", 0x000000, 0x80000, 0x5a0d7553 )
	ROM_LOAD32_BYTE( "mpr-16473.40", 0x000001, 0x80000, 0x876c5399 )
	ROM_LOAD32_BYTE( "mpr-16474.41", 0x000002, 0x80000, 0x5864a26f )
	ROM_LOAD32_BYTE( "mpr-16475.42", 0x000003, 0x80000, 0xb9266be9 )
ROM_END

ROM_START( wingwar )
	ROM_REGION( 0x1300000, REGION_CPU1, 0 ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16729.14", 0x200000, 0x80000, 0x7edec2cc )
	ROM_LOAD16_BYTE( "epr-16730.15", 0x200001, 0x80000, 0xbab24dee )

	ROM_LOAD( "epr-16951.4", 0xfc0000, 0x20000, 0x8df5a798 )
	ROM_RELOAD(          0x000000, 0x20000 )
	ROM_LOAD( "epr-16950.5", 0xfe0000, 0x20000, 0x841e2195 )
	ROM_RELOAD(          0x020000, 0x20000 )

	ROM_LOAD16_BYTE( "mpr-16738.6",  0x1000000, 0x80000, 0x51518ffa )
	ROM_LOAD16_BYTE( "mpr-16737.7",  0x1000001, 0x80000, 0x37b1379c )
	ROM_LOAD16_BYTE( "mpr-16736.8",  0x1100000, 0x80000, 0x10b6a025 )
	ROM_LOAD16_BYTE( "mpr-16735.9",  0x1100001, 0x80000, 0xc82fd198 )
	ROM_LOAD16_BYTE( "mpr-16734.10", 0x1200000, 0x80000, 0xf76371c1 )
	ROM_LOAD16_BYTE( "mpr-16733.11", 0x1200001, 0x80000, 0xe105847b )

	ROM_REGION( 0xc0000, REGION_CPU2, 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP("epr-16751.7", 0x000000, 0x20000, 0x23ba5ebc )
	ROM_LOAD16_WORD_SWAP("epr-16752.8", 0x020000, 0x20000, 0x6541c48f )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* Samples */
	ROM_LOAD("mpr-16753.32", 0x000000, 0x200000, 0x324a8333 )
	ROM_LOAD("mpr-16754.33", 0x200000, 0x200000, 0x144f3cf5 )

	ROM_REGION( 0x400000, REGION_SOUND2, 0 ) /* Samples */
	ROM_LOAD("mpr-16755.4", 0x000000, 0x200000, 0x4baaf878 )
	ROM_LOAD("mpr-16756.5", 0x200000, 0x200000, 0xd9c40672 )

	ROM_REGION32_LE( 0x1000000, REGION_USER1, 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16743.26", 0x000000, 0x200000, 0xa710d33c )
	ROM_LOAD32_WORD( "mpr-16744.27", 0x000002, 0x200000, 0xde796e1f )
	ROM_LOAD32_WORD( "mpr-16745.28", 0x400000, 0x200000, 0x905b689c )
	ROM_LOAD32_WORD( "mpr-16746.29", 0x400002, 0x200000, 0x163b312e )
	ROM_LOAD32_WORD( "mpr-16747.30", 0x800000, 0x200000, 0x7353bb12 )
	ROM_LOAD32_WORD( "mpr-16748.31", 0x800002, 0x200000, 0x8ce98d3a )
	ROM_LOAD32_WORD( "mpr-16749.32", 0xc00000, 0x200000, 0x0e36dc1a )
	ROM_LOAD32_WORD( "mpr-16750.33", 0xc00002, 0x200000, 0xe4f0b98d )

	ROM_REGION32_LE( 0x200000, REGION_USER2, 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16741.39", 0x000000, 0x80000, 0x84b2ffd8 )
	ROM_LOAD32_BYTE( "mpr-16742.40", 0x000001, 0x80000, 0xe9cc12bb )
	ROM_LOAD32_BYTE( "mpr-16739.41", 0x000002, 0x80000, 0x6c73e98f )
	ROM_LOAD32_BYTE( "mpr-16740.42", 0x000003, 0x80000, 0x44b31007 )
ROM_END

ROM_START( wingwara )
	ROM_REGION( 0x1300000, REGION_CPU1, 0 ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16729.14", 0x200000, 0x80000, 0x7edec2cc )
	ROM_LOAD16_BYTE( "epr-16730.15", 0x200001, 0x80000, 0xbab24dee )

	ROM_LOAD( "epr16953.bin", 0xfc0000, 0x20000, 0xc821a920 )
	ROM_RELOAD(          0x000000, 0x20000 )
	ROM_LOAD( "epr16952.bin", 0xfe0000, 0x20000, 0x03a3ecc5 )
	ROM_RELOAD(          0x020000, 0x20000 )

	ROM_LOAD16_BYTE( "mpr-16738.6",  0x1000000, 0x80000, 0x51518ffa )
	ROM_LOAD16_BYTE( "mpr-16737.7",  0x1000001, 0x80000, 0x37b1379c )
	ROM_LOAD16_BYTE( "mpr-16736.8",  0x1100000, 0x80000, 0x10b6a025 )
	ROM_LOAD16_BYTE( "mpr-16735.9",  0x1100001, 0x80000, 0xc82fd198 )
	ROM_LOAD16_BYTE( "mpr-16734.10", 0x1200000, 0x80000, 0xf76371c1 )
	ROM_LOAD16_BYTE( "mpr-16733.11", 0x1200001, 0x80000, 0xe105847b )

	ROM_REGION( 0xc0000, REGION_CPU2, 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP("epr17126.bin",0x000000, 0x20000, 0x50178e40 )
	ROM_LOAD16_WORD_SWAP("epr-16752.8", 0x020000, 0x20000, 0x6541c48f )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* Samples */
	ROM_LOAD("mpr-16753.32", 0x000000, 0x200000, 0x324a8333 )
	ROM_LOAD("mpr-16754.33", 0x200000, 0x200000, 0x144f3cf5 )

	ROM_REGION( 0x400000, REGION_SOUND2, 0 ) /* Samples */
	ROM_LOAD("mpr-16755.4", 0x000000, 0x200000, 0x4baaf878 )
	ROM_LOAD("mpr-16756.5", 0x200000, 0x200000, 0xd9c40672 )

	ROM_REGION32_LE( 0x1000000, REGION_USER1, 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16743.26", 0x000000, 0x200000, 0xa710d33c )
	ROM_LOAD32_WORD( "mpr-16744.27", 0x000002, 0x200000, 0xde796e1f )
	ROM_LOAD32_WORD( "mpr-16745.28", 0x400000, 0x200000, 0x905b689c )
	ROM_LOAD32_WORD( "mpr-16746.29", 0x400002, 0x200000, 0x163b312e )
	ROM_LOAD32_WORD( "mpr-16747.30", 0x800000, 0x200000, 0x7353bb12 )
	ROM_LOAD32_WORD( "mpr-16748.31", 0x800002, 0x200000, 0x8ce98d3a )
	ROM_LOAD32_WORD( "mpr-16749.32", 0xc00000, 0x200000, 0x0e36dc1a )
	ROM_LOAD32_WORD( "mpr-16750.33", 0xc00002, 0x200000, 0xe4f0b98d )

	ROM_REGION32_LE( 0x200000, REGION_USER2, 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16741.39", 0x000000, 0x80000, 0x84b2ffd8 )
	ROM_LOAD32_BYTE( "mpr-16742.40", 0x000001, 0x80000, 0xe9cc12bb )
	ROM_LOAD32_BYTE( "mpr-16739.41", 0x000002, 0x80000, 0x6c73e98f )
	ROM_LOAD32_BYTE( "mpr-16740.42", 0x000003, 0x80000, 0x44b31007 )
ROM_END

static MACHINE_DRIVER_START( model1 )
	MDRV_CPU_ADD(V60, 16000000/12) // Reality is 16Mhz
	MDRV_CPU_MEMORY(readmem,writemem)
	MDRV_CPU_PORTS(readport,0)
	MDRV_CPU_VBLANK_INT(model1_interrupt, 2)

	MDRV_CPU_ADD(M68000, 12000000)	// Confirmed 10 MHz on real PCB, run slightly faster here to prevent sync trouble
	MDRV_CPU_MEMORY(model1_snd_readmem, model1_snd_writemem)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	MDRV_MACHINE_INIT(model1)
	MDRV_NVRAM_HANDLER(generic_0fill)

	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_UPDATE_AFTER_VBLANK | VIDEO_RGB_DIRECT)
	MDRV_SCREEN_SIZE(62*8, 48*8)
	MDRV_VISIBLE_AREA(0*8, 62*8-1, 0*8, 48*8-1)
	MDRV_PALETTE_LENGTH(8192)

	MDRV_VIDEO_START(model1)
	MDRV_VIDEO_UPDATE(model1)
	MDRV_VIDEO_EOF(model1)

	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(YM3438, m1_ym3438_interface)
	MDRV_SOUND_ADD(MULTIPCM, m1_multipcm_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( model1a )
	MDRV_CPU_ADD(V60, 16000000/12) // Reality is 16Mhz
	MDRV_CPU_MEMORY(readmem,writemem)
	MDRV_CPU_PORTS(readport,0)
	MDRV_CPU_VBLANK_INT(model1_interrupt, 2)

	MDRV_CPU_ADD(M68000, 12000000)	// Confirmed 10 MHz on real PCB, run slightly faster here to prevent sync trouble
	MDRV_CPU_MEMORY(model1_snd_readmem, model1_snd_writemem)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	MDRV_MACHINE_INIT(model1)
	MDRV_NVRAM_HANDLER(generic_0fill)

	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_UPDATE_AFTER_VBLANK | VIDEO_RGB_DIRECT)
	MDRV_SCREEN_SIZE(62*8, 48*8)
	MDRV_VISIBLE_AREA(0*8, 62*8-1, 0*8, 48*8-1)
	MDRV_PALETTE_LENGTH(8192)

	MDRV_VIDEO_START(model1)
	MDRV_VIDEO_UPDATE(model1a)
	MDRV_VIDEO_EOF(model1)

	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(YM3438, m1_ym3438_interface)
	MDRV_SOUND_ADD(MULTIPCM, m1_multipcm_interface)
MACHINE_DRIVER_END

GAMEX( 1993, vf,      0, model1, vf,      0, ROT0, "Sega", "Virtua Fighter 1", GAME_IMPERFECT_GRAPHICS )
GAMEX( 1992, vr,       0, model1, vr,       0, ROT0, "Sega", "Virtua Racing", GAME_NOT_WORKING )
GAMEX( 1993, swa,      0, model1, swa,      0, ROT0, "Sega", "Star Wars Arcade", GAME_NOT_WORKING|GAME_NO_SOUND )
GAMEX( 1994, wingwar,  0, model1a, wingwar,  0, ROT0, "Sega", "Wing War (US)", GAME_NOT_WORKING )
GAMEX( 1994, wingwara,  wingwar, model1a, wingwar,  0, ROT0, "Sega", "Wing War", GAME_NOT_WORKING )
