/***************************************************************************

  Sony ZN1/ZN2 - Arcade PSX Hardware
  ==================================
  Driver by smf

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "cpu/mips/mips.h"
#include "cpu/z80/z80.h"
#include "sndhrdw/taitosnd.h"
//#include "includes/psx.h"

/*
  Capcom qsound games

  Based on information from:
   The cps1/cps2 qsound driver,
   Miguel Angel Horna
   Amuse.

  The main board is made by sony, capcom included the qsound on the game
  board. None of these have a bios dumped yet so only the music can be
  played for now.

  The qsound hardware is different to cps2 as it uses an i/o port and
  nmi's instead of shared memory. The driver uses 8bit i/o addresses
  but the real ZN1 hardware may use 16bit i/o addresses as the code
  always accesses port 0xf100. The ZN2 code seems to vary the top
  eight bits of the address ( which may or may not be important ).
*/

static int qcode;
static int qcode_last;
static int queue_data;
static int queue_len;

WRITE_HANDLER( qsound_queue_w )
{
	if( cpu_getstatus( 1 ) != 0 )
	{
		queue_data = data;
		queue_len = 2;
	}
}


static VIDEO_START( znqs )
{
	return 0;
}

static VIDEO_STOP( znqs )
{
}



static VIDEO_UPDATE( znqs )
{
	int refresh = 0;

	if( queue_len == 0 )
	{
		if( keyboard_pressed_memory( KEYCODE_UP ) )
		{
			qcode=( qcode & 0xff00 ) | ( ( qcode + 0x0001 ) & 0xff );
		}
		if( keyboard_pressed_memory( KEYCODE_DOWN ) )
		{
			qcode=( qcode & 0xff00 ) | ( ( qcode - 0x0001 ) & 0xff );
		}
		if( keyboard_pressed_memory( KEYCODE_RIGHT ) )
		{
			qcode=( ( qcode + 0x0100 ) & 0xff00 ) | ( qcode & 0xff );
		}
		if( keyboard_pressed_memory( KEYCODE_LEFT ) )
		{
			qcode=( ( qcode - 0x0100 ) & 0xff00 ) | ( qcode & 0xff );
		}
		if( qcode != qcode_last )
		{
			qsound_queue_w( 0, qcode );
			qcode_last = qcode;
			refresh = 1;
		}
	}

	if( refresh )
	{
		struct DisplayText dt[ 4 ];
		char text1[ 256 ];
		char text2[ 256 ];
		char text3[ 256 ];

		strcpy( text1, Machine->gamedrv->description );
		if( strlen( text1 ) > Machine->uiwidth / Machine->uifontwidth )
		{
			text1[ Machine->uiwidth / Machine->uifontwidth ] = 0;
		}
		sprintf( text2, "QSOUND CODE=%02x/%02x", qcode >> 8, qcode & 0xff );
		if( strlen( text2 ) > Machine->uiwidth / Machine->uifontwidth )
		{
			text2[ Machine->uiwidth / Machine->uifontwidth ] = 0;
		}
		strcpy( text3, "SELECT WITH RIGHT&LEFT/UP&DN" );
		if( strlen( text3 ) > Machine->uiwidth / Machine->uifontwidth )
		{
			text3[ Machine->uiwidth / Machine->uifontwidth ] = 0;
		}
		dt[ 0 ].text = text1;
		dt[ 0 ].color = UI_COLOR_NORMAL;
		dt[ 0 ].x = ( Machine->uiwidth - Machine->uifontwidth * strlen( dt[ 0 ].text ) ) / 2;
		dt[ 0 ].y = Machine->uiheight - Machine->uifontheight * 5;
		dt[ 1 ].text = text2;
		dt[ 1 ].color = UI_COLOR_NORMAL;
		dt[ 1 ].x = ( Machine->uiwidth - Machine->uifontwidth * strlen( dt[ 1 ].text ) ) / 2;
		dt[ 1 ].y = Machine->uiheight - Machine->uifontheight * 3;
		dt[ 2 ].text = text3;
		dt[ 2 ].color = UI_COLOR_NORMAL;
		dt[ 2 ].x = ( Machine->uiwidth - Machine->uifontwidth * strlen( dt[ 2 ].text ) ) / 2;
		dt[ 2 ].y = Machine->uiheight - Machine->uifontheight * 1;
		dt[ 3 ].text = 0; // terminate array 
		displaytext( Machine->scrbitmap, dt );
	}
}


static struct QSound_interface qsound_interface =
{
	QSOUND_CLOCK,
	REGION_SOUND1,
	{ 100,100 }
};

static WRITE_HANDLER( qsound_banksw_w )
{
	unsigned char *RAM = memory_region( REGION_CPU2 );
	if( ( data & 0xf0 ) != 0 )
	{
		logerror( "%08lx: qsound_banksw_w( %02x )\n", activecpu_get_pc(), data & 0xff );
	}
	cpu_setbank( 1, &RAM[ 0x10000 + ( ( data & 0x0f ) * 0x4000 ) ] );
}

static MEMORY_READ_START( qsound_readmem )
	{ 0x0000, 0x7fff, MRA_ROM },
	{ 0x8000, 0xbfff, MRA_BANK1 },	/* banked (contains music data) */
	{ 0xd007, 0xd007, qsound_status_r },
	{ 0xf000, 0xffff, MRA_RAM },
MEMORY_END

static MEMORY_WRITE_START( qsound_writemem )
	{ 0x0000, 0xbfff, MWA_ROM },
	{ 0xd000, 0xd000, qsound_data_h_w },
	{ 0xd001, 0xd001, qsound_data_l_w },
	{ 0xd002, 0xd002, qsound_cmd_w },
	{ 0xd003, 0xd003, qsound_banksw_w },
	{ 0xf000, 0xffff, MWA_RAM },
MEMORY_END

static PORT_READ_START( qsound_readport )
	{ 0x00, 0x00, soundlatch_r },
PORT_END

static struct GfxDecodeInfo znqs_gfxdecodeinfo[] =
{
	{ -1 } //
};


static MEMORY_READ16_START( znqs_readmem )
	{ 0x00000000, 0x001fffff, MRA16_RAM },	/* ram */
	{ 0xa0000000, 0xa01fffff, MRA16_BANK4 },	/* ram mirror */
	{ 0xbfc00000, 0xbfc7ffff, MRA16_BANK3 },	/* bios */
MEMORY_END

static MEMORY_WRITE16_START( znqs_writemem )
	{ 0x00000000, 0x001fffff, MWA16_RAM },	/* ram */
	{ 0xa0000000, 0xa01fffff, MWA16_BANK4 },	/* ram mirror */
	{ 0xbfc00000, 0xbfc7ffff, MWA16_ROM },	/* bios */
MEMORY_END


static INTERRUPT_GEN( qsound_interrupt )
{
	if( queue_len == 2 )
	{
		soundlatch_w( 0, queue_data >> 8 );
		queue_len--;
		cpu_set_irq_line(cpu_getactivecpu(), IRQ_LINE_NMI, PULSE_LINE);
	}
	else if( queue_len == 1 )
	{
		soundlatch_w( 0, queue_data & 0xff );
		queue_len--;
		cpu_set_irq_line(cpu_getactivecpu(), IRQ_LINE_NMI, PULSE_LINE);
	}
	else
		cpu_set_irq_line(cpu_getactivecpu(), 0, HOLD_LINE);
}





INPUT_PORTS_START( zn )
	PORT_START		/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE )	/* pause */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE	)	/* pause */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER2 )

	PORT_START		/* DSWA */
	PORT_DIPNAME( 0xff, 0xff, DEF_STR( Unknown ) )
	PORT_DIPSETTING(	0xff, DEF_STR( Off ) )
	PORT_DIPSETTING(	0x00, DEF_STR( On ) )

	PORT_START		/* DSWB */
	PORT_DIPNAME( 0xff, 0xff, DEF_STR( Unknown ) )
	PORT_DIPSETTING(	0xff, DEF_STR( Off ) )
	PORT_DIPSETTING(	0x00, DEF_STR( On ) )

	PORT_START		/* DSWC */
	PORT_DIPNAME( 0xff, 0xff, DEF_STR( Unknown ) )
	PORT_DIPSETTING(	0xff, DEF_STR( Off ) )
	PORT_DIPSETTING(	0x00, DEF_STR( On ) )

	PORT_START		/* Player 1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )

	PORT_START		/* Player 2 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )

	PORT_START		/* Player 3 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER3 )

	PORT_START		/* Player 4 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER4 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER4 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER4 )

	PORT_START /* Others */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER4 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

	PORT_START /* Others Part2 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SERVICE4 )

/* Input Dip Switch (Hack) */

#if 0	
	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Player3 ))
	PORT_DIPSETTING(	0x01, DEF_STR( Disable? ))
	PORT_DIPSETTING(	0x00, DEF_STR( Enable? ))
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Player4 ))
	PORT_DIPSETTING(	0x02, DEF_STR( Disable? ))
	PORT_DIPSETTING(	0x00, DEF_STR( Enable? ))
#endif




INPUT_PORTS_END







ROM_START( cpzn1 )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn1.bios", 0x0000000, 0x080000, 0x50033af6 )

ROM_END

ROM_START( cpzn2 )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn2.bios", 0x0000000, 0x080000, 0xe860ea8b )

ROM_END

ROM_START( rvschool )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn1.bios", 0x0000000, 0x080000, 0x50033af6 )

	ROM_REGION16_LE( 0x2480000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "jst-04a", 0x0000000, 0x080000, 0x034b1011 )
	ROM_LOAD16_WORD( "jst-05m", 0x0080000, 0x400000, 0x723372b8 )
	ROM_LOAD16_WORD( "jst-06m", 0x0480000, 0x400000, 0x4248988e )
	ROM_LOAD16_WORD( "jst-07m", 0x0880000, 0x400000, 0xc84c5a16 )
	ROM_LOAD16_WORD( "jst-08m", 0x0c80000, 0x400000, 0x791b57f3 )
	ROM_LOAD16_WORD( "jst-09m", 0x1080000, 0x400000, 0x6df42048 )
	ROM_LOAD16_WORD( "jst-10m", 0x1480000, 0x400000, 0xd7e22769 )
	ROM_LOAD16_WORD( "jst-11m", 0x1880000, 0x400000, 0x0a033ac5 )
	ROM_LOAD16_WORD( "jst-12m", 0x1c80000, 0x400000, 0x43bd2ddd )
	ROM_LOAD16_WORD( "jst-13m", 0x2080000, 0x400000, 0x6b443235 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "jst-02",  0x00000, 0x08000, 0x7809e2c3 )
	ROM_CONTINUE(		 0x10000, 0x18000 )
	ROM_LOAD( "jst-03",  0x28000, 0x20000, 0x860ff24d )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "jst-01m", 0x0000000, 0x400000, 0x9a7c98f9 )
ROM_END

ROM_START( jgakuen )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn1.bios", 0x0000000, 0x080000, 0x50033af6 )

	ROM_REGION16_LE( 0x2480000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "jst-04j", 0x0000000, 0x080000, 0x28b8000a )
	ROM_LOAD16_WORD( "jst-05m", 0x0080000, 0x400000, 0x723372b8 )
	ROM_LOAD16_WORD( "jst-06m", 0x0480000, 0x400000, 0x4248988e )
	ROM_LOAD16_WORD( "jst-07m", 0x0880000, 0x400000, 0xc84c5a16 )
	ROM_LOAD16_WORD( "jst-08m", 0x0c80000, 0x400000, 0x791b57f3 )
	ROM_LOAD16_WORD( "jst-09m", 0x1080000, 0x400000, 0x6df42048 )
	ROM_LOAD16_WORD( "jst-10m", 0x1480000, 0x400000, 0xd7e22769 )
	ROM_LOAD16_WORD( "jst-11m", 0x1880000, 0x400000, 0x0a033ac5 )
	ROM_LOAD16_WORD( "jst-12m", 0x1c80000, 0x400000, 0x43bd2ddd )
	ROM_LOAD16_WORD( "jst-13m", 0x2080000, 0x400000, 0x6b443235 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "jst-02",  0x00000, 0x08000, 0x7809e2c3 )
	ROM_CONTINUE(		 0x10000, 0x18000 )
	ROM_LOAD( "jst-03",  0x28000, 0x20000, 0x860ff24d )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "jst-01m", 0x0000000, 0x400000, 0x9a7c98f9 )
ROM_END

ROM_START( kikaioh )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn2.bios", 0x0000000, 0x080000, 0xe860ea8b )

	ROM_REGION16_LE( 0x3080000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "kioj-04.bin", 0x0000000, 0x080000, 0x3a2a3bc8 )
	ROM_LOAD16_WORD( "kio-05m.bin", 0x0080000, 0x800000, 0x98e9eb24 )
	ROM_LOAD16_WORD( "kio-06m.bin", 0x0880000, 0x800000, 0xbe8d7d73 )
	ROM_LOAD16_WORD( "kio-07m.bin", 0x1080000, 0x800000, 0xffd81f18 )
	ROM_LOAD16_WORD( "kio-08m.bin", 0x1880000, 0x800000, 0x17302226 )
	ROM_LOAD16_WORD( "kio-09m.bin", 0x2080000, 0x800000, 0xa34f2119 )
	ROM_LOAD16_WORD( "kio-10m.bin", 0x2880000, 0x800000, 0x7400037a )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "kio-02.bin",  0x00000, 0x08000, 0x174309b3 )
	ROM_CONTINUE(		 0x10000, 0x18000 )
	ROM_LOAD( "kio-03.bin",  0x28000, 0x20000, 0x0b313ae5 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "kio-01m.bin", 0x0000000, 0x400000, 0x6dc5bd07 )
ROM_END

ROM_START( sfex )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn1.bios", 0x0000000, 0x080000, 0x50033af6 )

	ROM_REGION16_LE( 0x1880000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "sfe-04a", 0x0000000, 0x080000, 0x08247bd4 )
	ROM_LOAD16_WORD( "sfe-05m", 0x0080000, 0x400000, 0xeab781fe )
	ROM_LOAD16_WORD( "sfe-06m", 0x0480000, 0x400000, 0x999de60c )
	ROM_LOAD16_WORD( "sfe-07m", 0x0880000, 0x400000, 0x76117b0a )
	ROM_LOAD16_WORD( "sfe-08m", 0x0c80000, 0x400000, 0xa36bbec5 )
	ROM_LOAD16_WORD( "sfe-09m", 0x1080000, 0x400000, 0x62c424cc )
	ROM_LOAD16_WORD( "sfe-10m", 0x1480000, 0x400000, 0x83791a8b )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfe-02",  0x00000, 0x08000, 0x1908475c )
	ROM_CONTINUE(		 0x10000, 0x18000 )
	ROM_LOAD( "sfe-03",  0x28000, 0x20000, 0x95c1e2e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "sfe-01m", 0x0000000, 0x400000, 0xf5afff0d )
ROM_END

ROM_START( sfexj )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn1.bios", 0x0000000, 0x080000, 0x50033af6 )

	ROM_REGION16_LE( 0x1880000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "sfe-04j", 0x0000000, 0x080000, 0xea100607 )
	ROM_LOAD16_WORD( "sfe-05m", 0x0080000, 0x400000, 0xeab781fe )
	ROM_LOAD16_WORD( "sfe-06m", 0x0480000, 0x400000, 0x999de60c )
	ROM_LOAD16_WORD( "sfe-07m", 0x0880000, 0x400000, 0x76117b0a )
	ROM_LOAD16_WORD( "sfe-08m", 0x0c80000, 0x400000, 0xa36bbec5 )
	ROM_LOAD16_WORD( "sfe-09m", 0x1080000, 0x400000, 0x62c424cc )
	ROM_LOAD16_WORD( "sfe-10m", 0x1480000, 0x400000, 0x83791a8b )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfe-02",  0x00000, 0x08000, 0x1908475c )
	ROM_CONTINUE(		 0x10000, 0x18000 )
	ROM_LOAD( "sfe-03",  0x28000, 0x20000, 0x95c1e2e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "sfe-01m", 0x0000000, 0x400000, 0xf5afff0d )
ROM_END

ROM_START( sfexp )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn1.bios", 0x0000000, 0x080000, 0x50033af6 )

	ROM_REGION16_LE( 0x1880000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "sfp-04e", 0x0000000, 0x080000, 0x305e4ec0 )
	ROM_LOAD16_WORD( "sfp-05",  0x0080000, 0x400000, 0xac7dcc5e )
	ROM_LOAD16_WORD( "sfp-06",  0x0480000, 0x400000, 0x1d504758 )
	ROM_LOAD16_WORD( "sfp-07",  0x0880000, 0x400000, 0x0f585f30 )
	ROM_LOAD16_WORD( "sfp-08",  0x0c80000, 0x400000, 0x65eabc61 )
	ROM_LOAD16_WORD( "sfp-09",  0x1080000, 0x400000, 0x15f8b71e )
	ROM_LOAD16_WORD( "sfp-10",  0x1480000, 0x400000, 0xc1ecf652 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfe-02",  0x00000, 0x08000, 0x1908475c )
	ROM_CONTINUE(		 0x10000, 0x18000 )
	ROM_LOAD( "sfe-03",  0x28000, 0x20000, 0x95c1e2e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "sfe-01m", 0x0000000, 0x400000, 0xf5afff0d )
ROM_END

ROM_START( sfexpj )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn1.bios", 0x0000000, 0x080000, 0x50033af6 )

	ROM_REGION16_LE( 0x1880000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "sfp-04j", 0x0000000, 0x080000, 0x18d043f5 )
	ROM_LOAD16_WORD( "sfp-05",  0x0080000, 0x400000, 0xac7dcc5e )
	ROM_LOAD16_WORD( "sfp-06",  0x0480000, 0x400000, 0x1d504758 )
	ROM_LOAD16_WORD( "sfp-07",  0x0880000, 0x400000, 0x0f585f30 )
	ROM_LOAD16_WORD( "sfp-08",  0x0c80000, 0x400000, 0x65eabc61 )
	ROM_LOAD16_WORD( "sfp-09",  0x1080000, 0x400000, 0x15f8b71e )
	ROM_LOAD16_WORD( "sfp-10",  0x1480000, 0x400000, 0xc1ecf652 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sfe-02",  0x00000, 0x08000, 0x1908475c )
	ROM_CONTINUE(		 0x10000, 0x18000 )
	ROM_LOAD( "sfe-03",  0x28000, 0x20000, 0x95c1e2e0 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "sfe-01m", 0x0000000, 0x400000, 0xf5afff0d )
ROM_END

ROM_START( sfex2 )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn2.bios", 0x0000000, 0x080000, 0xe860ea8b )

	ROM_REGION16_LE( 0x2480000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "ex2j-04", 0x0000000, 0x080000, 0x5d603586 )
	ROM_LOAD16_WORD( "ex2-05m", 0x0080000, 0x800000, 0x78726b17 )
	ROM_LOAD16_WORD( "ex2-06m", 0x0880000, 0x800000, 0xbe1075ed )
	ROM_LOAD16_WORD( "ex2-07m", 0x1080000, 0x800000, 0x6496c6ed )
	ROM_LOAD16_WORD( "ex2-08m", 0x1880000, 0x800000, 0x3194132e )
	ROM_LOAD16_WORD( "ex2-09m", 0x2080000, 0x400000, 0x075ae585 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ex2-02",  0x00000, 0x08000, 0x9489875e )
	ROM_CONTINUE(		 0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "ex2-01m", 0x0000000, 0x400000, 0x14a5bb0e )
ROM_END

ROM_START( sfex2p )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn2.bios", 0x0000000, 0x080000, 0xe860ea8b )

	ROM_REGION16_LE( 0x3080000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "sf2p-04", 0x0000000, 0x080000, 0xc6d0aea3 )
	ROM_LOAD16_WORD( "sf2p-05", 0x0080000, 0x800000, 0x4ee3110f )
	ROM_LOAD16_WORD( "sf2p-06", 0x0880000, 0x800000, 0x4cd53a45 )
	ROM_LOAD16_WORD( "sf2p-07", 0x1080000, 0x800000, 0x11207c2a )
	ROM_LOAD16_WORD( "sf2p-08", 0x1880000, 0x800000, 0x3560c2cc )
	ROM_LOAD16_WORD( "sf2p-09", 0x2080000, 0x800000, 0x344aa227 )
	ROM_LOAD16_WORD( "sf2p-10", 0x2880000, 0x800000, 0x2eef5931 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "sf2p-02", 0x00000, 0x08000, 0x3705de5e )
	ROM_CONTINUE(		 0x10000, 0x18000 )
	ROM_LOAD( "sf2p-03", 0x28000, 0x20000, 0x6ae828f6 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "ex2-01m", 0x0000000, 0x400000, 0x14a5bb0e )
ROM_END

ROM_START( shiryu2 )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn2.bios", 0x0000000, 0x080000, 0xe860ea8b )

	ROM_REGION16_LE( 0x2c80000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "hr2j-04.bin", 0x0000000, 0x080000, 0x0824ee5f )
	ROM_LOAD16_WORD( "hr2-05m.bin", 0x0080000, 0x800000, 0x18716fe8 )
	ROM_LOAD16_WORD( "hr2-06m.bin", 0x0880000, 0x800000, 0x6f13b69c )
	ROM_LOAD16_WORD( "hr2-07m.bin", 0x1080000, 0x800000, 0x3925701b )
	ROM_LOAD16_WORD( "hr2-08m.bin", 0x1880000, 0x800000, 0xd844c0dc )
	ROM_LOAD16_WORD( "hr2-09m.bin", 0x2080000, 0x800000, 0xcdd43e6b )
	ROM_LOAD16_WORD( "hr2-10m.bin", 0x2880000, 0x400000, 0xd95b3f37 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "hr2-02.bin",  0x00000, 0x08000, 0xacd8d385 )
	ROM_CONTINUE(		 0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "hr2-01m.bin", 0x0000000, 0x200000, 0x510a16d1 )
	ROM_RELOAD( 0x0200000, 0x200000 )
ROM_END

ROM_START( tgmj )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn2.bios", 0x0000000, 0x080000, 0xe860ea8b )

	ROM_REGION16_LE( 0x0880000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "ate-04j", 0x0000000, 0x080000, 0xbb4bbb96 )
	ROM_LOAD16_WORD( "ate-05",  0x0080000, 0x400000, 0x50977f5a )
	ROM_LOAD16_WORD( "ate-06",  0x0480000, 0x400000, 0x05973f16 )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ate-02",  0x00000, 0x08000, 0xf4f6e82f )
	ROM_CONTINUE(		 0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "ate-01",  0x0000000, 0x400000, 0xa21c6521 )
ROM_END

ROM_START( ts2j )
	ROM_REGION16_LE( 0x0080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "cpzn1.bios", 0x0000000, 0x080000, 0x50033af6 )

	ROM_REGION16_LE( 0x0e80000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "ts2j-04", 0x0000000, 0x080000, 0x4aba8c5e )
	ROM_LOAD16_WORD( "ts2-05",  0x0080000, 0x400000, 0x7f4228e2 )
	ROM_LOAD16_WORD( "ts2-06m", 0x0480000, 0x400000, 0xcd7e0a27 )
	ROM_LOAD16_WORD( "ts2-08m", 0x0880000, 0x400000, 0xb1f7f115 )
	ROM_LOAD16_WORD( "ts2-10",  0x0c80000, 0x200000, 0xad90679a )

	ROM_REGION( 0x50000, REGION_CPU2, 0 ) /* 64k for the audio CPU (+banks) */
	ROM_LOAD( "ts2-02",  0x00000, 0x08000, 0x2f45c461 )
	ROM_CONTINUE(		 0x10000, 0x18000 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY ) /* Q Sound Samples */
	ROM_LOAD( "ts2-01",  0x0000000, 0x400000, 0xd7a505e0 )
ROM_END


/*
  Other ZN1 games

  These have had their bios dumped, but they all fail the self test on
  what appears to be protection hardware & the screen turns red
  ( just enough of the psx chipset is emulated to display this ).

  There is an another romset for Gallop Racer 2 with link hw (z80-a, osc 32.0000MHz,
  upd72103agc & lh540202u x2) which has different versions of rom 119 & 120.
*/

#define VRAM_SIZE ( 1024L * 512L * 2L )

static unsigned short *m_p_vram;
static UINT32 m_p_gpu_buffer[ 16 ];
static unsigned char m_n_gpu_buffer_offset;

static VIDEO_START( zn )
{
	m_p_vram = auto_malloc( VRAM_SIZE );
	if( m_p_vram == NULL )
		return 1;
	return 0;
}

PALETTE_INIT( zn )
{
	UINT16 n_r;
	UINT16 n_g;
	UINT16 n_b;
	UINT32 n_colour;

	for( n_colour = 0; n_colour < 0x10000; n_colour++ )
	{
		n_r = n_colour & 0x1f;
		n_g = ( n_colour >> 5 ) & 0x1f;
		n_b = ( n_colour >> 10 ) & 0x1f;

		n_r = ( n_r * 0xff ) / 0x1f;
		n_g = ( n_g * 0xff ) / 0x1f;
		n_b = ( n_b * 0xff ) / 0x1f;
		
		palette_set_color(n_colour, n_r, n_g, n_b);
	}
}

static VIDEO_UPDATE( zn )
{
	UINT16 n_x;
	UINT16 n_y;
	pen_t *pens = Machine->pens;

	if( bitmap->depth == 16 )
	{
		UINT16 *p_line;
		for( n_y = 0; n_y < bitmap->height; n_y++ )
		{
			p_line = (UINT16 *)bitmap->line[ n_y ];
			for( n_x = 0; n_x < bitmap->width; n_x++ )
			{
				*( p_line++ ) = pens[ m_p_vram[ ( n_y * 1024 ) + n_x ] ];
			}
		}
	}
	else
	{
		UINT8 *p_line;
		for( n_y = 0; n_y < bitmap->height; n_y++ )
		{
			p_line = (UINT8 *)bitmap->line[ n_y ];
			for( n_x = 0; n_x < bitmap->width; n_x++ )
			{
				*( p_line++ ) = pens[ m_p_vram[ ( n_y * 1024 ) + n_x ] ];
			}
		}
	}
}

static void triangle( UINT32 n_x1, UINT32 n_y1, UINT32 n_x2, UINT32 n_y2, UINT32 n_x3, UINT32 n_y3, UINT32 n_colour )
{
	UINT32 n_x;
	UINT32 n_y;
	UINT32 n_mx;
	UINT32 n_my;
	UINT32 n_cx1;
	UINT32 n_cx2;
	INT32 n_dx1;
	INT32 n_dx2;

	n_colour = ( ( n_colour >> 3 ) & 0x1f ) |
		( ( ( n_colour >> 11 ) & 0x1f ) << 5 ) |
		( ( ( n_colour >> 19 ) & 0x1f ) << 10 );

	n_cx1 = n_x1 << 16;
	n_cx2 = n_x1 << 16;
	if( n_y1 < n_y3 )
	{
		n_dx1 = (INT32)( ( n_x3 << 16 ) - n_cx1 ) / (INT32)( n_y3 - n_y1 );
	}
	else
	{
		n_dx1 = 0;
	}
	if( n_y1 < n_y2 )
	{
		n_dx2 = (INT32)( ( n_x2 << 16 ) - n_cx2 ) / (INT32)( n_y2 - n_y1 );
	}
	else
	{
		n_dx2 = 0;
	}

	n_my = n_y2;
	if( n_my < n_y3 )
	{
		n_my = n_y3;
	}
	if( n_my > 512 )
	{
		n_my = 512;
	}
	n_y = n_y1;
	while( n_y < n_my )
	{
		if( n_y == n_y2 )
		{
			n_cx2 = n_x2 << 16;
			n_dx2 = (INT32)( n_cx2 - ( n_x3 << 16 ) ) / (INT32)( n_y2 - n_y3 );
		}
		if( n_y == n_y3 )
		{
			n_cx1 = n_x3 << 16;
			n_dx1 = (INT32)( n_cx1 - ( n_x2 << 16 ) ) / (INT32)( n_y3 - n_y2 );
		}
		n_mx = ( n_cx2 >> 16 );
		if( n_mx > 1024 )
		{
			n_mx = 1024;
		}
		n_x = ( n_cx1 >> 16 );
		while( n_x < n_mx )
		{
			m_p_vram[ n_y * 1024 + n_x ] = n_colour;
			n_x++;
		}
		n_cx1 += n_dx1;
		n_cx2 += n_dx2;
		n_y++;
	}
	set_vh_global_attribute(NULL,0);
}

void gpu32_w( UINT32 offset, UINT32 data )
{
	switch( offset )
	{
	case 0x00:
		m_p_gpu_buffer[ m_n_gpu_buffer_offset ] = data;
		switch( m_p_gpu_buffer[ 0 ] >> 24 )
		{
		case 0x28:
			if( m_n_gpu_buffer_offset < 4 )
			{
				m_n_gpu_buffer_offset++;
			}
			else
			{
				triangle( m_p_gpu_buffer[ 1 ] & 0xffff, m_p_gpu_buffer[ 1 ] >> 16, m_p_gpu_buffer[ 2 ] & 0xffff, m_p_gpu_buffer[ 2 ] >> 16, m_p_gpu_buffer[ 3 ] & 0xffff, m_p_gpu_buffer[ 3 ] >> 16, m_p_gpu_buffer[ 0 ] & 0xffffff );
				triangle( m_p_gpu_buffer[ 2 ] & 0xffff, m_p_gpu_buffer[ 2 ] >> 16, m_p_gpu_buffer[ 4 ] & 0xffff, m_p_gpu_buffer[ 4 ] >> 16, m_p_gpu_buffer[ 3 ] & 0xffff, m_p_gpu_buffer[ 3 ] >> 16, m_p_gpu_buffer[ 0 ] & 0xffffff );
				m_n_gpu_buffer_offset = 0;
			}
			break;
		}
		break;
	}
}

UINT32 gpu32_r( UINT32 offset )
{
	switch( offset )
	{
	case 0x04:
		return 0x14802000;
	}
	return 0;
}

/*

Simple 16 to 32bit bridge.

Only one register can be written to / read per instruction.
Writes occur after the program counter has been incremented.
Transfers are always 32bits no matter what the cpu reads / writes.

*/

static UINT32 m_n_bridge_data;
void *m_p_bridge_timer_w, *m_p_bridge_timer_r;
void ( *m_p_bridge32_w )( UINT32 offset, UINT32 data );

static void bridge_w_flush( int offset )
{
	m_p_bridge32_w( offset, m_n_bridge_data );
}

void bridge_w( void ( *p_bridge32_w )( UINT32 offset, UINT32 data ), UINT32 offset, UINT32 data )
{
	if( ( offset % 4 ) != 0 )
	{
		m_n_bridge_data = ( m_n_bridge_data & 0x0000ffff ) | ( data << 16 );
	}
	else
	{
		m_n_bridge_data = ( m_n_bridge_data & 0xffff0000 ) | ( data & 0xffff );
	}
	
	timer_adjust(m_p_bridge_timer_w, TIME_NOW, offset & ~3, 0 );
}

static void bridge_r_flush( int offset )
{
}

UINT16 bridge_r( UINT32 ( *p_bridge32_r )( UINT32 offset ), UINT32 offset )
{
	timer_adjust(m_p_bridge_timer_r, TIME_NOW, 0, 0 );
	m_n_bridge_data = p_bridge32_r( offset & ~3 );

	if( ( offset % 4 ) != 0 )
	{
		return m_n_bridge_data >> 16;
	}
	return m_n_bridge_data & 0xffff;
}

WRITE16_HANDLER( gpu_w )
{
	bridge_w( gpu32_w, offset<<1, data );
}

READ16_HANDLER( gpu_r )
{
	return bridge_r( gpu32_r, offset<<1 );
}

static MEMORY_READ16_START( zn_readmem )
	{ 0x00000000, 0x001fffff, MRA16_RAM },	/* ram */
	{ 0x1f801810, 0x1f801817, gpu_r },
	{ 0xa0000000, 0xa01fffff, MRA16_BANK1 },	/* ram mirror */
	{ 0xbfc00000, 0xbfc7ffff, MRA16_BANK2 },	/* bios */
MEMORY_END

static MEMORY_WRITE16_START( zn_writemem )
	{ 0x00000000, 0x001fffff, MWA16_RAM },	/* ram */
	{ 0x1f801810, 0x1f801817, gpu_w },
	{ 0xa0000000, 0xa01fffff, MWA16_BANK1 },	/* ram mirror */
	{ 0xbfc00000, 0xbfc7ffff, MWA16_ROM },	/* bios */
MEMORY_END

static MEMORY_READ16_START( namcozn2cpu_readmem )
	{ 0x00000000, 0x001fffff, MRA16_RAM },	/* ram */
	{ 0x1f801810, 0x1f801817, gpu_r },
	{ 0x1fc00000, 0x1fffffff, MRA16_BANK3 },	/* bios */
	{ 0xa0000000, 0xa01fffff, MRA16_BANK1 },	/* ram mirror */
	{ 0xbfc00000, 0xbfffffff, MRA16_BANK2 },	/* bios */
MEMORY_END

static MEMORY_WRITE16_START( namcozn2cpu_writemem )
	{ 0x00000000, 0x001fffff, MWA16_RAM },	/* ram */
	{ 0x1f801810, 0x1f801817, gpu_w },
	{ 0x1fc00000, 0x1fffffff, MWA16_ROM },	/* bios */
	{ 0xa0000000, 0xa01fffff, MWA16_BANK1 },	/* ram mirror */
	{ 0xbfc00000, 0xbfffffff, MWA16_ROM },	/* bios */
MEMORY_END

static MEMORY_READ16_START( namcozn1cpu_readmem )
	{ 0x00000000, 0x001fffff, MRA16_RAM },	/* ram */
	{ 0x1f801810, 0x1f801817, gpu_r },
	{ 0x1fc00000, 0x1fffffff, MRA16_BANK3 },	/* bios */
	{ 0xa0000000, 0xa01fffff, MRA16_BANK1 },	/* ram mirror */
	{ 0xbfc00000, 0xbfffffff, MRA16_BANK2 },	/* bios */
MEMORY_END

static MEMORY_WRITE16_START( namcozn1cpu_writemem )
	{ 0x00000000, 0x001fffff, MWA16_RAM },	/* ram */
	{ 0x1f801810, 0x1f801817, gpu_w },
	{ 0x1fc00000, 0x1fffffff, MWA16_ROM },	/* bios */
	{ 0xa0000000, 0xa01fffff, MWA16_BANK1 },	/* ram mirror */
	{ 0xbfc00000, 0xbfffffff, MWA16_ROM },	/* bios */
MEMORY_END

static MACHINE_INIT( znqs )
{
	// stop CPU1 as it doesn't do anything useful yet...
	timer_suspendcpu( 0, 1, SUSPEND_REASON_DISABLE );

	// but give it some memory so it can reset. 
	cpu_setbank( 3, memory_region( REGION_USER1 ) );
	cpu_setbank( 4, memory_region( REGION_CPU1 ) );


	if( strcmp( Machine->gamedrv->name, "sfex2" ) == 0 ||
		strcmp( Machine->gamedrv->name, "sfex2p" ) == 0 ||
		strcmp( Machine->gamedrv->name, "tgmj" ) == 0 )
	{
		qcode = 0x0400;
	}
	else if( strcmp( Machine->gamedrv->name, "kikaioh" ) == 0 )
	{
		qcode = 0x8000;
	}
	else
	{
		qcode = 0x0000;
	}
	qcode_last = -1;
	queue_len = 0;
	qsound_banksw_w( 0, 0 );

}


static MACHINE_INIT( zn )
{
	cpu_setbank( 1, memory_region( REGION_CPU1 ) );
	cpu_setbank( 2, memory_region( REGION_USER1 ) );


	m_n_gpu_buffer_offset = 0;
	memset( m_p_vram, 0x00, VRAM_SIZE );

	m_p_bridge_timer_w = timer_alloc(bridge_w_flush);
	m_p_bridge_timer_r = timer_alloc(bridge_r_flush);
}

static MACHINE_INIT( namcozn )
{
	cpu_setbank( 1, memory_region( REGION_CPU1 ) );
	cpu_setbank( 2, memory_region( REGION_USER1 ) );
	cpu_setbank( 3, memory_region( REGION_USER1 ) );

	m_n_gpu_buffer_offset = 0;
	memset( m_p_vram, 0x00, VRAM_SIZE );

	m_p_bridge_timer_w = timer_alloc(bridge_w_flush);
	m_p_bridge_timer_r = timer_alloc(bridge_r_flush);
}

static MACHINE_DRIVER_START( znqs )

	/* basic machine hardware */
	MDRV_CPU_ADD(PSXCPU, 33000000 / 2) /* 33MHz ?? */
	MDRV_CPU_MEMORY(znqs_readmem,znqs_writemem)
//	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)

	MDRV_CPU_ADD(Z80, 8000000)
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)  /* 8MHz ?? */
	MDRV_CPU_MEMORY(qsound_readmem,qsound_writemem)
	MDRV_CPU_PORTS(qsound_readport,0)
	MDRV_CPU_VBLANK_INT(qsound_interrupt,4) /* 4 interrupts per frame ?? */

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)
	MDRV_MACHINE_INIT(znqs)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(1024, 1024)
	MDRV_VISIBLE_AREA(0, 639, 0, 479)
	MDRV_GFXDECODE(znqs_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(65535)

//	MDRV_PALETTE_INIT(zn)
//	MDRV_VIDEO_START(zn)
//	MDRV_VIDEO_UPDATE(zn)

	MDRV_VIDEO_START(znqs)
	MDRV_VIDEO_UPDATE(znqs)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(QSOUND, qsound_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( zn )

	/* basic machine hardware */
	MDRV_CPU_ADD(PSXCPU, 33868800 / 2) /* 33MHz ?? */
	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)

	MDRV_MACHINE_INIT(zn)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(1024, 512)
	MDRV_VISIBLE_AREA(0, 639, 0, 479)
	MDRV_PALETTE_LENGTH(65536)

	MDRV_PALETTE_INIT(zn)
	MDRV_VIDEO_START(zn)
	MDRV_VIDEO_UPDATE(zn)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
MACHINE_DRIVER_END

static MEMORY_READ16_START( namcozn1spu_map2 )


MEMORY_END

static MEMORY_WRITE16_START( namcozn1spu_map )



MEMORY_END

static MACHINE_DRIVER_START( namcozn1 )
	/* basic machine hardware */
	MDRV_CPU_ADD( PSXCPU, 33868800 / 2 ) /* 33MHz ?? */
	MDRV_CPU_MEMORY(namcozn1cpu_readmem,namcozn1cpu_writemem)

	MDRV_CPU_ADD( M68000, 16868800 / 2 ) /* 16MHz ?? */
	MDRV_CPU_MEMORY( namcozn1spu_map2 ,namcozn1spu_map )

	MDRV_FRAMES_PER_SECOND( 60 )
	MDRV_VBLANK_DURATION( 0 )

	MDRV_MACHINE_INIT(namcozn)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES( VIDEO_TYPE_RASTER )
	MDRV_SCREEN_SIZE( 1024, 1024 )
	MDRV_VISIBLE_AREA( 0, 1023, 0, 679 )
	MDRV_PALETTE_LENGTH( 65536 )

	MDRV_PALETTE_INIT( zn )
	MDRV_VIDEO_START( zn )
	MDRV_VIDEO_UPDATE( zn )

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( namcozn2 )
	/* basic machine hardware */
	MDRV_CPU_ADD( PSXCPU, 33868800 / 2 ) /* 33MHz ?? */
	MDRV_CPU_MEMORY(namcozn2cpu_readmem,namcozn2cpu_writemem)

	MDRV_FRAMES_PER_SECOND( 60 )
	MDRV_VBLANK_DURATION( 0 )

	MDRV_MACHINE_INIT(namcozn)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES( VIDEO_TYPE_RASTER )
	MDRV_SCREEN_SIZE( 1024, 1024 )
	MDRV_VISIBLE_AREA( 0, 639, 0, 479 )
	MDRV_PALETTE_LENGTH( 65536 )

	MDRV_PALETTE_INIT( zn )
	MDRV_VIDEO_START( zn )
	MDRV_VIDEO_UPDATE( zn )

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( fx1a )
	/* basic machine hardware */
	MDRV_CPU_ADD(PSXCPU, 33000000 / 2) /* 33MHz ?? */
	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)
//	MDRV_CPU_VBLANK_INT(psx,1)

	MDRV_CPU_ADD(Z80,16000000/4)
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)	/* 4 MHz */
//	MDRV_CPU_MEMORY(fx1a_sound_readmem,fx1a_sound_writemem)
//	MDRV_CPU_VBLANK_INT(fx1a_sound_interrupt,1) /* 4 interrupts per frame ?? */

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)

	MDRV_MACHINE_INIT(zn)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(640, 480)
	MDRV_VISIBLE_AREA(0, 639, 0, 479)
	MDRV_PALETTE_LENGTH(65536)

	MDRV_PALETTE_INIT( zn )
	MDRV_VIDEO_START( zn )
	MDRV_VIDEO_UPDATE( zn )

	/* sound hardware */
//	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
//	MDRV_SOUND_ADD(YM2610, ym2610_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( gnet )
	/* basic machine hardware */
	MDRV_CPU_ADD(PSXCPU, 33000000 / 2) /* 33MHz ?? */
	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)
//	MDRV_CPU_VBLANK_INT(psx,1)

//	MDRV_CPU_ADD(Z80,16000000/4)
//	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)	/* 4 MHz */
//	MDRV_CPU_MEMORY(fx1a_sound_readmem,fx1a_sound_writemem)
//	MDRV_CPU_VBLANK_INT(fx1a_sound_interrupt,1) /* 4 interrupts per frame ?? */

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)

	MDRV_MACHINE_INIT(zn)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(640, 480)
	MDRV_VISIBLE_AREA(0, 639, 0, 479)
	MDRV_PALETTE_LENGTH(65536)

	MDRV_PALETTE_INIT( zn )
	MDRV_VIDEO_START( zn )
	MDRV_VIDEO_UPDATE( zn )

	/* sound hardware */
//	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
//	MDRV_SOUND_ADD(YM2610, ym2610_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( psarc95 )
	/* basic machine hardware */
	MDRV_CPU_ADD( PSXCPU, 33868800 / 2 ) /* 33MHz ?? */
	MDRV_CPU_MEMORY(zn_readmem,zn_writemem)

	MDRV_CPU_ADD( M68000, 12000000 ) /* 16MHz ?? */


	MDRV_FRAMES_PER_SECOND( 60 )
	MDRV_VBLANK_DURATION( 0 )

	MDRV_MACHINE_INIT(zn)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES( VIDEO_TYPE_RASTER )
	MDRV_SCREEN_SIZE( 1024, 1024 )
	MDRV_VISIBLE_AREA( 0, 639, 0, 479 )
	MDRV_PALETTE_LENGTH( 65536 )

	MDRV_PALETTE_INIT( zn )
	MDRV_VIDEO_START( zn )
	MDRV_VIDEO_UPDATE( zn )

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)

//	MDRV_SOUND_ADD( PSXSPU , 0 ) /* Un Suported */

//	MDRV_SOUND_ADD( YMF271, ymf271_interface )

MACHINE_DRIVER_END

ROM_START( tps )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION16_LE( 0x080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "mg-bios.bin",  0x0000000, 0x080000, 0x69ffbcb4 )
ROM_END

ROM_START( doapp )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION16_LE( 0x080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "mg-bios.bin",  0x0000000, 0x080000, 0x69ffbcb4 )

	ROM_REGION16_LE( 0x01a00000, REGION_USER2, 0 )
	ROM_LOAD16_BYTE( "doapp119.bin", 0x0000001, 0x100000, 0xbbe04cef )
	ROM_LOAD16_BYTE( "doapp120.bin", 0x0000000, 0x100000, 0xb614d7e6 )
	ROM_LOAD16_WORD( "doapp-0.216",  0x0200000, 0x400000, 0xacc6c539 )
	ROM_LOAD16_WORD( "doapp-1.217",  0x0600000, 0x400000, 0x14b961c4 )
	ROM_LOAD16_WORD( "doapp-2.218",  0x0a00000, 0x400000, 0x134f698f )
	ROM_LOAD16_WORD( "doapp-3.219",  0x0e00000, 0x400000, 0x1c6540f3 )
	ROM_LOAD16_WORD( "doapp-4.220",  0x1200000, 0x400000, 0xf83bacf7 )
	ROM_LOAD16_WORD( "doapp-5.221",  0x1600000, 0x400000, 0xe11e8b71 )
ROM_END

ROM_START( videozn )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION16_LE( 0x080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "kn-bios.bin",  0x0000000, 0x080000, 0x5ff165f3 )
ROM_END

ROM_START( glpracr2 )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION16_LE( 0x080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "mg-bios.bin",  0x0000000, 0x080000, 0x69ffbcb4 )

	ROM_REGION16_LE( 0x02200000, REGION_USER2, 0 )
	ROM_LOAD16_BYTE( "1.119",        0x0000001, 0x100000, 0x0fe2d2df )
	ROM_LOAD16_BYTE( "1.120",        0x0000000, 0x100000, 0x8e3fb1c0 )
	ROM_LOAD16_WORD( "gra2-0.217",   0x0200000, 0x400000, 0xa077ffa3 )
	ROM_LOAD16_WORD( "gra2-1.218",   0x0600000, 0x400000, 0x28ce033c )
	ROM_LOAD16_WORD( "gra2-2.219",   0x0a00000, 0x400000, 0x0c9cb7da )
	ROM_LOAD16_WORD( "gra2-3.220",   0x0e00000, 0x400000, 0x264e3a0c )
	ROM_LOAD16_WORD( "gra2-4.221",   0x1200000, 0x400000, 0x2b070307 )
	ROM_LOAD16_WORD( "gra2-5.222",   0x1600000, 0x400000, 0x94a363c1 )
	ROM_LOAD16_WORD( "gra2-6.223",   0x1a00000, 0x400000, 0x8c6b4c4c )
	ROM_LOAD16_WORD( "gra2-7.323",   0x1e00000, 0x400000, 0x7dfb6c54 )
ROM_END

ROM_START( sncwgltd )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION16_LE( 0x080000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "kn-bios.bin",  0x0000000, 0x080000, 0x5ff165f3 )

	ROM_REGION16_LE( 0x01a80000, REGION_USER2, 0 )
	ROM_LOAD16_WORD( "ic5.bin",      0x0000000, 0x080000, 0x458f14aa )
	ROM_LOAD16_WORD( "ic6.bin",      0x0080000, 0x080000, 0x8233dd1e )
	ROM_LOAD16_WORD( "ic7.bin",      0x0100000, 0x080000, 0xdf5ba2f7 )
	ROM_LOAD16_WORD( "ic8.bin",      0x0180000, 0x080000, 0xe8145f2b )
	ROM_LOAD16_WORD( "ic9.bin",      0x0200000, 0x080000, 0x605c9370 )
	ROM_LOAD16_WORD( "ic11.bin",     0x0280000, 0x400000, 0xa93f6fee )
	ROM_LOAD16_WORD( "ic12.bin",     0x0680000, 0x400000, 0x9f584ef7 )
	ROM_LOAD16_WORD( "ic13.bin",     0x0a80000, 0x400000, 0x652e9c78 )
	ROM_LOAD16_WORD( "ic14.bin",     0x0e80000, 0x400000, 0xc4ef1424 )
	ROM_LOAD16_WORD( "ic15.bin",     0x1280000, 0x400000, 0x2551d816 )
	ROM_LOAD16_WORD( "ic16.bin",     0x1680000, 0x400000, 0x21b401bc )
ROM_END

/* Namco ZN1 ( System 11, System 12) */

ROM_START( aquarush )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION32_LE( 0x00400000, REGION_USER1, 0 ) /* main prg */
	ROM_LOAD16_BYTE( "aq1vera.2l",    0x0000000, 0x200000, 0x91eb9258 )
	ROM_LOAD16_BYTE( "aq1vera.2p",   0x0000001, 0x200000, 0xa92f21aa )

	ROM_REGION32_LE( 0x1000000, REGION_USER2, 0 ) /* main data */
	ROM_LOAD(        "aq1rom0.7",    0x000000, 0x800000, 0x3e64dd33 )
	ROM_LOAD(        "aq1rom1.8",    0x800000, 0x800000, 0xe4d415cf )

	ROM_REGION( 0x0080000, REGION_CPU2, 0 ) /* sound prg */
	ROM_LOAD16_WORD_SWAP( "aq1vera.11s", 0x000000, 0x080000, 0x78277e02 )

	ROM_REGION( 0x0800000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "aq1wave0.2",          0x000000, 0x800000, 0x0cf7278d )
ROM_END



ROM_START( pacapp )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION32_LE( 0x00400000, REGION_USER1, 0 ) /* main prg */
	ROM_LOAD16_BYTE( "ppp1vera.2l",  0x0000000, 0x200000, 0x6e74bd05 )
	ROM_LOAD16_BYTE( "ppp1vera.2p",  0x0000001, 0x200000, 0xb7a2f724 )

	ROM_REGION32_LE( 0x1800000, REGION_USER2, 0 ) /* main data */
	ROM_LOAD16_BYTE( "ppp1rom0l.6",  0x0000000, 0x400000, 0xb152fdd8 )
	ROM_LOAD16_BYTE( "ppp1rom0u.9",  0x0000001, 0x400000, 0xc615c26e )
	ROM_LOAD16_BYTE( "ppp1rom1l.7",  0x0800000, 0x400000, 0x46eaedbd )
	ROM_LOAD16_BYTE( "ppp1rom1u.10", 0x0800001, 0x400000, 0x32f27dce )
	ROM_LOAD16_BYTE( "ppp1rom2l.8",  0x1000000, 0x400000, 0xdca7e5ed )
	ROM_LOAD16_BYTE( "ppp1rom2u.11", 0x1000001, 0x400000, 0xcada7a0d )

	ROM_REGION( 0x0080000, REGION_CPU2, 0 ) /* sound prg */
	ROM_LOAD16_WORD_SWAP( "ppp1vera.11s", 0x0000000, 0x080000, 0x22242317 )

	ROM_REGION( 0x1000000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "ppp1wave0.5",         0x0000000, 0x800000, 0x184ccc7d )
	ROM_LOAD( "ppp1wave1.4",         0x0800000, 0x800000, 0xcbcf74c5 )
ROM_END

ROM_START( pacappsp )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION32_LE( 0x00400000, REGION_USER1, 0 ) /* main prg */
	ROM_LOAD16_BYTE( "psp1vera.2l",  0x0000000, 0x200000, 0x4b6943af )
	ROM_LOAD16_BYTE( "psp1vera.2p",  0x0000001, 0x200000, 0x91397f04 )

	ROM_REGION32_LE( 0x1800000, REGION_USER2, 0 ) /* main data */
	ROM_LOAD(        "psp1rom0.ic7", 0x0000000, 0x800000, 0x7c26ff47 )
	ROM_LOAD(        "psp1rom1.ic8", 0x0800000, 0x800000, 0x4b764fc3 )
	ROM_LOAD(        "psp1rom2.ic9", 0x1000000, 0x800000, 0xa3fa6730 )

	ROM_REGION( 0x0080000, REGION_CPU2, 0 ) /* sound prg */
	ROM_LOAD16_WORD_SWAP( "psp1vera.11s", 0x0000000, 0x080000, 0x47b9ccab )

	ROM_REGION( 0x1000000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "psp1wave0.ic2",       0x0000000, 0x800000, 0x44b9a327 )
	ROM_LOAD( "psp1wave1.ic1",       0x0800000, 0x800000, 0xc045b09c )
ROM_END

ROM_START( pacapp2 )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION32_LE( 0x00400000, REGION_USER1, 0 ) /* main prg */
	ROM_LOAD16_BYTE( "pks1vera.2l",  0x0000000, 0x200000, 0xaec428d3 )
	ROM_LOAD16_BYTE( "pks1vera.2p",  0x0000001, 0x200000, 0x289e6e8a )

	ROM_REGION32_LE( 0x1800000, REGION_USER2, 0 ) /* main data */
	ROM_LOAD(        "pks1rom0.ic7", 0x0000000, 0x800000, 0x91f151fc )
	ROM_LOAD(        "pks1rom1.ic8", 0x0800000, 0x800000, 0x8195f981 )
	ROM_LOAD(        "pks1rom2.ic9", 0x1000000, 0x800000, 0xfea93ddc )

	ROM_REGION( 0x0080000, REGION_CPU2, 0 ) /* sound prg */
	ROM_LOAD16_WORD_SWAP( "pks1vera.11s", 0x0000000, 0x080000, 0x4942b588 )

	ROM_REGION( 0x1000000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "pks1wave0.ic2",       0x0000000, 0x800000, 0x3ebd27c3 )
	ROM_LOAD( "pks1wave1.ic1",       0x0800000, 0x800000, 0x07e426b7 )
ROM_END

/* Namco ZN2 ( System 10 ) */

ROM_START( knpuzzle )
	ROM_REGION( 0x200400, REGION_CPU1, 0 )

	ROM_REGION32_LE( 0x080000, REGION_USER1, 0 )
	ROM_FILL( 0x00000000, 0x080000, 0x55)

	ROM_REGION( 0x3000000, REGION_USER2, 0 ) /* main prg */
	ROM_LOAD( "kpm1a_0.bin",  0x0000000, 0x1000000, 0x606d001b )
	ROM_LOAD( "kpm1a_1.bin",  0x1000000, 0x1000000, 0xa506d767 )
	ROM_LOAD( "kpm1a_2.bin",  0x2000000, 0x1000000, 0x5f40c402 )
ROM_END

/* Taito FX1a/FX1b */

#define TAITOFX1_BIOS \
	ROM_REGION( 0x480400, REGION_CPU1, 0 ) \
	ROM_LOAD( "tt-bios.bin",  0x0400000, 0x080000, 0xe3f23b6e )

ROM_START( taitofx1 )
	TAITOFX1_BIOS
ROM_END

ROM_START( ftimpcta )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00e00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e25-14.3",     0x0000001, 0x100000, 0x0c5f474f )
	ROM_LOAD16_BYTE( "e25-13.4",     0x0000000, 0x100000, 0x7f078d7b )
	ROM_LOAD( "e25-01.1",            0x0200000, 0x400000, 0x8cc4be0c )
	ROM_LOAD( "e25-02.2",            0x0600000, 0x400000, 0x8e8b4c82 )
	ROM_LOAD( "e25-03.12",           0x0a00000, 0x400000, 0x43b1c085 )

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD( "e25-10.14",    0x0000000, 0x080000, 0x2b2ad1b1 )

	ROM_REGION( 0x600000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e25-04.27",    0x0000000, 0x400000, 0x09a66d35 )
	ROM_LOAD( "e25-05.28",    0x0040000, 0x200000, 0x3fb57636 )
ROM_END

ROM_START( gdarius )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00e00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e39-05.3",     0x0000001, 0x100000, 0x750e5b13 )
	ROM_LOAD16_BYTE( "e39-06.4",     0x0000000, 0x100000, 0x2980c30d )
	ROM_LOAD( "e39-01.1",            0x0200000, 0x400000, 0xbdaaa251 )
	ROM_LOAD( "e39-02.2",            0x0600000, 0x400000, 0xa47aab5d )
	ROM_LOAD( "e39-03.12",           0x0a00000, 0x400000, 0xa883b6a5 )

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD( "e39-07.14",    0x0000000, 0x080000, 0x2252c7c1 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e39-04.27",    0x0000000, 0x400000, 0x6ee35e68 )
ROM_END

ROM_START( gdarius2 )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00e00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e39-11.3",     0x0000001, 0x100000, 0x766f73df )
	ROM_LOAD16_BYTE( "e39-12.4",     0x0000000, 0x100000, 0xb23266c3 )
	ROM_LOAD( "e39-01.1",            0x0200000, 0x400000, 0xbdaaa251 )
	ROM_LOAD( "e39-02.2",            0x0600000, 0x400000, 0xa47aab5d )
	ROM_LOAD( "e39-03.12",           0x0a00000, 0x400000, 0xa883b6a5 )

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD( "e39-07.14",    0x0000000, 0x080000, 0x2252c7c1 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e39-04.27",    0x0000000, 0x400000, 0x6ee35e68 )
ROM_END

ROM_START( mgcldate )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00c00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e32-08.2",     0x0000000, 0x100000, 0x3d42cd28 ) 
	ROM_LOAD16_BYTE( "e32-09.7",     0x0000001, 0x100000, 0xdb7ec115 ) 
	ROM_LOAD( "e32-01.1",            0x0200000, 0x400000, 0xcf5f1d01 )
	ROM_LOAD( "e32-02.6",            0x0600000, 0x400000, 0x61c8438c )
	ROM_LOAD( "e32-03.12",           0x0a00000, 0x200000, 0x190d1618 )

	ROM_REGION( 0x2c000, REGION_CPU2, 0 )     /* 64k for Z80 code */
	ROM_LOAD( "e32-10.22",           0x0000000, 0x004000, 0xadf3feb5 )
	ROM_CONTINUE(                    0x0010000, 0x01c000 ) /* banked stuff */

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e32-04.15",           0x0000000, 0x400000, 0xc72f9eea )
ROM_END

ROM_START( mgcldtea )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00c00000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e32-05.2",     0x0000000, 0x080000, 0x72fc7f7b )
	ROM_LOAD16_BYTE( "e32-06.7",     0x0000001, 0x080000, 0xd11c3881 )
	ROM_LOAD( "e32-01.1",            0x0200000, 0x400000, 0xcf5f1d01 )
	ROM_LOAD( "e32-02.6",            0x0600000, 0x400000, 0x61c8438c )
	ROM_LOAD( "e32-03.12",           0x0a00000, 0x200000, 0x190d1618 )

	ROM_REGION( 0x2c000, REGION_CPU2, 0 )     /* 64k for Z80 code */
	ROM_LOAD( "e32-10.22",           0x0000000, 0x004000, 0xadf3feb5 )
	ROM_CONTINUE(                    0x0010000, 0x01c000 ) /* banked stuff */

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e32-04.15",           0x0000000, 0x400000, 0xc72f9eea )
ROM_END

ROM_START( psyforce )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00900000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e22-05.2",     0x0000000, 0x080000, 0x7770242c )
	ROM_LOAD16_BYTE( "e22-10.7",     0x0000001, 0x080000, 0xf6341d63 )
	ROM_LOAD( "e22-02.16",           0x0100000, 0x200000, 0x03b50064 )
	ROM_LOAD( "e22-03.1",            0x0300000, 0x200000, 0x8372f839 )
	ROM_LOAD( "e22-04.21",           0x0500000, 0x200000, 0x397b71aa )

	ROM_REGION( 0x2c000, REGION_CPU2, 0 )     /* 64k for Z80 code */
	ROM_LOAD( "e22-07.22",           0x0000000, 0x004000, 0x739af589 )
	ROM_CONTINUE(                    0x0010000, 0x01c000 ) /* banked stuff */

	ROM_REGION( 0x200000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e22-01.15",           0x000000,  0x200000, 0x808b8340 )
ROM_END

ROM_START( raystorm )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00900000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e24-06.3",     0x0000001, 0x080000, 0xd70cdf46 )
	ROM_LOAD16_BYTE( "e24-05.4",     0x0000000, 0x080000, 0x40097ab9 )
	ROM_LOAD( "e24-02.1",            0x0100000, 0x400000, 0x9f70950d )
	ROM_LOAD( "e24-03.2",            0x0500000, 0x400000, 0x6c1f0a5d )

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD( "e24-09.14",    0x0000000, 0x080000, 0x808589e1 )

	ROM_REGION( 0x400000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e24-04.27",    0x0000000, 0x400000, 0xf403493a )
ROM_END

ROM_START( sfchamp )
	TAITOFX1_BIOS

	ROM_REGION32_LE( 0x00900000, REGION_USER1, 0 )
	ROM_LOAD16_BYTE( "e18-07.2",     0x0000000, 0x080000, 0x1b484e1c )
	ROM_LOAD16_BYTE( "e18-08.7",     0x0000001, 0x080000, 0x6a5558cd )
	ROM_LOAD( "e18-02.12",           0x0100000, 0x200000, 0xc7b4fe29 )
	ROM_LOAD( "e18-03.16",           0x0300000, 0x200000, 0x76392346 )
	ROM_LOAD( "e18-04.19",           0x0500000, 0x200000, 0xfc3731da )
	ROM_LOAD( "e18-05.21",           0x0700000, 0x200000, 0x2e984c50 )

	ROM_REGION( 0x2c000, REGION_CPU2, 0 )     /* 64k for Z80 code */
	ROM_LOAD( "e18-09.22",           0x0000000, 0x004000, 0xbb5a5319 )
	ROM_CONTINUE(                    0x0010000, 0x01c000 ) /* banked stuff */

	ROM_REGION( 0x200000, REGION_SOUND1, ROMREGION_SOUNDONLY )
	ROM_LOAD( "e18-01.15",           0x0000000, 0x200000, 0xdbd1408c )
ROM_END

/* Taito G-NET */



ROM_START( taitogn )
	ROM_REGION32_LE( 0x080000, REGION_USER1, 0 ) 
	ROM_LOAD( "coh-3002t.353", 0x0000000, 0x080000, 0x03967fa7) 

	ROM_REGION32_LE( 0x0200000, REGION_USER2, 0 ) 
    	ROM_LOAD( "flash.u30",    0x000000, 0x200000, 0x2d6740fc) 

	ROM_REGION( 0x080000, REGION_CPU2, 0 ) 
    	ROM_LOAD( "flash.u27",    0x000000, 0x080000, 0x75bd9c51) 

ROM_END

ROM_START( taitogn1 )
	ROM_REGION32_LE( 0x080000, REGION_USER1, 0 ) 
	ROM_LOAD( "coh-3002t.353", 0x000000, 0x080000, 0x03967fa7) 
	ROM_REGION( 0x200000, REGION_USER2, 0 ) 
    	ROM_LOAD( "flashv1.u30", 0x000000, 0x200000, 0xc48c8236) 
ROM_END


ROM_START( taitogn2 )
	ROM_REGION32_LE( 0x080000, REGION_USER1, 0 ) 
	ROM_LOAD( "coh-3002t.353", 0x000000, 0x080000, 0x03967fa7) 
	ROM_REGION( 0x200000, REGION_USER2, 0 ) 
    	ROM_LOAD( "flashv2.u30", 0x000000, 0x200000, 0xCAE462D3)
ROM_END

/* Atlus */


ROM_START( atluspsx )
	ROM_REGION32_LE( 0x080000, REGION_USER1, 0 ) 
	ROM_LOAD( "coh-1001l.353", 0x000000, 0x080000, 0x6721146b )
ROM_END

ROM_START( hvnsgate )
	ROM_REGION32_LE( 0x080000, REGION_USER1, 0 ) 
	ROM_LOAD( "coh-1001l.353", 0x000000, 0x080000, 0x6721146b )

	ROM_REGION32_LE( 0x02000000, REGION_USER2, 0 )
	ROM_LOAD16_BYTE( "athg-01b.18",  0x0000001, 0x080000, 0xe820136f )
	ROM_LOAD16_BYTE( "athg-02b.17",  0x0000000, 0x080000, 0x11bfa89b) 
	ROM_LOAD( "athg-07.027",         0x0100000, 0x400000, 0x46411f67) 
	ROM_LOAD( "athg-08.028",        0x0500000, 0x400000, 0x85289345)
	ROM_LOAD( "athg-09.210",         0x0900000, 0x400000, 0x19e558b5) 
	ROM_LOAD( "athg-10.029",         0x0d00000, 0x400000, 0x748f936e) 
	ROM_LOAD( "athg-11.215",         0x1100000, 0x200000, 0xac8e53bd) 

	ROM_REGION( 0x040000, REGION_CPU2, 0 ) /* 68000 code, 10.000MHz */
	ROM_LOAD16_BYTE( "athg-04.21",   0x000001, 0x020000, 0x18523e85) 
	ROM_LOAD16_BYTE( "athg-03.22",   0x000000, 0x020000, 0x7eef7e68) 

	ROM_REGION( 0x400000, REGION_SOUND1, 0 ) /* YMZ280B Sound Samples */
	ROM_LOAD( "athg-05.4136", 0x000000, 0x200000, 0x74469a15 )
	ROM_LOAD( "athg-06.4134", 0x200000, 0x200000, 0x443ade73)
ROM_END

/* PS Arcade 95 */

#define PSARC95_BIOS \
	ROM_REGION32_LE( 0x080000, REGION_USER1, 0 ) \
	ROM_LOAD( "coh-1002e.353", 0x000000, 0x080000, 0x910f3a8b)

ROM_START( psarc95 )
	PSARC95_BIOS
ROM_END

ROM_START( beastrzr )
	PSARC95_BIOS

	ROM_REGION32_LE( 0x1800000, REGION_USER2, 0 )
	ROM_LOAD16_BYTE( "b.roar_u0213",   0x000001, 0x080000, 0x2c586534 ) /* For U0212 & U0213, 8ing used indentical rom labels */
	ROM_LOAD16_BYTE( "b.roar_u0212",   0x000000, 0x080000, 0x1c85d7fb) /* even though the content changes between versions   */
	ROM_LOAD16_BYTE( "b.roar-u0215",   0x100001, 0x080000, 0x31c8e055)
	ROM_LOAD16_BYTE( "b.roar-u0214",   0x100000, 0x080000, 0x1cdc450a)
	ROM_LOAD( "ra-b-roar_rom-1.u0217", 0x400000, 0x400000, 0x11f1ba36) 
	ROM_LOAD( "ra-b.roar_rom-2.u0216", 0x800000, 0x400000, 0xd46d46b7)

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD16_BYTE( "b.roar-u046",  0x000001, 0x040000, 0xd4bb261a) 
	ROM_LOAD16_BYTE( "b.roar-u042",  0x000000, 0x040000, 0x4d537f88) 

	ROM_REGION( 0x400000, REGION_SOUND1, 0 )
	ROM_LOAD( "ra-b.roar3_rom-3.u0326", 0x000000, 0x400000, 0xb74cc4d1) 
ROM_END

ROM_START( bldyroar )
	PSARC95_BIOS

	ROM_REGION32_LE( 0x1800000, REGION_USER2, 0 )
	ROM_LOAD16_BYTE( "b.roar-u0213",   0x000001, 0x080000, 0x63769342 ) /* For U0212 & U0213, 8ing used indentical rom labels */
	ROM_LOAD16_BYTE( "b.roar-u0212",   0x000000, 0x080000, 0x966b7169) /* even though the content changes between versions   */
	ROM_LOAD16_BYTE( "b.roar-u0215",   0x100001, 0x080000, 0x31c8e055)
	ROM_LOAD16_BYTE( "b.roar-u0214",   0x100000, 0x080000, 0x1cdc450a)
	ROM_LOAD( "ra-b-roar_rom-1.u0217", 0x400000, 0x400000, 0x11f1ba36) 
	ROM_LOAD( "ra-b.roar_rom-2.u0216", 0x800000, 0x400000, 0xd46d46b7)

	ROM_REGION( 0x080000, REGION_CPU2, 0 )
	ROM_LOAD16_BYTE( "b.roar-u046",  0x000001, 0x040000, 0xd4bb261a )
	ROM_LOAD16_BYTE( "b.roar-u042",  0x000000, 0x040000, 0x4d537f88 ) 

	ROM_REGION( 0x400000, REGION_SOUND1, 0 )
	ROM_LOAD( "ra-b.roar3_rom-3.u0326", 0x000000, 0x400000, 0xb74cc4d1)
ROM_END

ROM_START( beastrzb )
	PSARC95_BIOS

	ROM_REGION32_LE( 0x0a80000, REGION_USER1, 0 )
	ROM_LOAD( "27c160.1",     0x0000000, 0x200000, 0x820855e2 )
	ROM_LOAD( "27c160.2",     0x0200000, 0x200000, 0x1712af34 )
	ROM_LOAD( "27c800.3",     0x0400000, 0x100000, 0x7192eb4e )
	ROM_LOAD( "4",            0x0500000, 0x200000, 0xbff21f44 )
	ROM_LOAD( "27c160.5",     0x0700000, 0x200000, 0x10fe6f4d )
	ROM_LOAD( "27c4096.1",    0x0900000, 0x080000, 0x217734a1 )
	ROM_LOAD( "27c4096.2",    0x0980000, 0x080000, 0xd1f2a9b2 )
	ROM_LOAD( "27c240.3",     0x0a00000, 0x080000, 0x509cdc8b )
ROM_END

/* Capcom ZN1&ZN2 */

/* ZN1&ZN2 Bios */
/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the cpzn1.bin file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1996, cpzn1,	  0,		znqs, zn, 0, ROT0, "Sony / Capcom", "ZN-1 BIOS", NOT_A_DRIVER )

/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the cpzn2.bin file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1997, cpzn2,	  0,		znqs, zn, 0, ROT0, "Sony / Capcom", "ZN-2 BIOS", NOT_A_DRIVER )
/* ZN1&ZN2 Bios End */

GAME( 1995, ts2j,	  cpzn1,		znqs, zn, 0, ROT0, "Capcom / Takara", "Battle Arena Toshinden 2 (JAPAN 951124) (Sound Player Only)" )
GAME( 1996, sfex,	  cpzn1,		znqs, zn, 0, ROT0, "Capcom / Arika", "Street Fighter EX (ASIA 961219) (Sound Player Only)" )
GAME( 1996, sfexj,	  sfex, 	znqs, zn, 0, ROT0, "Capcom / Arika", "Street Fighter EX (JAPAN 961130) (Sound Player Only)" )
GAME( 1997, sfexp,	  cpzn1,		znqs, zn, 0, ROT0, "Capcom / Arika", "Street Fighter EX Plus (USA 970311) (Sound Player Only)" ) 
GAME( 1997, sfexpj,   sfexp,	znqs, zn, 0, ROT0, "Capcom / Arika", "Street Fighter EX Plus (JAPAN 970311) (Sound Player Only)" )
GAME( 1997, rvschool, cpzn1,		znqs, zn, 0, ROT0, "Capcom", "Rival Schools (ASIA 971117) (Sound Player Only)" )
GAME( 1997, jgakuen,  rvschool, znqs, zn, 0, ROT0, "Capcom", "Justice Gakuen (JAPAN 971117) (Sound Player Only)" )

GAME( 1998, sfex2,	  cpzn2,		znqs, zn, 0, ROT0, "Capcom / Arika", "Street Fighter EX 2 (JAPAN 980312) (Sound Player Only)"  )
GAME( 1998, tgmj,	  cpzn2,		znqs, zn, 0, ROT0, "Capcom / Arika", "Tetris The Grand Master (JAPAN 980710) (Sound Player Only)" )
GAME( 1998, kikaioh,  cpzn2,		znqs, zn, 0, ROT0, "Capcom", "Kikaioh (JAPAN 980914) (Sound Player Only)" )
GAME( 1999, sfex2p,   cpzn2,		znqs, zn, 0, ROT0, "Capcom / Arika", "Street Fighter EX 2 Plus (JAPAN 990611) (Sound Player Only)" )
GAME( 1999, shiryu2,  cpzn2,		znqs, zn, 0, ROT0, "Capcom", "Strider Hiryu 2 (JAPAN 991213) (Sound Player Only)" )

/* Video System ZN1 */
/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the kn-bios.bin file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1996, videozn,	0,	  zn, zn, 0, ROT0, "Sony / Video System Co.", "Video System ZN-1 BIOS", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS | NOT_A_DRIVER )

GAMEX( 1996, sncwgltd,	videozn,	  zn, zn, 0, ROT0, "Video System Co.", "Sonic Wings Limited (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )

/* Tecmo ZN1 */
/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the mg-bios.bin file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1997, tps,	0,	  zn, zn, 0, ROT0, "Sony / Tecmo", "Tecmo ZN-1 BIOS", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS | NOT_A_DRIVER )

GAMEX( 1997, glpracr2,	tps,	  zn, zn, 0, ROT0, "Tecmo", "Gallop Racer 2 (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1998, doapp,		tps,	  zn, zn, 0, ROT0, "Tecmo", "Dead Or Alive ++ (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )

/* Namco ZN2 (System 10) */
GAMEX( 2001, knpuzzle,		0,	  namcozn2, zn, 0, ROT0, "Namco", "Kotoba No Puzzle Mojipittan (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )

/* Namco ZN1 (System 11, System 12)*/
GAMEX( 1999, aquarush,  0,		namcozn1, zn, 0, ROT0, "Namco", "Aqua Rush (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1999, pacapp,  0,		namcozn1, zn, 0, ROT0, "Produce/Namco", "Paca Paca Passion (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1999, pacapp2,  0,		namcozn1, zn, 0, ROT0, "Produce / Namco", "Paca Paca Passion 2 (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1999, pacappsp,  0,		namcozn1, zn, 0, ROT0, "Produce / Namco", "Paca Paca Passion Special (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )

/* Taito FX-1 */

/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the tt-bios.bin file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1995, taitofx1, 0,        fx1a,     zn, 0,   ROT0, "Sony / Taito", "Taito FX-1", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS | NOT_A_DRIVER )

GAMEX( 1995, psyforce, taitofx1, fx1a,     zn, 0,   ROT0, "Taito", "Psychic Force (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1995, sfchamp,  taitofx1, fx1a,     zn, 0,   ROT0, "Taito", "Super Football Champ (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1996, raystorm, taitofx1, fx1a,     zn, 0, ROT0, "Taito", "Ray Storm (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1996, ftimpcta, taitofx1, fx1a,     zn, 0, ROT0, "Taito", "Fighter's Impact Ace (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1996, mgcldate, taitofx1, fx1a,     zn, 0,   ROT0, "Taito", "Magical Date (JAPAN) set 1", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1996, mgcldtea, mgcldate, fx1a,     zn, 0, ROT0, "Taito", "Magical Date (JAPAN) set 2", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1997, gdarius,  taitofx1, fx1a,     zn, 0, ROT0, "Taito", "G-Darius (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1997, gdarius2, gdarius,  fx1a,     zn, 0, ROT0, "Taito", "G-Darius Ver.2 (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )

/* Taito G-NET */

/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the flash.u30 file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1997, taitogn,  0,        gnet, zn, 0, ROT0, "Sony / Taito", "Taito G-NET", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS | NOT_A_DRIVER  )

GAMEX( 1997, taitogn1,  taitogn,        gnet, zn, 0, ROT0, "Sony / Taito", "Taito G-NET (Newer Ver.1)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS  )

GAMEX( 1997, taitogn2,  taitogn,        gnet, zn, 0, ROT0, "Sony / Taito", "Taito G-NET (Newer Ver.2)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS  )

/* PS Arcade 95 */

/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the coh-1002e.353 file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1997, psarc95,  0,        psarc95, zn, 0, ROT0, "Sony/Eighting/Raizing", "PS Arcade 95", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS | NOT_A_DRIVER )

GAMEX( 1997, beastrzb, psarc95, psarc95,       zn, 0,   ROT0, "Eighting / Raizing", "Beastorizer (bootleg)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1997, beastrzr,  psarc95,  psarc95, zn, 0, ROT0, "Eighting / Raizing", "Beastorizer (USA)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
GAMEX( 1997, bldyroar,  beastrzr, psarc95, zn, 0, ROT0, "Eighting / Raizing", "Bloody Roar (JAPAN)", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )

/* Atlus ZN1 */

/* A dummy driver, so that the bios can be debugged, and to serve as */
/* parent for the coh-1002l.353 file, so that we do not have to include */
/* it in every zip file */
GAMEX( 1996, atluspsx,  0,       zn, zn, 0, ROT0, "Sony / Atlus", "Atlus ZN-1 BIOS", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS | NOT_A_DRIVER )

GAMEX( 1996, hvnsgate, atluspsx, zn, zn, 0, ROT0, "Atlus / RACDYM", "Heaven's Gate", GAME_NOT_WORKING | GAME_UNEMULATED_PROTECTION | GAME_NO_SOUND | GAME_IMPERFECT_GRAPHICS )
