﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Net;
using NaGet.Packages;
using NaGet.Packages.Install;
using NaGet.SubCommands;
using NaGet.SubCommands.SubTask;
using NaGet.Tasks;

namespace NaGet.SubCommands
{
	/// <summary>
	/// ソフトリストなどのアップデート処理
	/// </summary>
	public class NaGetUpdate2 : NaGetTaskSet2
	{
		private PackageListsManager pkgListMan = null;
		
		private bool downloadPackageLists = true;
		
		private RepositoriesList repoList = null;
		
		private IList<string> tempRepoFiles = null;
		
		public NaGetUpdate2(PackageListsManager pkgListMan)
			: this(pkgListMan, true)
		{
		}
		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="pkgMan">参照パッケージマネージャ</param>
		/// <param name="downloadPackageListsFlag">リストをダウンロードするか?</param>
		public NaGetUpdate2(PackageListsManager pkgMan, bool downloadPackageListsFlag)
		{
			this.pkgListMan = pkgMan;
			this.downloadPackageLists = downloadPackageListsFlag;
			
			// taskセットの初期化
			initSubTask();
			if (this.downloadPackageLists) {
				// repos.list.xmlがあるとき、そこからよみとる。
				repoList = NaGet.Utils.GetDeserializedObject<RepositoriesList>(NaGet.Env.RepositoriesListFile);
				tempRepoFiles = new List<string>();
				
				foreach (RepositoryInfo repo in repoList.EnabledRepositories) {
					string filepath = Path.GetTempFileName();
					
					tempRepoFiles.Add(filepath);
					registSubTask(string.Format("リスト取得: {0}", repo.Url.Href),
					              new DownloadSubTask(repo.Url.Href, filepath));
				}
				registSubTask(string.Format("リスト更新: {0}", NaGet.Env.PackageListFile),
				              new FunctionalSubTask<object>(runUpdatePackageListFile, null));
			}
			registSubTask("インストール済みのソフトリスト更新",
			              new LocalUpdateSubTask(this.pkgListMan));
		}
		
		public override void Run()
		{
			NotifyStarted();
			RaiseTaskSetEvent(TaskEventType.STARTED, string.Empty);
			
			try {
				while (hasMoreSubTask) {
					RaiseTaskSetEvent(TaskEventType.STARTED_SUBTASK, currentSubTaskName);
					currentSubTask.Run();
					RaiseTaskSetEvent(TaskEventType.COMPLETED_SUBTASK, currentSubTaskName);
					
					NotifyGoToNextSubTask();
				}
			} catch (TaskCanceledException) {
				cancelCalled = true;
			} catch (Exception e) {
				RaiseTaskSetEvent(TaskEventType.ERROR, e.Message);
			} finally {
				runDeleteTempFiles();
			}
			
			if (cancelCalled) {
				NotifyCancelled();
				RaiseTaskSetEvent(TaskEventType.CANCELED, "キャンセルされました");
			} else {
				NotifyCompleted();
				RaiseTaskSetEvent(TaskEventType.COMPLETED, string.Empty);
			}
		}
		
		private void runUpdatePackageListFile(object dummy)
		{
			int i = 0;
			PackageList<Package> avaiablePackageList = new PackageList<Package>();
			
			// かならず常にrepositoryリストに書き込む。
			NaGet.Utils.PutSerializeObject(NaGet.Env.RepositoriesListFile, repoList);
			
			foreach (RepositoryInfo repo in repoList.EnabledRepositories) {
				if (repo.Type == RepositoryType.APPLISTATION_NATIVE_XML_1_0) {
					try {
						string tmpfileName = tempRepoFiles[i];
						PackageList<Package> pkgList = NaGet.Utils.GetDeserializedObject<PackageList<Package>>(tmpfileName);
						pkgList.FixPackageListName(); // PackageListNameとの紐付けを行う
						
						// RepositoryReferenceの名前を読み込む
						repo.Name = (string.IsNullOrEmpty(pkgList.Name))? repo.Name : pkgList.Name;
						
						avaiablePackageList.AddPackages(pkgList);
					} catch (InvalidOperationException) {
						RaiseTaskSetEvent(TaskEventType.ERROR, string.Format("レポジトリ'{0}'はAppliStation Native XML softlist形式ではありません。", repo.Name ?? repo.Url.Href));
					}
				} else {
					RaiseTaskSetEvent(TaskEventType.WARNING, string.Format("レポジトリ'{0}'の設定が不正です。", repo.Name ?? repo.Url.Href));
				}
				i++;
			}
			
			pkgListMan.availablePkgList = avaiablePackageList; // Mediatorのリストを更新
			pkgListMan.SaveAvailablePackageList();
		}
		
		private void runDeleteTempFiles()
		{
			if (tempRepoFiles != null) {
				foreach (string file in tempRepoFiles) {
					if (File.Exists(file)) {
						File.Delete(file);
					}
				}
			}
		}
		
		public override bool Cancelable {
			get {
				return !cancelCalled && Running && isDuringDownloading;
			}
		}
		
		private bool isDuringDownloading {
			get {
				return Running && (currentSubTask is DownloadSubTask);
			}
		}
		
	}
}
