/*
 * Copyright (C) 2007 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package tests.api.javax.net.ssl;

import dalvik.annotation.TestTargetClass;
import dalvik.annotation.TestLevel;
import dalvik.annotation.TestTargetNew;

import java.security.InvalidAlgorithmParameterException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactorySpi;
import javax.net.ssl.ManagerFactoryParameters;

import junit.framework.TestCase;

import org.apache.harmony.xnet.tests.support.KeyManagerFactorySpiImpl;

@TestTargetClass(KeyManagerFactorySpi.class) 
public class KeyManagerFactorySpiTest extends TestCase {

    /**
     * @tests javax.net.ssl.KeyManagerFactorySpi#KeyManagerFactorySpi()
     */
    @TestTargetNew(
        level = TestLevel.COMPLETE,
        notes = "",
        method = "KeyManagerFactorySpi",
        args = {}
    )
    public void test_Constructor() {
        try {
            KeyManagerFactorySpiImpl kmf = new KeyManagerFactorySpiImpl();
            assertTrue(kmf instanceof KeyManagerFactorySpi);
        } catch (Exception e) {
            fail("Unexpected Exception " + e.toString());
        }
    }
    
    /**
     * @tests javax.net.ssl.KeyManagerFactorySpi#KengineInit(KeyStore ks, char[] password)
     */
    @TestTargetNew(
        level = TestLevel.COMPLETE,
        notes = "",
        method = "engineInit",
        args = {java.security.KeyStore.class, char[].class}
    )
    public void test_engineInit_01() {
        KeyManagerFactorySpiImpl kmf = new KeyManagerFactorySpiImpl();
        KeyStore ks;
        char[] psw = "password".toCharArray();
        
        try {
            kmf.engineInit(null, null);
            fail("NoSuchAlgorithmException wasn't thrown");
        } catch (NoSuchAlgorithmException kse) {
            //expected
        } catch (Exception e) {
            fail(e + " was thrown instead of NoSuchAlgorithmException");
        }
        
        try {
            kmf.engineInit(null, psw);
            fail("KeyStoreException wasn't thrown");
        } catch (KeyStoreException uke) {
            //expected
        } catch (Exception e) {
            fail(e + " was thrown instead of KeyStoreException");
        }
        
        try {
            ks = KeyStore.getInstance(KeyStore.getDefaultType());
            kmf.engineInit(ks, null);
            fail("UnrecoverableKeyException wasn't thrown");
        } catch (UnrecoverableKeyException uke) {
            //expected
        } catch (Exception e) {
            fail(e + " was thrown instead of UnrecoverableKeyException");
        }
        
        try {
            KeyStore kst = KeyStore.getInstance(KeyStore.getDefaultType());
            kst.load(null, null);
            kmf.engineInit(kst, psw);
        } catch (Exception e) {
            fail("Unexpected exception " + e);
        }
    }
    
    /**
     * @tests javax.net.ssl.KeyManagerFactorySpi#KengineInit(ManagerFactoryParameters spec)
     */
    @TestTargetNew(
        level = TestLevel.COMPLETE,
        notes = "",
        method = "engineInit",
        args = {javax.net.ssl.ManagerFactoryParameters.class}
    )
    public void test_engineInit_02() {
        KeyManagerFactorySpiImpl kmf = new KeyManagerFactorySpiImpl();
        
        try {
            kmf.engineInit(null);
            fail("InvalidAlgorithmParameterException wasn't thrown");
        } catch (InvalidAlgorithmParameterException iape) {
            //expected
        } catch (Exception e) {
            fail(e + " was thrown instead of InvalidAlgorithmParameterException");
        }
        
        try {
            char[] psw = "password".toCharArray();
            Parameters pr = new Parameters(psw);
            kmf.engineInit(pr);
        } catch (Exception e) {
            fail(e + " unexpected exception was thrown");
        }
    }
    
    /**
     * @tests javax.net.ssl.KeyManagerFactorySpi#engineGetKeyManagers()
     */
    @TestTargetNew(
        level = TestLevel.COMPLETE,
        notes = "",
        method = "engineGetKeyManagers",
        args = {}
    )
    public void test_engineGetKeyManagers() {
        KeyManagerFactorySpiImpl kmf = new KeyManagerFactorySpiImpl();
        
        try {
            KeyManager[] km = kmf.engineGetKeyManagers();
            fail("IllegalStateException wasn't thrown");
        } catch (IllegalStateException ise) {
            //expected
        } catch (Exception e) {
            fail(e + " was thrown instead of IllegalStateException");
        }
        
        try {
            char[] psw = "password".toCharArray();
            Parameters pr = new Parameters(psw);
            kmf.engineInit(pr);
            KeyManager[] km = kmf.engineGetKeyManagers();
            assertNull("Object is not NULL", km);
        } catch (Exception e) {
            fail(e + " unexpected exception was thrown");
        }
    }
    
    public class Parameters implements ManagerFactoryParameters {
        private char[] passWD;
        
        public Parameters (char[] pass) {
            this.passWD = pass;
        }
        public char[] getPassword() {
            return passWD;
        }
    }
    
}
