package jp.sf.amateras.functions.utils;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

/**
 * 日付関係のユーティリティメソッドを提供します。
 *
 * @author Naoki Takezoe
 */
public class DateUtils {

	private static String datePattern = "yyyy/MM/dd";
	private static String datetimePattern = "yyyy/MM/dd HH:mm:ss";
	private static String timePattern = "HH:mm:ss";

	static {
		// functions.propertiesの内容で設定
		Properties properties = FunctionsProperties.getProperties();

		if(properties != null){
			setDatePattern(properties.getProperty("datePattern"));
			setDatetimePattern(properties.getProperty("datetimePattern"));
			setTimePattern(properties.getProperty("timePattern"));
		}
	}

	/**
	 * 日付のフォーマットパターンを設定します。
	 * デフォルトは<code>&quot;yyyy/MM/dd&quot;</code>です。
	 * <p>
	 * このメソッドで設定したフォーマットパターンは{@link #formatDate(Date)}で使用されます。
	 *
	 * @param datePattern 日付のフォーマットパターン
	 */
	public static void setDatePattern(String datePattern){
		DateUtils.datePattern = datePattern;
	}

	/**
	 * 日時のフォーマットパターンを設定します。
	 * デフォルトは<code>&quot;yyyy/MM/dd HH:mm:ss&quot;</code>です。
	 * <p>
	 * このメソッドで設定したフォーマットパターンは{@link #formatDatetime(Date)}で使用されます。
	 *
	 * @param datetimePattern 日時のフォーマットパターン
	 */
	public static void setDatetimePattern(String datetimePattern){
		DateUtils.datetimePattern = datetimePattern;
	}

	/**
	 * 時間のフォーマットパターンを設定します。
	 * デフォルトは<code>&quot;HH:mm:ss&quot;</code>です。
	 * <p>
	 * このメソッドで設定したフォーマットパターンは{@link #formatTime(Date)}で使用されます。
	 *
	 * @param timePattern 時間のフォーマットパターン
	 */
	public static void setTimePattern(String timePattern){
		DateUtils.timePattern = timePattern;
	}

	/**
	 * 指定したフォーマットで<code>java.util.Date</code>オブジェクトをフォーマットします。
	 *
	 * @param date フォーマットする<code>java.util.Date</code>オブジェクト
	 * @param pattern フォーマットパターン（<code>java.text.SimpleDateFormat</code>で指定可能なパターン）
	 * @return フォーマットされた文字列。引数<code>date</code>が<code>null</code>だった場合は空文字列
	 */
	public static String formatDate(Date date, String pattern){
		if(date == null){
			return "";
		}
		SimpleDateFormat formatter = new SimpleDateFormat(pattern);
		return formatter.format(date);
	}

	/**
	 * <code>java.util.Date</code>オブジェクトを日付形式にフォーマットします。
	 * <p>
	 * デフォルトでは<code>&quot;yyyy/MM/dd&quot;</code>形式にフォーマットされますが、
	 * {@link #setDatePattern(String)}メソッドでパターンを設定することも可能です。
	 *
	 * @param date フォーマットする<code>java.util.Date</code>オブジェクト
	 * @return フォーマットされた文字列。引数<code>date</code>が<code>null</code>だった場合は空文字列
	 */
	public static String formatDate(Date date){
		return formatDate(date, datePattern);
	}

	/**
	 * <code>java.util.Date</code>オブジェクトを日時形式にフォーマットします。
	 * <p>
	 * デフォルトでは<code>&quot;yyyy/MM/dd HH:mm:SS&quot;</code>形式にフォーマットされますが、
	 * {@link #setDatetimePattern(String)}メソッドでパターンを設定することも可能です。
	 *
	 * @param date フォーマットする<code>java.util.Date</code>オブジェクト
	 * @return フォーマットされた文字列。引数<code>date</code>が<code>null</code>だった場合は空文字列
	 */
	public static String formatDatetime(Date date){
		return formatDate(date, datetimePattern);
	}

	/**
	 * <code>java.util.Date</code>オブジェクトを時間形式にフォーマットします。
	 * <p>
	 * デフォルトでは<code>&quot;HH:mm:SS&quot;</code>形式にフォーマットされますが、
	 * {@link #setTimePattern(String)}メソッドでパターンを設定することも可能です。
	 *
	 * @param date フォーマットする<code>java.util.Date</code>オブジェクト
	 * @return フォーマットされた文字列。引数<code>date</code>が<code>null</code>だった場合は空文字列
	 */
	public static String formatTime(Date date){
		return formatDate(date, timePattern);
	}
}
