#!/usr/local/bin/python

# dump-dbcs.py (c) 2008-2010 exeal
#
# Converts UTR #22 based XML file into 16x16x256 native-to-UCS or UCS-to-native
# mapping array and prints the result to the standard output.
# Do not use with non-DBCS encoding file.

import sys
import datetime
import re
import xml.sax
import xml.sax.handler

if len(sys.argv) != 3 or not re.match('-(?:u2n|n2u)', sys.argv[1]):
	sys.exit('usage: python dump-dbcs.py (-u2n|-n2u) <input-file-name>')

NATIVE_REPLACEMENT_BYTES = '0x0000'
UNICODE_REPLACEMENT_CHARACTER = '0xfffd'
u2n_mapping = sys.argv[1] == '-u2n'
input_filename = sys.argv[2]
table = {}

class MyContentHandler(xml.sax.ContentHandler):
	def startElement(self, name, attributes):
		if name == 'a':
			m = re.match(r'([0-9A-Fa-f]{1,2})\s([0-9A-Fa-f]{1,2})', attributes.getValue('b'))
			if m != None:
				dbcs = (int(m.group(1), 16) << 8) | int(m.group(2), 16)
			else:
				m = re.match(r'[0-9A-Fa-f]{1,2}', attributes.getValue('b'))
				if m != None:
					dbcs = int(m.group(), 16)
				else:
					sys.exit('error: unknown byte sequence.')
			if u2n_mapping:
				table[int(attributes.getValue('u'), 16)] = dbcs
			else:
				table[dbcs] = int(attributes.getValue('u'), 16)

sax_reader = xml.sax.make_parser()
sax_reader.setFeature(xml.sax.handler.feature_validation, False)
sax_reader.setFeature(xml.sax.handler.feature_external_ges, False)
sax_reader.setContentHandler(MyContentHandler())
sax_reader.parse(input_filename)

# verify 7-bit ASCII compatibility
for c in range(0x00, 0x80):
	if c in table and table[c] != c:
		exit('error: This encoding is not compatible with US-ASCII.')

# dump mapping table
print('// automatically generated by `dump-dbcs.py %s "%s"` at %s' % (sys.argv[1], input_filename, datetime.datetime.now()))
print('// contains', len(table), 'characters')
high_min, high_max, low_min, low_max = 0xff, 0x00, 0xff, 0x00
for high in range(0x00, 0x100):
	empty_lines = [True, True, True, True, True, True, True, True, True, True, True, True, True, True, True, True]
	for low in range(0x00, 0x100):
		if((low | (high << 8)) in table):
			empty_lines[low >> 4] = False
	if all(empty_lines):
		sys.stdout.write('0')
	else:
		if high != 0x00:
			high_min, high_max = min(high, high_min), max(high, high_max)
		if u2n_mapping:
			sys.stdout.write('CharWire<\n')
		else:
			sys.stdout.write('DBCSWire<\n')
		low = 0x00
		while low < 0x100:
			if low % 16 == 0:
				if empty_lines[low >> 4]:
					if u2n_mapping:
						sys.stdout.write('\tEmptyCharLine')
					else:
						sys.stdout.write('\tEmptyDBCSLine')
					low += 16
					if low < 0x100:
						sys.stdout.write(',')
					sys.stdout.write('\n')
					continue
				if u2n_mapping:
					sys.stdout.write('\tCharLine<')
				else:
					sys.stdout.write('\tDBCSLine<')
			c = low | (high << 8)
			if c in table:
				sys.stdout.write('0x%04x' % table[c])
				if high != 0x00:
					low_min, low_max = min(low, low_min), max(low, low_max)
			elif u2n_mapping:
				sys.stdout.write(NATIVE_REPLACEMENT_BYTES)
			else:
				sys.stdout.write(UNICODE_REPLACEMENT_CHARACTER)
			if low % 16 != 15:
				sys.stdout.write(',')
			elif low != 0xff:
				sys.stdout.write('>,\n')
			else:
				sys.stdout.write('>\n')
			low += 1
		sys.stdout.write('>::VALUES')
	if high != 0xff:
		sys.stdout.write(',')
	sys.stdout.write('\n')

if not u2n_mapping:
	# report bytes range
	print(r'// lead byte range : [\x%02x-\x%02x]' % (high_min, high_max))
	print(r'// trail byte range : [\x%02x-\x%02x]' % (low_min, low_max))
