#!/usr/local/bin/perl

use strict;
use warnings;
use integer;
use IO::File;

# converts a list of code points to a C++ wide string literal
sub cp2str(@) {
	my $result = '';
	for(@_) {
		my $cp = hex $_;
		if($cp < 0x010000) {
			$result .= sprintf('\x%X', $cp);
		} else {
			$result .= sprintf('\x%X\x%X', (($cp >> 10) & 0xFFFF) + 0xD7C0, ($cp & 0x03FF) | 0xDC00);
		}
	}
	return 'L"' . $result . '"';
}

sub s2str($) {
	return cp2str(split(' ', shift));
}

my $fileName = shift @ARGV;
my $input = new IO::File($fileName) or die("Can't open the file $fileName.\n");
my $out = new IO::File('>normalization-test.cpp') or die("Can't open the output file.\n");

print $out '// automatically generated by `perl gen-uax15-test.pl` at $ ' . scalar(localtime) . " \$\n\n";
print $out <<'HEADER';
// normalization-test.cpp

#include "stdafx.h"
#include "../unicode.hpp"
using namespace ascension;
using namespace ascension::unicode;

namespace {
	void testPair(const String& lhs, const String& rhs, Normalizer::Form form) {
		BOOST_CHECK_EQUAL(lhs, Normalizer::normalize(StringCharacterIterator(rhs), form));
		BOOST_CHECK_EQUAL(Normalizer::compare(lhs, rhs, form, CASE_SENSITIVE), 0);
	}
	void testLine(const String& s1, const String& s2, const String& s3, const String& s4, const String& s5) {
		// NFC
		testPair(s2, s1, Normalizer::FORM_C);
		testPair(s2, s2, Normalizer::FORM_C);
		testPair(s2, s3, Normalizer::FORM_C);
		testPair(s4, s4, Normalizer::FORM_C);
		testPair(s4, s5, Normalizer::FORM_C);
		// NFD
		testPair(s3, s1, Normalizer::FORM_D);
		testPair(s3, s2, Normalizer::FORM_D);
		testPair(s3, s3, Normalizer::FORM_D);
		testPair(s5, s4, Normalizer::FORM_D);
		testPair(s5, s5, Normalizer::FORM_D);
		// NFKC
		testPair(s4, s1, Normalizer::FORM_KC);
		testPair(s4, s2, Normalizer::FORM_KC);
		testPair(s4, s3, Normalizer::FORM_KC);
		testPair(s4, s4, Normalizer::FORM_KC);
		testPair(s4, s5, Normalizer::FORM_KC);
		// NFKD
		testPair(s5, s1, Normalizer::FORM_KD);
		testPair(s5, s2, Normalizer::FORM_KD);
		testPair(s5, s3, Normalizer::FORM_KD);
		testPair(s5, s4, Normalizer::FORM_KD);
		testPair(s5, s5, Normalizer::FORM_KD);
	}
}

void testNormalization() {
HEADER

while(<$input>) {
	next unless /^([^;]+);([^;]+);([^;]+);([^;]+);([^;]+);/;
	printf($out "\ttestLine(%s, %s, %s, %s, %s);\n", s2str($1), s2str($2), s2str($3), s2str($4), s2str($5));
}

$input->close;

print $out "}\n";

$out->close;
