<?php
/**
 * DSN Class
 * @package	Adelie
 * @copyright	Copyright (c) 2012, Adelie Development Team
 */
namespace Adelie;

final class Dsn
{
	const DRIVERS = "mysql,sqlite,pgsql";
	const PARAM_MYSQL = "host,port,dbname,unix_socket,charset,user,password,namespace";
	const PARAM_SQLITE = "host,namespace";
	const PARAM_PGSQL = "host,port,dbname,user,password,namespace";

	private $vals;

	/**
	 * Constructor
	 * @access	public
	 * @param	string	$dsn
	 */
	public function __construct ($dsn)
	{
		$this->vals = array();
		$this->parse($dsn);
	}

	/**
	 * DSNパラメータを取得
	 * @access	public
	 * @param	string	$name
	 * @param	array	$args
	 * @return	string
	 */
	public function __call($name, Array $args)
	{
		$key = namespace\String::toUnderScore(\substr($name, 3));
		return isset($this->vals[$key]) ? $this->vals[$key] : "";
	}

	/**
	 * DSNをパース
	 * @access	private
	 * @param	string	$dsn	DSN文字列
	 * @return	void
	 */
	private function parse ($dsn)
	{
		$tmp = \explode(";", \trim($dsn, "; "));
		foreach ($tmp as $item) {
			list($key, $val) = \explode("=", \trim($item));
			$key = \strtolower(\trim($key));
			$this->vals[$key] = \trim($val);
		}
	}

	/**
	 * create DB object name
	 * @access	public
	 * @param	boolean	$namespace
	 * @return	string
	 */
	public function createDbObjName ($namespace=false)
	{
		if ($namespace) {
			return \sprintf("\\%s\\Db%s", __NAMESPACE__, \ucwords(@$this->vals["driver"]));
		} else {
			return \sprintf("Db%s", \ucwords(@$this->vals["driver"]));
		}
	}

	/**
	 * create DB information object name
	 * @access	public
	 * @param	boolean	$namespace
	 * @return	string
	 */
	public function createDbInfoObjName ($namespace=false)
	{
		if ($namespace) {
			return \sprintf("\\%s\\Db%sInfo", __NAMESPACE__, \ucwords(@$this->vals["driver"]));
		} else {
			return \sprintf("Db%sInfo", \ucwords(@$this->vals["driver"]));
		}
	}

	/**
	 * create DB object path
	 * @access	public
	 * @return	string
	 */
	public function createDbObjPath ()
	{
		return \sprintf("phar://%s/core/db/%s.php", namespace\PHAR, $this->createDbObjName());
	}

	/**
	 * create DB information object path
	 * @access	public
	 * @return	string
	 */
	public function createDbInfoObjPath ()
	{
		return \sprintf("phar://%s/core/db/%s.php", namespace\PHAR, $this->createDbInfoObjName());
	}

	/**
	 * get a namespace of dao
	 * @access	public
	 * @return	string
	 */
	public function createNamespace ()
	{
		switch (\strtolower(@$this->vals["driver"])) {
			case "mysql":
			case "pgsql":
				$namespace = isset($this->vals["dbname"]) ? \strtolower($this->vals["dbname"]) : "";
				$namespace = namespace\String::toPascal($namespace);
				break;
			case "sqlite":
				$namespace = isset($this->vals["host"]) ? \strtolower($this->vals["host"]) : "";
				$tmp = \explode(".", \basename($namespace));
				$namespace = namespace\String::toPascal($tmp[0]);
				break;
			default:
				$namespace = "";
				break;
		}
		return $namespace;
	}

	/**
	 * PDO用のDSNを生成
	 * @access	public
	 * @return	string
	 */
	public function create ()
	{
		$driver = isset($this->vals["driver"]) ? $this->vals["driver"] : "";
		switch ($driver) {
			case "sqlite":
				$path = @$this->vals["host"];
				$dsn = "sqlite:{$path}";
				break;
			case "mysql";
			case "pgsql":
				$valids = \explode(",", \constant("self::PARAM_".\strtoupper($driver)));
				$dsn = "{$driver}:";
				foreach ($this->vals as $key=>$val) {
					if (!\in_array($key, $valids)) { continue; }
					$dsn .= "{$key}={$val};";
				}
				break;
			default:
				throw new \UnexpectedValueException("Invalid DSN parameter.");
		}
		return $dsn;
	}
}

