<?php
/**
 * Abstract DB Class
 * @package	Adelie
 * @copyright	Copyright (c) 2012, Adelie Development Team
 */
namespace Adelie;

require_once \sprintf("phar://%s/core/db/DbException.php", namespace\PHAR);
require_once \sprintf("phar://%s/core/db/DbResult.php", namespace\PHAR);

abstract class AbstractDb
{
	protected static $instance;

	protected $dsn;
	protected $pdo;
	protected $opts;
	protected $stmt;
	protected $row;
	protected $binds;
	protected $info; // DbInfo

	/**
	 * constructor
	 * @access	protected
	 * @param	string	$dsn	\Adelie\DSN
	 * @param	array	$opts	array of driver-specific connection options
	 * @return	void
	 */
	protected function __construct (Dsn $dsn, Array $opts=array())
	{
		$this->dsn = $dsn;
		$this->pdo = null;
		$this->opts = $opts + array(
			\PDO::ATTR_CASE => \PDO::CASE_LOWER,
			\PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION,
			\PDO::ATTR_DEFAULT_FETCH_MODE => \PDO::FETCH_OBJ
		);
		$this->stmt = null;
		$this->row = array();
		$this->binds = array();
		$this->info = null;
	}

	/**
	 * 複製禁止
	 * @access	public
	 * @return	void
	 */
	public function __clone ()
	{
		$msg = namespace\Error::get(namespace\Error::E_CLONING);
		throw new namespace\AdelieException($msg, namespace\Error::E_CLONING);
	}

	/**
	 * create instance
	 * @access	public
	 * @static
	 * @param	\Adelie\Dsn	$dsn
	 * @return	\Adelie\AbstractDb
	 */
	public static function getInstance (namespace\Dsn $dsn)
	{
		$id = \sha1(\serialize($dsn));
		if (static::$instance[$id]==null) {
			$obj = $dsn->createDbObjName(true);
			static::$instance[$id] = new $obj($dsn);
		}
		return static::$instance[$id];
	}

	/**
	 * connect
	 * @access	protected
	 * @param	\Adelie\Dsn	$dsn
	 * @return	void
	 */
	protected function connect ()
	{
		$msg = namespace\Error::get(namespace\Error::E_INHERIT, \get_class($this), __CLASS__);
		throw new namespace\AdelieException($msg, namespace\Error::E_INHERIT);
	}

	/**
	 * 接続されているかを判定
	 * @access	protected
	 * @return	boolean
	 */
	protected function isConnected()
	{
		return ($this->pdo instanceof \PDO) ? true : false;
	}

	/**
	 * prepares a statement for execution
	 * @access	public
	 * @param	string	$sql
	 * @return	void
	 * @throws	\PDOException
	 */
	public function prepare ($sql, $opts=array())
	{
		if (!$this->isConnected()) {
			$this->connect();
		}
		try {
			$this->stmt = $this->pdo->prepare($sql, $opts);
		} catch (\PDOException $e) {
			throw new namespace\AdelieException($e->getMessage(), $e->getCode(), $e);
		}
	}

	/**
	 * binds a value to a parameter
	 * @access	public
	 * @param	string	$key
	 * @param	mixed	$value
	 * @param	integer	$type
	 * @param	integer	$len
	 * @return	void
	 */
	public function bind ($key, $value, $type=\PDO::PARAM_STR)
	{
		$this->binds[$key] = array(
			"val" => $value,
			"type" => $type
		);
	}

	/**
	 * executes a prepared statement
	 * @access	public
	 * @return	\PDOStatement
	 * @throws	\PDOException
	 */
	public function exec ()
	{
		try {
			foreach ($this->binds as $key=>$val) {
				$value = $val["val"];
				$type = $val["type"];
				$this->stmt->bindValue(":{$key}", $value, $type);
			}
			$start = \microtime(true);
			$this->stmt->execute();
			$time = \microtime(true) - $start;
			return new namespace\DbResult($this->stmt, $this->binds, $time);
		} catch (\PDOException $e) {
			$this->error($e);
			throw $e;
		}
	}

	/**
	 * fetches the next row and returns it as an object.
	 * @access	public
	 * @return	boolean
	 */
	public function fetch ()
	{
		return ($this->row=$this->stmt->fetchObject()) ? true : false;
	}

	/**
	 * get variables
	 * @access	public
	 * @param	string	$column
	 * @return	string
	 */
	public function get ($column)
	{
		return isset($this->row->{$column}) ? $this->row->{$column} : "";
	}

	/**
	 * get a row
	 * @access	public
	 * @return	array
	 */
	public function getRow ()
	{
		return $this->row;
	}

	/**
	 * returns the ID of the last inserted row or sequence value
	 * @access	public
	 * @param	string	$seq
	 * @return	integer
	 */
	public function getKey ($seq="")
	{
		if (\strlen($seq)) {
			return $this->pdo->lastInsertId($seq);
		} else {
			return $this->pdo->lastInsertId();
		}
	}

	/**
	 * frees the memory associated with a result
	 * @access	public
	 * @return	void
	 */
	public function free ()
	{
		$this->stmt = null;
		$this->row = null;
		$this->binds = array();
	}

	/**
	 * closes a previously opened database connection
	 * @access	public
	 * @return	void
	 */
	public function close ()
	{
		$id = \sha1(\serialize($this->dsn));
		if (isset(self::$instance[$id])) {
			unset(self::$instance[$id]);
		}
	}

	/**
	 * エラーSQLをDBにロギング
	 * @access	private
	 * @param	\PDOException	$e
	 * @return	void
	 */
	private function error (\PDOException $e)
	{
		$sql  = "INSERT INTO sql_error ( ";
		$sql .=   "err_trace, err_msg, err_stmt, err_date, del_flg ";
		$sql .= ") VALUES ( ";
		$sql .=   ":err_trace, :err_msg, :err_stmt, :err_date, 0 ";
		$sql .= ") ";

		// トレース
		$traces = array();
		$tmp = $e->getTrace();
		foreach ($tmp as $val) {
			$file = isset($val["file"]) ? \basename($val["file"]) : $file;
			if ($file=="Dispatcher.php") { break; } // Dispatcher以降は取得しない
			$line = isset($val["line"]) ? $val["line"] : $line;
			$obj = \sprintf("%s%s%s", @$val["class"], @$val["type"], @$val["function"]);
			$traces[] = \sprintf("%s[%d] %s", $file, $line, $obj);
		}
		$err_trace = \implode("\n", $traces);

		// SQLステートメント
		$binds = array();
		foreach ($this->binds as $key=>$val) {
			$binds[] = \sprintf("\"%s\" = %s", $key, $val["val"]);
		}
		$err_sql = $this->stmt->queryString;
		if (\count($binds)) {
			$err_sql .= "\n[Bind Data]\n" . \implode("\n", $binds);
		}

		try {
			$stmt = $this->pdo->prepare($sql);
			$err_msg = $e->getMessage();
			$err_date = \date("Y-m-d H:i:s", namespace\LOCAL_TIME);
			$stmt->bindValue(":err_trace", $err_trace);
			$stmt->bindValue(":err_msg", $err_msg);
			$stmt->bindValue(":err_stmt", $err_sql);
			$stmt->bindValue(":err_date", $err_date);
			$stmt->execute();
		} catch (\PDOException $e) {
			// ここでエラーが起きても処理できないので、何もしない。
			// ログぐらい吐いてもいいかも。
		}
	}

	/**
	 * データベース名を取得
	 * @access	public
	 * @return	string
	 */
	public function getDbName ()
	{
		return $this->dsn->getDbname();
	}

	/**
	 * get driver name
	 * @access	public
	 * @return	string
	 * @todo	check PDO constant name
	 */
	public function getDriver ()
	{
		return $this->dsn->getDriver();
	}

	/**
	 * 直近のSQLで影響のあった行数を取得
	 * @access	public
	 * @return	integer
	 */
	public function getRowCount ()
	{
		return \is_callable(array($this->stmt, "rowCount")) ? $this->stmt->rowCount() : 0;
	}

	/**
	 * get database information class
	 * @access	public
	 * @return	\Adelie\AbstractDbInfo
	 */
	public function getInfo ()
	{
		if (!($this->info instanceof namespace\AbstractDbInfo)) {
			$obj = $this->dsn->createDbInfoObjName(true);
			$this->info = new $obj($this);
		}
		return $this->info;
	}
}
