<?php
/**
 * Response Class
 * @package	Adelie
 * @copyright	Copyright (c) 2012, Adelie Development Team
 */
namespace Adelie;

require_once \sprintf("phar://%s/core/View.php", namespace\PHAR);

final class Response
{
	const RES_NOTHING = 0; // 何もしない
	const RES_DISPLAY = 1; // 表示
	const RES_OUTPUT = 2; // バイナリ出力
	const RES_REDIRECT = 3; // リダイレクト
	const RES_STATUS = 4; // ステータス返却

	const DEFAULT_STATUS = 200;

	private static $instance;

	private $view;
	private $response;
	private $status;
	private $mime;
	private $http_status;

	/**
	 * constructor
	 * @access	protected
	 * @param	\Adelie\Request	$request
	 */
	protected function __construct (namespace\Request $request)
	{
		$this->status = self::DEFAULT_STATUS;
		$this->mime = @\ini_get("default_mimetype");
		$this->response = self::RES_NOTHING;
		$this->view = namespace\View::getInstance($request);
		$this->http_status = array(
			100 => array("ver"=>"1.1", "msg"=>"Continue"),
			101 => array("ver"=>"1.1", "msg"=>"Switching Protocols"),
			200 => array("ver"=>"1.0", "msg"=>"OK"),
			201 => array("ver"=>"1.0", "msg"=>"Created"),
			202 => array("ver"=>"1.0", "msg"=>"Accepted"),
			203 => array("ver"=>"1.1", "msg"=>"Non-Authoritative Information"),
			204 => array("ver"=>"1.0", "msg"=>"No Content"),
			205 => array("ver"=>"1.1", "msg"=>"Reset Content"),
			206 => array("ver"=>"1.1", "msg"=>"Partial Content"),
			300 => array("ver"=>"1.1", "msg"=>"Multiple Choices"),
			301 => array("ver"=>"1.0", "msg"=>"Moved Permanently"),
			302 => array("ver"=>"1.0", "msg"=>"Moved Temporarily"),
			303 => array("ver"=>"1.1", "msg"=>"See Other"),
			304 => array("ver"=>"1.0", "msg"=>"Not Modified"),
			305 => array("ver"=>"1.1", "msg"=>"Use Proxy"),
			400 => array("ver"=>"1.0", "msg"=>"Bad Request"),
			401 => array("ver"=>"1.0", "msg"=>"Unauthorized"),
			402 => array("ver"=>"1.1", "msg"=>"Payment Required"),
			403 => array("ver"=>"1.0", "msg"=>"Forbidden"),
			404 => array("ver"=>"1.0", "msg"=>"Not Found"),
			405 => array("ver"=>"1.1", "msg"=>"Method Not Allowed"),
			406 => array("ver"=>"1.1", "msg"=>"Not Acceptable"),
			407 => array("ver"=>"1.1", "msg"=>"Proxy Authentication Required"),
			408 => array("ver"=>"1.1", "msg"=>"Request Time-out"),
			409 => array("ver"=>"1.1", "msg"=>"Conflict"),
			410 => array("ver"=>"1.1", "msg"=>"Gone"),
			411 => array("ver"=>"1.1", "msg"=>"Length Required"),
			412 => array("ver"=>"1.1", "msg"=>"Precondition Failed"),
			413 => array("ver"=>"1.1", "msg"=>"Request Entity Too Large"),
			414 => array("ver"=>"1.1", "msg"=>"Request-URI Too Large"),
			415 => array("ver"=>"1.1", "msg"=>"Unsupported Media Type"),
			500 => array("ver"=>"1.0", "msg"=>"Internal Server Error"),
			501 => array("ver"=>"1.0", "msg"=>"Not Implemented"),
			502 => array("ver"=>"1.0", "msg"=>"Bad Gateway"),
			503 => array("ver"=>"1.0", "msg"=>"Service Unavailable"),
			504 => array("ver"=>"1.1", "msg"=>"Gateway Time-out"),
			505 => array("ver"=>"1.1", "msg"=>"HTTP Version not supported")
		);
	}

	/**
	 * get an instance
	 * @access	public
	 * @static
	 * @param	\Adelie\Request	$request
	 * @return	\Adelie\Response
	 */
	public static function getInstance (namespace\Request $request)
	{
		if (self::$instance==null) {
			self::$instance = new self($request);
		}
		return self::$instance;
	}

	/**
	 * set HTTP status
	 * @access	public
	 * @param	integer	$status
	 * @param	boolean	$
	 * @return	void
	 * @todo	ResponseExceptionの置き換え
	 */
	public function setStatus ($status, $change=true)
	{
		if (isset($this->http_status[$status])) {
			$this->response = self::RES_STATUS;
			$this->status = $status;
		} else {
			throw new ResponseException("不正なステータスが設定されました.");
		}
	}

	/**
	 * HTTPステータスを出力
	 * @access	private
	 * @return	void
	 */
	private function status ()
	{
		$ver = isset($this->http_status[$this->status]["ver"]) ? $this->http_status[$this->status]["ver"] : "";
		$msg = isset($this->http_status[$this->status]["msg"]) ? $this->http_status[$this->status]["msg"] : "";
		if (!\headers_sent() && \strlen($ver) && \strlen($msg)) {
			\header("HTTP/{$ver} {$this->status} {$msg}", true, $this->status);
		}
	}

	/**
	 * MIMEを設定
	 * @access	public
	 * @param	string	$mime
	 * @return	void
	 */
	public function setMime ($mime)
	{
		$this->mime = $mime;
	}

	/**
	 * ダウンロード宣言
	 * @access	public
	 * @param	string	$filename
	 * @param	integer	$status
	 * @return	void
	 * @todo	ResponseExceptionの置き換え
	 */
	public function download ($filename, $status=self::DEFAULT_STATUS)
	{
		$this->setStatus($status);
		$this->status();
		$this->response = self::RES_OUTPUT;
		if (\headers_sent()) {
			throw new ResponseException("すでに出力が開始されているため、ヘッダの書き換えはできません.");
		}
		\header("Content-type: application/octet-stream");
		\header("Content-Disposition: attachment; filename=\"{$filename}\"");
	}

	/**
	 * 出力宣言
	 * @access	public
	 * @param	string	$mime
	 * @param	integer	$status
	 * @return	void
	 * @todo	ResponseExceptionの置き換え
	 */
	public function output ($mime="application/octet-stream", $status=self::DEFAULT_STATUS)
	{
		$this->setStatus($status);
		$this->status();
		$this->response = self::RES_OUTPUT;
		if (\headers_sent()) {
			throw new ResponseException("すでに出力が開始されているため、ヘッダの書き換えはできません.");
		}
		\header("Content-type: {$mime}");
	}

	/**
	 * redirect
	 * @access	public
	 * @param	string	$uri
	 * @param	boolean	$permanently
	 * @return	void
	 * @todo	ResponseExceptionの置き換え
	 */
	public function redirect ($uri, $permanently=false)
	{
		if (\headers_sent()) {
			throw new ResponseException("すでに出力が開始しているため、ヘッダの書き換えはできません.");
		}
		if ($permanently) {
			$this->setStatus(301);
			$this->status();
		}
		\header("Location: {$uri}");
		exit;
	}

	/**
	 * 変数をassign
	 * @access	private
	 * @param	string	$key
	 * @param	mixed	$value
	 * @return	void
	 */
	public function assign ($key, $value)
	{
		$this->view->assign($key, $value);
	}

	/**
	 * テンプレートを設定
	 * @access	public
	 * @param	string	$tpl
	 * @param	integer	$type
	 * @param	integer	$status
	 * @return	void
	 */
	public function display ($tpl, $status=self::DEFAULT_STATUS)
	{
		$this->setStatus($status);
		$this->response = self::RES_DISPLAY;
		$this->view->setTemplate($tpl);
	}

	/**
	 * 結果処理を行う
	 * @access	public
	 * @return	void
	 */
	public function respond ()
	{
		switch ($this->response) {
			case self::RES_DISPLAY:
				$this->status();
				$this->view->render();
				break;
			case self::RES_STATUS:
				$this->status();
				break;
			case self::RES_REDIRECT:
			case self::RES_OUTPUT:
			case self::RES_NOTHING:
			default:
				break;
		}
	}
}
