<?php
/**
 * Request Class
 * @package	Adelie
 * @copyright	Copyright (c) 2012, Adelie Development Team
 */
namespace Adelie;

require_once \sprintf("phar://%s/lib/ExternalVariable.php", namespace\PHAR);

final class Request
{
	private static $instance;

	private $site_id;
	private $params;
	private $param_count;
	private $path;
	private $controller;
	private $action;

	/**
	 * Constructor
	 * @access	protected
	 * @param	string	$site_id
	 */
	protected function __construct ($site_id)
	{
		$this->site_id = $site_id;
		$this->params = array();
		$this->param_count = 0;
		$this->path = isset($_SERVER["PATH_INFO"]) ? $_SERVER["PATH_INFO"] : "";
		$this->controller = "";
		$this->action = "";
		$this->parse();
	}

	/**
	 * create instance
	 * @static
	 * @access	public
	 * @param	string	$site_id
	 * @return	Request
	 */
	public static function getInstance ($site_id)
	{
		if (!(self::$instance instanceof namespace\Request)) {
			self::$instance = new self($site_id);
		}
		return self::$instance;
	}

	/**
	 * parse
	 * @access	private
	 * @return	void
	 */
	private function parse ()
	{
		// Parameter
		$paths = \explode("/", \trim($this->path, "/"));
		$this->param_count = \count($paths);
		foreach ($paths as $idx=>$path) {
			$tmp = \explode(namespace\PARAM_DELIMITER, $path);
			if (\count($tmp)>1) {
				$key = \array_shift($tmp);
				$delimiter = (\count($tmp)>1) ? namespace\PARAM_DELIMITER : "";
				$val = \implode($delimiter, $tmp);
				$this->set($key, $val);
			} else {
				$val = $path;
			}
			$this->set($idx, $val);
		}
	}

	/**
	 * set the controller
	 * @access	public
	 * @param	string	$controller
	 * @return	void
	 */
	public function setController ($controller=namespace\DEFAULT_CONTROLLER)
	{
		$this->controller = namespace\String::toPascal($controller);
	}

	/**
	 * get the controller
	 * @access	public
	 * @return	string
	 */
	public function getController ()
	{
		return $this->controller;
	}

	/**
	 * get the name of controller class
	 * @access	public
	 * @param	boolean	$ns_flg
	 * @return	string
	 */
	public function getControllerName ($ns_flg=false)
	{
		if ($ns_flg) {
			return \sprintf("\\%s\\%sController", __NAMESPACE__, $this->controller);
		} else {
			return "{$this->controller}Controller";
		}
	}

	/**
	 * get the path of controller file.
	 * @access	public
	 * @return	string
	 */
	public function getControllerPath ()
	{
		$site_id = $this->site_id ? "/{$this->site_id}" : "";
		$controller_path = \sprintf("%s%s/%s.php", namespace\DIR_CTL, $site_id, $this->getControllerName());
		return \realpath($controller_path);
	}

	/**
	 * set the action
	 * @access	public
	 * @param	string	$action
	 * @return	void
	 */
	public function setAction ($action=namespace\DEFAULT_ACTION)
	{
		$this->action = namespace\String::toPascal($action);
	}

	/**
	 * get the action
	 * @access	public
	 * @return	string
	 */
	public function getAction ()
	{
		return	$this->action;
	}

	/**
	 * get the action method
	 * @access	public
	 * @return	string
	 */
	public function getActionMethod ()
	{
		return	"do{$this->action}Action";
	}

	/**
	 * set a parameter from external source
	 * @access	private
	 * @param	string	$key
	 * @param	mixed	$val
	 * @return	void
	 */
	private function set ($key, $val)
	{
		if (isset($this->params[$key])) {
			if (!\is_array($this->params[$key])) {
				$this->params[$key] = (array)$this->params[$key];
			}
			$this->params[$key][] = $val;
		} else {
			$this->params[$key] = $val;
		}
	}

	/**
	 * get a valiable from external source as integer
	 * @access	public
	 * @param	string	$key
	 * @return	integer
	 */
	public function getInt ($key)
	{
		if (!isset($this->params[$key])) {
			return 0;
		}
		return \intval($this->params[$key]);
	}

	/**
	 * get a valiable from external source as floating point number
	 * @access	public
	 * @param	string	$key
	 * @return	float
	 */
	public function getDbl ($key)
	{
		if (!isset($this->params[$key])) {
			return 0.0;
		}
		return \doubleval($this->params[$key]);
	}

	/**
	 * get a valiable from external source as string
	 * @access	public
	 * @param	string	$key
	 * @param	boolean	$lines	falseの時、改行を削除
	 * @return	string
	 */
	public function getStr ($key, $lines=false)
	{
		if (!isset($this->params[$key])) {
			return "";
		}
		return \strval(\str_replace(array("\r", "\n", "\0"), "", $this->params[$key]));
	}

	/**
	 * get a valiable from external source as boolean
	 * @access	public
	 * @param	string	$key
	 * @return	boolean
	 */
	public function getBool ($key)
	{
		if (!isset($this->params[$key])) {
			return false;
		}
		return \filter_var($this->params[$key], \FILTER_VALIDATE_BOOLEAN);
	}

	/**
	 * get the valiable count from external source as floating point number
	 * @access	public
	 * @return	integer
	 */
	public function getCount()
	{
		return $this->param_count;
	}

	/**
	 * get the external variable object
	 * @access	public
	 * @return	\Adelie\ExternalVariable
	 */
	public function getExternalVariable ()
	{
		return namespace\ExternalVariable::getInstance();
	}

	/**
	 * get site ID
	 * @access	public
	 * @return	string
	 */
	public function getSiteId ()
	{
		return $this->site_id;
	}

	/**
	 * リクエストパスを取得
	 * @access	public
	 * @return	string
	 */
	public function getPath ()
	{
		return $this->path;
	}
}
