/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.math.BigInteger;

import net.morilib.lisp.sos.LispType;
import net.morilib.util.Inclimentor;

public abstract class LispInteger extends LispExactReal
implements Inclimentor<LispInteger> {
	
	
	private static final BigInteger INT_MAX =
		BigInteger.valueOf(Integer.MAX_VALUE);
	private static final BigInteger INT_MIN =
		BigInteger.valueOf(Integer.MIN_VALUE);
	
	
	public static final LispInteger ZERO = LispInteger.valueOf(0);
	
	
	public static final LispInteger ONE  = LispInteger.valueOf(1);
	
	
	public static LispInteger valueOf(BigInteger val) {
		if(val.compareTo(INT_MAX) > 0 || val.compareTo(INT_MIN) < 0) {
			return new LispBigInt(val);
		} else {
			return new LispSmallInt(val.intValue());
		}
	}
	
	
	public static LispInteger valueOf(int val) {
		return new LispSmallInt(val);
	}
	
	
	public static LispInteger valueOf(long val) {
		if(val > Integer.MAX_VALUE || val < Integer.MIN_VALUE) {
			return new LispBigInt(BigInteger.valueOf(val));
		} else {
			return new LispSmallInt((int)val);
		}
	}
	
	
	public BigInteger getNumerator() {
		return getBigInteger();
	}
	
	
	public BigInteger getDenominator() {
		return BigInteger.ONE;
	}


	@Override
	public boolean isInteger() {
		return true;
	}


	@Override
	public boolean isRational() {
		return true;
	}


	@Override
	public boolean isReal() {
		return true;
	}
	
	
	public boolean isExact() {
		return true;
	}


	/* (non-Javadoc)
	 * @see net.morilib.util.Inclimentor#getObject()
	 */
	public LispInteger getObject() {
		return this;
	}


	/* (non-Javadoc)
	 * @see net.morilib.util.Inclimentor#getZero()
	 */
	public Inclimentor<LispInteger> getZero() {
		return LispInteger.ZERO;
	}


	/* (non-Javadoc)
	 * @see net.morilib.util.Inclimentor#hasNext()
	 */
	public boolean hasNext() {
		return true;
	}
	
	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getType()
	 */
	@Override
	public LispType getType() {
		return LispType.INTEGER;
	}

}
