/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.math.BigDecimal;
import java.math.BigInteger;

import net.morilib.util.Inclimentor;

public final class LispBigInt extends LispInteger {
	
	//
	private BigInteger value;
	
	
	/*package*/ LispBigInt(BigInteger value) {
		if(value == null) {
			throw new NullPointerException();
		}
		this.value = value;
	}
	
	
	public LispNumber add(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return LispComplex.newComplex(
					getRealDouble() + c.getRealDouble(),
					c.getImagDouble());
		} else if(x instanceof LispInteger) {
			return LispInteger.valueOf(value.add(x.getBigInteger()));
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger nd = r.getDenominator();
			BigInteger nn = value.multiply(
					r.getDenominator()).add(r.getNumerator());
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			return new LispDouble(
					value.doubleValue() + ((LispDouble)x).doubleValue());
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public LispNumber div(LispNumber x) {
		if(x instanceof LispComplex) {
			double xr = ((LispComplex)x).getRealDouble();
			double xi = ((LispComplex)x).getImagDouble();
			double n  = getRealDouble();
			
			if(xr == 0.0) {
				return LispComplex.newComplex(0, -n / xi);
			} else {
				return LispComplex.newComplex(
						(n * xr)    / (xr * xr + xi * xi),
						(-(n * xi)) / (xr * xr + xi * xi));
			}
		} else if(x instanceof LispInteger) {
			return LispRational.newRational(value, x.getBigInteger());
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger nd = r.getNumerator();
			BigInteger nn = value.multiply(r.getDenominator());
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			return new LispDouble(
					value.doubleValue() / ((LispDouble)x).doubleValue());
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public LispNumber mul(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			if(c.getRealDouble() == 0.0) {
				return LispComplex.newComplex(
						0, getRealDouble() * c.getImagDouble());
			} else {
				return LispComplex.newComplex(
						getRealDouble() * c.getRealDouble(),
						getRealDouble() * c.getImagDouble());
			}
		} else if(x instanceof LispInteger) {
			return LispInteger.valueOf(
					value.multiply(x.getBigInteger()));
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger nd = r.getDenominator();
			BigInteger nn = value.multiply(r.getNumerator());
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			return new LispDouble(
					value.doubleValue() * ((LispDouble)x).doubleValue());
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public LispNumber sub(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return LispComplex.newComplex(
					getRealDouble() - c.getRealDouble(),
					-c.getImagDouble());
		} else if(x instanceof LispInteger) {
			return LispInteger.valueOf(
					value.subtract(x.getBigInteger()));
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger nd = r.getDenominator();
			BigInteger nn = value.multiply(
					r.getDenominator()).subtract(r.getNumerator());
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			return new LispDouble(
					value.doubleValue() - ((LispDouble)x).doubleValue());
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public LispNumber uminus() {
		return new LispBigInt(value.negate());
	}

	
	public boolean isEqualTo(LispNumber x) {
		if(x instanceof LispComplex) {
			return false;
		} else if(x instanceof LispSmallInt) {
			return false;
		} else if(x instanceof LispBigInt) {
			return value.equals(x.getBigInteger());
		} else if(x instanceof LispRational) {
			return false;
		} else if(x instanceof LispDouble) {
			return value.doubleValue() == ((LispDouble)x).doubleValue();
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public boolean isLessThan(LispReal x) {
		if(x instanceof LispInteger) {
			return value.compareTo(x.getBigInteger()) < 0;
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger n1 = value.multiply(r.getDenominator());
			BigInteger n2 = r.getNumerator();
			
			return n1.compareTo(n2) < 0;
		} else if(x instanceof LispDouble) {
			return value.doubleValue() < x.getRealDouble();
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public boolean isMoreThan(LispReal x) {
		if(x instanceof LispInteger) {
			return value.compareTo(x.getBigInteger()) > 0;
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger n1 = value.multiply(r.getDenominator());
			BigInteger n2 = r.getNumerator();
			
			return n1.compareTo(n2) > 0;
		} else if(x instanceof LispDouble) {
			return value.doubleValue() > x.getRealDouble();
		}
		throw new IllegalArgumentException(x.toString());
	}
	
	
	public int signum() {
		return value.signum();
	}
	
	
	public LispNumber toInexact() {
		return new LispDouble(value.doubleValue());
	}
	
	
	public boolean equals(Object x) {
		if(x instanceof LispBigInt) {
			return value.equals(((LispBigInt)x).value);
		}
		return false;
	}
	
	
	public int hashCode() {
		int l = 17;
		
		l = 37 * l + value.hashCode();
		return l;
	}
	
	
	public String toString() {
		return value.toString();
	}


	public String print() {
		return value.toString();
	}


	public String getResult() {
		return value.toString();
	}
	
	
	public LispString toLispString(int radix) {
		if(radix < 2 || radix > 36) {
			throw new IndexOutOfBoundsException("radix is out of range");
		}
		
		return new LispString(value.toString(radix));
	}
	
	
	public boolean isOne() {
		return value.equals(BigInteger.ONE);
	}


	@Override
	public BigInteger getBigInteger() {
		return value;
	}


	@Override
	public int getInt() {
		return value.intValue();
	}
	
	
	public long getLong() {
		return value.longValue();
	}
	
	
	public BigDecimal getBigDecimal() {
		return new BigDecimal(value);
	}


	@Override
	public double getRealDouble() {
		return value.doubleValue();
	}


	public boolean equalIncliment(Inclimentor<?> i) {
		if(i instanceof LispBigInt) {
			return value.equals(((LispBigInt)i).value);
		} else {
			return i.equalInt(value);
		}
	}


	public boolean equalInt(int i) {
		return value.equals(BigInteger.valueOf(i));
	}


	public boolean equalInt(BigInteger i) {
		return value.equals(i);
	}


	public Inclimentor<LispInteger> suc() {
		return LispInteger.valueOf(value.add(BigInteger.ONE));
	}


	public Inclimentor<LispInteger> suc(int step) {
		return LispInteger.valueOf(
				value.add(BigInteger.valueOf(step)));
	}


	public int toInt() {
		return value.intValue();
	}

}
