/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.math.BigDecimal;
import java.math.BigInteger;

import net.morilib.lisp.sos.LispType;

public final class LispDouble extends LispReal {
	
	
	public static final LispDouble ZERO = new LispDouble(0.0);
	
	
	public static final LispDouble ONE = new LispDouble(1.0);
	
	
	public static final LispDouble POSITIVE_INFINITY =
		new LispDouble(Double.POSITIVE_INFINITY);
	
	
	public static final LispDouble NEGATIVE_INFINITY =
		new LispDouble(Double.NEGATIVE_INFINITY);
	
	
	public static final LispDouble NaN = new LispDouble(Double.NaN);
	
	
	//
	private double number;
	
	
	public LispDouble(double x) {
		this.number = x;
	}
	
	
	public static LispReal toExact(double val) {
		if(Double.isInfinite(val) || Double.isNaN(val)) {
			throw new LispNotSupportedException(
					"err.notsupported.exactinfinity");
		}
		
		BigDecimal v = BigDecimal.valueOf(val);
		return LispUtils.bigDecimalToRational(v);
		/*BigInteger n = v.unscaledValue();
		
		if(v.scale() > 0) {
			BigInteger d = BigInteger.TEN.pow(v.scale());
			
			return LispRational.newRational(n, d);
		} else if(v.scale() < 0) {
			BigInteger d = BigInteger.TEN.pow(-v.scale());
			
			return LispInteger.valueOf(n.multiply(d));
		} else {
			return LispInteger.valueOf(n);
		}*/
	}
	
	
	public BigInteger getNumerator() {
		LispReal r = LispDouble.toExact(number);
		
		return r.getNumerator();
	}
	
	
	public BigInteger getDenominator() {
		LispReal r = LispDouble.toExact(number);
		
		return r.getDenominator();
	}
	
	
	public LispNumber add(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return LispComplex.newComplex(
					number + c.getRealDouble(), c.getImagDouble());
		} else if(x instanceof LispReal) {
			return new LispDouble(number + x.getRealDouble());
		}
		throw new IllegalArgumentException(x.toString());
	}
	
	
	public LispNumber sub(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return LispComplex.newComplex(
					number - c.getRealDouble(), -c.getImagDouble());
		} else if(x instanceof LispReal) {
			return new LispDouble(number - x.getRealDouble());
		}
		throw new IllegalArgumentException(x.toString());
	}
	
	
	public LispNumber mul(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			if(c.getRealDouble() == 0.0) {
				return LispComplex.newComplex(
						0, number * c.getImagDouble());
			} else {
				return LispComplex.newComplex(
						number * c.getRealDouble(),
						number * c.getImagDouble());
			}
		} else if(x instanceof LispReal) {
			return new LispDouble(number * x.getRealDouble());
		}
		throw new IllegalArgumentException(x.toString());
	}
	
	
	public LispNumber div(LispNumber x) {
		if(x instanceof LispComplex) {
			double xr = ((LispComplex)x).getRealDouble();
			double xi = ((LispComplex)x).getImagDouble();
			
			if(xr == 0.0) {
				return LispComplex.newComplex(0, -number / xi);
			} else {
				return LispComplex.newComplex(
						(number * xr)    / (xr * xr + xi * xi),
						(-(number * xi)) / (xr * xr + xi * xi));
			}
		} else if(x instanceof LispReal) {
			return new LispDouble(number / x.getRealDouble());
		}
		throw new IllegalArgumentException(x.toString());
	}
	
	
	public LispNumber uminus() {
		return new LispDouble(-number);
	}
	
	
	public boolean isEqualTo(LispNumber x) {
		if(x instanceof LispComplex) {
			return false;
		} else if(x instanceof LispReal) {
			return number == x.getRealDouble();
		}
		throw new IllegalArgumentException(x.toString());
	}
	
	
	public boolean isLessThan(LispReal x) {
		return number < x.getRealDouble();
	}
	
	
	public boolean isMoreThan(LispReal x) {
		return number > x.getRealDouble();
	}

	public BigInteger bigIntegerValue() {
		BigDecimal dec = new BigDecimal(number);
		return dec.toBigInteger();
	}

	public double doubleValue() {
		return number;
	}
	
	
	public int signum() {
		return (number > 0) ? 1 : ((number < 0) ? -1 : 0);
	}
	
	
	public LispNumber toExact() {
		return LispDouble.toExact(number);
	}
	
	
	public LispNumber toInexact() {
		return this;
	}
	
	
	public boolean equals(Object x) {
		if(x instanceof LispDouble) {
			return number == ((LispDouble)x).number;
		}
		return false;
	}
	
	
	public int hashCode() {
		int l = 17;
		long lr = Double.doubleToLongBits(number);
		
		l = 37 * l + (int)(lr ^ (lr >>> 32));
		return l;
	}
	
	
	public String toString() {
		return Double.toString(number);
	}
	
	
	public String print() {
		return disp(number);
	}

	
	public String getResult() {
		return disp(number);
	}

	@Override
	public boolean isInteger() {
		return IntLispUtils.toIntegerExact(number) != null;
	}

	@Override
	public boolean isRational() {
		return !(Double.isInfinite(number) || Double.isNaN(number));
	}

	@Override
	public boolean isReal() {
		return true;
	}
	
	
	public boolean isExact() {
		return false;
	}
	
	
	public LispString toLispString(int radix) {
		if(radix < 2 || radix > 36) {
			throw new IndexOutOfBoundsException("radix is out of range");
		} else if(radix != 10) {
			throw new IllegalArgumentException(
					"radix except 10 is not supported");
		}
		
		if(Double.isNaN(number)) {
			return new LispString("+nan.0");
		} else if(number == Double.POSITIVE_INFINITY) {
			return new LispString("+inf.0");
		} else if(number == Double.NEGATIVE_INFINITY) {
			return new LispString("-inf.0");
		} else {
			return new LispString(Double.toString(number));
		}
	}
	
	
	public boolean isNaN() {
		return Double.isNaN(number);
	}
	
	
	public boolean isOne() {
		return number == 1.0;
	}


	@Override
	public BigInteger getBigInteger() {
		return getBigDecimal().toBigInteger();
	}


	@Override
	public int getInt() {
		return getBigInteger().intValue();
	}


	@Override
	public long getLong() {
		return getBigInteger().longValue();
	}
	
	
	public BigDecimal getBigDecimal() {
		if(Double.isInfinite(number) || Double.isNaN(number)) {
			throw new NumberFormatException(
					"Infinities or NaNs is not supported");
		}
		
		return BigDecimal.valueOf(number);
	}


	@Override
	public double getRealDouble() {
		return number;
	}
	
	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getType()
	 */
	@Override
	public LispType getType() {
		return LispType.REAL;
	}
	
}
