/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: ConsoleView.cpp,v 1.4 2004/04/21 10:49:30 orrisroot Exp $ */

#include "StdAfx.h"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "WinShell.h"

#include "ConsoleDoc.h"
#include "ConsoleView.h"
#include "shell/satellite4.h"
#include "win32_console.h"

#include <process.h>
#include <commctrl.h>   // includes the common control header
#include <direct.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define TTY_MUTEX_NAME "consoleview_tty_mutex"

/* shell main routine */
DWORD __cdecl Satellite4Shell( CConsoleView* pConsole )
{
  win32_console console;
  sl_shell      shell(&console);

  console.SetConsoleView(pConsole);

  if(shell.initialize(NULL,NULL)) {
    // main loop
    shell.shell_main();
  }
  pConsole->PostMessage(ID_TTY_EXIT, 0, 0L);
  return 0;
}

/////////////////////////////////////////////////////////////////////////////
// CConsoleView

IMPLEMENT_DYNCREATE(CConsoleView, CScrollView)

BEGIN_MESSAGE_MAP(CConsoleView, CScrollView)
	//{{AFX_MSG_MAP(CConsoleView)
	ON_WM_CREATE()
	ON_WM_CHAR()
	ON_WM_VSCROLL()
	ON_WM_KEYDOWN()
	ON_WM_SIZE()
	ON_WM_MOUSEMOVE()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
    ON_WM_MOUSEWHEEL()
    ON_WM_SETFOCUS()
    ON_WM_KILLFOCUS()
	ON_COMMAND(ID_EDIT_PASTE, OnEditPaste)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PASTE, OnUpdateEditPaste)
	ON_COMMAND(ID_APP_EXIT, OnAppExit)
	//}}AFX_MSG_MAP
	// WR}h
	ON_COMMAND(ID_FILE_PRINT, CScrollView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_DIRECT, CScrollView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CScrollView::OnFilePrintPreview)
  	ON_MESSAGE(ID_TTY_TEXT_PRINT, OnTtyTextPrint)
	ON_MESSAGE(ID_TTY_CURSOR, OnTtyCursor)
	ON_MESSAGE(ID_TTY_EXIT, OnTtyExit)
    ON_MESSAGE(ID_TTY_TEXT_MODE, OnTtyTextMode)
	ON_UPDATE_COMMAND_UI(ID_INDICATOR_LINE, OnUpdateLineNumber)

    ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
    ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateEditCopy)
    ON_WM_SETCURSOR()
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CConsoleView NX̍\z/

CConsoleView::CConsoleView()
{
  m_IsAutoNL  = true;
  m_HasCaret  = false;
  m_ShowCaret = false;
  m_xCurPos = 0;
  m_yCurPos = 0;
  m_yHiden  = 0;
  m_font = NULL;
  m_attr = TEXT_ATTRIBUTE_NORMAL;
  m_textColor = RGB(0,0,0);
  m_backColor = RGB(255,255,255);
  m_textColor_sel = SELECTED_FGCOLOR;
  m_backColor_sel = SELECTED_BGCOLOR;

  // properties for scroll bar;
  m_VscrollPos    = 0;
  m_yHiden        = 0;

  // properties for copy and paste
  m_bHasCopyArea = false;
  m_bSelCopyArea = false;
  m_xsCopyArea = 0;
  m_xeCopyArea = 0;
  m_ysCopyArea = 0;
  m_yeCopyArea = 0;

  // properties for status bar
  m_nLine = 1;

  // SATELLITE shell handling properties
  m_hShellThread = INVALID_HANDLE_VALUE; /* shell thread handle */
  m_hStdinW      = INVALID_HANDLE_VALUE; /* TTY input handle */

  // TTY mutex
  hTTYMutex = CreateMutex(NULL, FALSE, TTY_MUTEX_NAME);

  // TTY input handle
  m_hStdinW = INVALID_HANDLE_VALUE;
}

CConsoleView::~CConsoleView()
{
  if(m_hShellThread != INVALID_HANDLE_VALUE){
    if(m_hStdinW != INVALID_HANDLE_VALUE){
      CloseHandle(m_hStdinW);
    }
    WaitForSingleObject(m_hShellThread, INFINITE);
    CloseHandle(m_hShellThread);
  }
  CloseHandle(hTTYMutex);
  if(m_font != NULL){
    m_font->DeleteObject();
    delete m_font;
  }
}

/////////////////////////////////////////////////////////////////////////////
// CConsoleView NX̕`

void CConsoleView::OnDraw(CDC* pDC)
{
  int x,y;
  int asave,xsave;
  const char *ptr;
  size_t len;
  CString str;
  CConsoleDoc* pDoc;
  pDoc = GetDocument();
  for(y=0; y<m_yMax; y++){
    len   = 1;
    xsave =  0;
    ptr   = pDoc->GetTextBuffer(xsave,y+m_VscrollPos);
    ASSERT(ptr != NULL);
    asave = pDoc->GetTextAttribute(xsave,y+m_VscrollPos);
    for(x=1; x<m_xMax; x++){
      if(asave == pDoc->GetTextAttribute(x,y+m_VscrollPos)){
        len++;
      }else{
        DrawTextToConsole(xsave, y, asave, ptr, len);
        len   = 1;
        xsave = x;
        ptr   = pDoc->GetTextBuffer(xsave,y+m_VscrollPos);
        ASSERT(ptr != NULL);
        asave = pDoc->GetTextAttribute(xsave,y+m_VscrollPos);
      }
    }
    DrawTextToConsole(xsave, y, asave, ptr, len);
  }
}

void CConsoleView::GetWindowProperties(){
  static const int padding = 2;
  BOOL horz, vart;
  int xtmp;
  int xclient,yclient;
  CRect rcClient;
  TEXTMETRIC tm;
  CFont* pOldFont;
  CDC* pdc;

  /* get font size */
  pdc = GetDC();
  pOldFont = pdc->SelectObject(m_font);
  pdc->GetTextMetrics( &tm );
  pdc->SelectObject(pOldFont);
  ReleaseDC(pdc);
  m_yCharSize = tm.tmHeight;
  m_xCharSize = tm.tmAveCharWidth;

  /* get window size */
  GetClientRect(rcClient);
  xclient = rcClient.Width();
  yclient = rcClient.Height();
  xtmp = xclient;
#ifdef _VC6
  CheckScrollBars(&horz,&vart);
#else
  CheckScrollBars(horz,vart);
#endif
  if(!vart) /* console window has no scrollbar */
    xtmp -= GetSystemMetrics(SM_CXVSCROLL);
  m_xMax = (xtmp - padding) / m_xCharSize;
  m_yMax = (yclient - padding) / m_yCharSize;
}

void CConsoleView::SetCaretVisible(bool flag){
  if(!m_IsAutoNL && m_HasCaret){
    if(flag){
      POINT pos;
      pos.x = m_xCurPos;
      pos.y = m_yCurPos - m_VscrollPos;
      if(pos.x < m_xMax && pos.y < m_yMax){
        ShowCaret();
        pos.x = m_xCurPos * m_xCharSize;;
        pos.y = (m_yCurPos - m_VscrollPos) * m_yCharSize;
        SetCaretPos( pos );
        m_ShowCaret = true;
      }
    }else{
      if(m_ShowCaret){
        HideCaret();
        m_ShowCaret = false;
      }
    }
  }
}


void CConsoleView::SetConsoleScrollPos(bool do_update){
  POINT pt;
  pt.x = 0;
  pt.y = m_yCharSize * m_VscrollPos;
  ScrollToPosition(pt);
  if(do_update){
    Invalidate();
    UpdateWindow();
    SetCaretVisible(false);
    SetCaretVisible(true); 
  }
}

void CConsoleView::OnInitialUpdate()
{
  int rtval;
  DWORD  dwShellThreadID;
  CConsoleDoc* pDoc;
  CScrollView::OnInitialUpdate();

  // get window properties
  GetWindowProperties();

  // initialize text buffers
  pDoc = GetDocument();
  VERIFY(pDoc->InitializeTextBuffer(m_xMax, MAX_CONSOLE_LINE));

  m_yHiden     = 0;
  m_VscrollPos = 0;
  SetScrollSizes(MM_TEXT, CSize(m_xCharSize * m_xMax, 
                                m_yCharSize * (m_yMax + m_yHiden)));

  // create satellite shell
  m_hShellThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)Satellite4Shell,
                                this, 0, &dwShellThreadID);
  rtval = GetThreadPriority( m_hShellThread );
  if(rtval == THREAD_PRIORITY_ERROR_RETURN) {
    AfxMessageBox("GetThreadPriority error.",MB_OK,NULL);
    SendMessage(WM_CLOSE);
  }
  if(!SetThreadPriority(m_hShellThread, THREAD_PRIORITY_BELOW_NORMAL)) {
    AfxMessageBox("SetThreadPriority error.",MB_OK,NULL);
    SendMessage(WM_CLOSE);
  }

  // Set title
  char   *title;
  size_t  tlen,i;
  tlen = strlen(PACKAGE_STRING);
  title = (char*)malloc(sizeof(char) * (tlen + 1));
  if(title){
    for(i=0; i<tlen; i++)
      title[i]=toupper(PACKAGE_STRING[i]);
    title[tlen] = '\0';
	this->GetParent()->SendMessage(WM_SETTEXT,0,(LPARAM)title);
    free(title);
  }
	// SC_CLOSE{^֎~
	CMenu*	pMenu;
	pMenu = this->GetParent()->GetSystemMenu( NULL );
	pMenu->EnableMenuItem( SC_CLOSE, MF_DISABLED );
}

/////////////////////////////////////////////////////////////////////////////
// CConsoleView NẌ

BOOL CConsoleView::OnPreparePrinting(CPrintInfo* pInfo)
{
	// ftHg̈
	return DoPreparePrinting(pInfo);
}

void CConsoleView::OnBeginPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: O̓ʂȏǉĂB
}

void CConsoleView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: ̌㏈ǉĂB
}

/////////////////////////////////////////////////////////////////////////////
// CConsoleView NX̐ff

#ifdef _DEBUG
void CConsoleView::AssertValid() const
{
	CScrollView::AssertValid();
}

void CConsoleView::Dump(CDumpContext& dc) const
{
	CScrollView::Dump(dc);
}

CConsoleDoc* CConsoleView::GetDocument() // fobO o[W̓CCłB
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CConsoleDoc)));
	return (CConsoleDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CConsoleView NX̃bZ[W nh

int CConsoleView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CScrollView::OnCreate(lpCreateStruct) == -1)
		return -1;
	
	// TODO: ̈ʒuɌŗL̍쐬pR[hǉĂ
	// set font
    m_font = new CFont;
    m_font->CreateFont(
    14,                        // nHeight
    0,                         // nWidth
    0,                         // nEscapement
    0,                         // nOrientation
    FW_NORMAL,                 // nWeight
    FALSE,                     // bItalic
    FALSE,                     // bUnderline
    0,                         // cStrikeOut
    SHIFTJIS_CHARSET,          // nCharSet
    OUT_CHARACTER_PRECIS,      // nOutPrecision
    CLIP_DEFAULT_PRECIS,       // nClipPrecision
    DEFAULT_QUALITY,           // nQuality
    FIXED_PITCH | FF_MODERN ,  // nPitchAndFamily
    "lr@o");           // lpszFacename
	SetFont( m_font );

	// Change icon
	HICON hIcon = AfxGetApp()->LoadIcon(IDI_TTY);
	ASSERT(hIcon);
	this->GetParent()->SendMessage(WM_SETICON, TRUE, (LPARAM)hIcon);
	
	return 0;
}

void CConsoleView::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
  /* Control Key and normal keys */
  unsigned int i;
  int keycode;
  DWORD dwLen;
  keycode = (int)nChar;
  /* binary input mode hook */
  // if(m_bInputMode && keycode == 0x0d) return;
  switch(keycode){
  case 0x03: /* control C */
    setsigint_win32();
    break;
  case 0x20: /* control space hook */
    if(GetKeyState(VK_CONTROL) & 0xFF00) keycode = 0x00;
    break;
  }
  /* restore scroll bar position */
  if(m_VscrollPos != m_yHiden){
    m_VscrollPos = m_yHiden;
    ScrollToPosition(CPoint(0,m_yCharSize * m_VscrollPos));
    UpdateWindow();
  }
  for(i=0; i<nRepCnt; i++){
    WriteFile(m_hStdinW, &keycode, sizeof(int), &dwLen, NULL);
  }
  // CScrollView::OnChar(nChar, nRepCnt, nFlags);
}

void CConsoleView::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  int vscroll_prev;
  vscroll_prev = m_VscrollPos;
  switch( nSBCode ) {
  case SB_TOP:           m_VscrollPos = 0;                    break;
  case SB_BOTTOM:        m_VscrollPos = m_yHiden;             break;
  case SB_LINEUP:        m_VscrollPos--;                      break;
  case SB_LINEDOWN:      m_VscrollPos++;                      break;
  case SB_PAGEUP:        m_VscrollPos -= m_yMax;              break;
  case SB_PAGEDOWN:      m_VscrollPos += m_yMax;              break;
  case SB_THUMBTRACK:
  case SB_THUMBPOSITION:
    if(nPos != 0) nPos += m_yCharSize;
    m_VscrollPos = nPos/m_yCharSize;
    break;
  }
  if(vscroll_prev != m_VscrollPos){
    if(m_VscrollPos < 0)
      m_VscrollPos=0;
    if(m_VscrollPos > m_yHiden)
      m_VscrollPos = m_yHiden;
    SetConsoleScrollPos(true);
  }
}

void CConsoleView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
  /* extended key */
  bool do_input;
  int keycode;
  DWORD dwLen;
  do_input = false;
  switch(nChar){
  case VK_DELETE: keycode = 0x7f;   do_input = true; break;
  case VK_F1:     keycode = SL_TTY::F1KEY    << 8; do_input = true; break;
  case VK_F2:     keycode = SL_TTY::F2KEY    << 8; do_input = true; break;
  case VK_F3:     keycode = SL_TTY::F3KEY    << 8; do_input = true; break;
  case VK_F4:     keycode = SL_TTY::F4KEY    << 8; do_input = true; break;
  case VK_F5:     keycode = SL_TTY::F5KEY    << 8; do_input = true; break;
  case VK_F6:     keycode = SL_TTY::F6KEY    << 8; do_input = true; break;
  case VK_F7:     keycode = SL_TTY::F7KEY    << 8; do_input = true; break;
  case VK_F8:     keycode = SL_TTY::F8KEY    << 8; do_input = true; break;
  case VK_F9:     keycode = SL_TTY::F9KEY    << 8; do_input = true; break;
  case VK_F10:    keycode = SL_TTY::F10KEY   << 8; do_input = true; break;
  case VK_F11:    keycode = SL_TTY::F11KEY   << 8; do_input = true; break;
  case VK_F12:    keycode = SL_TTY::F12KEY   << 8; do_input = true; break;
  case VK_HOME:   keycode = SL_TTY::HOMEKEY  << 8; do_input = true; break;
  case VK_END:    keycode = SL_TTY::ENDKEY   << 8; do_input = true; break;
  case VK_NEXT:   keycode = SL_TTY::PGUPKEY  << 8; do_input = true; break;
  case VK_PRIOR:  keycode = SL_TTY::PGDNKEY  << 8; do_input = true; break;
  case VK_UP:     keycode = SL_TTY::UPKEY    << 8; do_input = true; break;
  case VK_DOWN:   keycode = SL_TTY::DOWNKEY  << 8; do_input = true; break;
  case VK_RIGHT:  keycode = SL_TTY::RIGHTKEY << 8; do_input = true; break;
  case VK_LEFT:   keycode = SL_TTY::LEFTKEY  << 8; do_input = true; break;
  case VK_SCROLL: keycode = SL_TTY::SCRLKKEY << 8; do_input = true; break;
  }
  if(do_input){
    unsigned int i;
    /* restore scroll bar position */
    if(m_VscrollPos != m_yHiden){
      m_VscrollPos = m_yHiden;
      SetConsoleScrollPos(true);
    }
    for(i=0; i<nRepCnt; i++){
      WriteFile(m_hStdinW, &keycode, sizeof(int), &dwLen, NULL);
    }
    return;
  }

  CScrollView::OnKeyDown(nChar, nRepCnt, nFlags);
}


void CConsoleView::OnSize(UINT nType, int cx, int cy) 
{
  CScrollView::OnSize(nType, cx, cy);

  if(IsWindowVisible()){
    CConsoleDoc *pDoc;
    int i,fix;
    int xmax_pre, ymax_pre;
    int yhiden_pre;

    pDoc = GetDocument();
    xmax_pre = m_xMax;
    ymax_pre = m_yMax;
    yhiden_pre = m_yHiden;
    GetWindowProperties();

    if(xmax_pre == m_xMax && ymax_pre == m_yMax) return;

    if(ymax_pre < m_yMax){
      if(m_yMax + m_yHiden > MAX_CONSOLE_LINE){
        m_yHiden = MAX_CONSOLE_LINE - m_yMax;
        fix = yhiden_pre - m_yHiden;
        m_yCurPos-=fix;
        for(i=0; i<fix; i++){
          VERIFY(pDoc->ScrollTextBuffer(true));
        }
        m_VscrollPos-=fix;
      }else{

      }
    }
    if(ymax_pre > m_yMax){
      int diff,less;
      diff = ymax_pre - m_yMax;
      less = diff;
      fix = ymax_pre - m_yHiden - m_yCurPos - 1;
      /* , ɋ󔒍sA΂̕ */
      if(fix - diff >= 0){
        /* resize  */
        less = 0;
      }else{
        /* ̗ʂłȂ΂̕邱ƂOƂĂ */
        less -= fix;
      }
      /* Ȃ, B̈ɔCĂ݂ */
      fix = MAX_CONSOLE_LINE - ymax_pre - m_yHiden;
      /* B̈ɏ\]TȂ, ꂾ */
      if(fix >= less){
        m_yHiden += less;
        m_VscrollPos += less;
        less =0;
      }else{
        /* Ȃ 镪܂킷 */
        m_yHiden += fix;
        m_VscrollPos += fix;
        less -= fix;
      }
      /* łȂ XN[ */
      for(i=0;i < less; i++){
        VERIFY(pDoc->ScrollTextBuffer(true));
      }
      /* J[\̃|WV̓XN[ */
      m_yCurPos -= less;
    }

    /* cursor position fixing */
    if(m_xMax <= m_xCurPos) m_xCurPos = m_xMax - 1;
    if(m_yCurPos < 0) m_yCurPos = 0;
    if(m_yCurPos >= m_yHiden + m_yMax) m_yCurPos = m_yHiden + m_yMax - 1;

    /* resize text buffers */
    VERIFY(pDoc->ResizeTextBuffer(m_xMax, m_yMax + m_yHiden));

    /* window size fixing */
    SetScrollSizes(MM_TEXT, CSize(m_xCharSize * m_xMax,
                                  m_yCharSize * (m_yMax + m_yHiden)));
    SetConsoleScrollPos(true);
  }
}

void CConsoleView::FixedSelectedCopyArea()
{
  int tmp;
  if(m_xsCopyArea < 0) m_xsCopyArea = 0;
  if(m_ysCopyArea < 0) m_ysCopyArea = 0;
  if(m_xeCopyArea < 0) m_xeCopyArea = 0;
  if(m_yeCopyArea < 0) m_yeCopyArea = 0;

  /* swap y pos and x pos */
  if(m_ysCopyArea > m_yeCopyArea){
    tmp = m_yeCopyArea; m_yeCopyArea = m_ysCopyArea; m_ysCopyArea = tmp;
    tmp = m_xeCopyArea; m_xeCopyArea = m_xsCopyArea; m_xsCopyArea = tmp;
  }else if(m_ysCopyArea == m_yeCopyArea && m_xsCopyArea > m_xeCopyArea){
    tmp = m_xeCopyArea; m_xeCopyArea = m_xsCopyArea; m_xsCopyArea = tmp;
  }

  /* check buffer size */
  if(m_yeCopyArea >= m_yMax + m_yHiden) m_yeCopyArea = m_yMax + m_yHiden - 1;
  if(m_xeCopyArea > m_xMax)             m_xeCopyArea = m_xMax;
}

void CConsoleView::OnMouseMove(UINT nFlags, CPoint point) 
{
  if(m_bSelCopyArea){
    m_xeCopyArea = point.x / m_xCharSize;
    m_yeCopyArea = point.y / m_yCharSize + m_VscrollPos;
    if(m_xeCopyArea < 0) m_xeCopyArea = 0;
    if(m_yeCopyArea < 0) m_yeCopyArea = 0;
    GetDocument()->ClearSelectedBuffer();
    VERIFY(GetDocument()->SelectTextBuffer(m_xsCopyArea, m_ysCopyArea, 
                                           m_xeCopyArea, m_yeCopyArea));
    Invalidate(FALSE);
  }
  CScrollView::OnMouseMove(nFlags, point);
}

void CConsoleView::OnLButtonDown(UINT nFlags, CPoint point) 
{
  // if already selected copy area then clear selected flag
  m_bHasCopyArea = false;
  GetDocument()->ClearSelectedBuffer();
  Invalidate(TRUE);
  UpdateWindow();
  m_xsCopyArea = point.x / m_xCharSize;
  m_ysCopyArea = point.y / m_yCharSize + m_VscrollPos;
  if(m_xsCopyArea < 0) m_xsCopyArea = 0;
  if(m_ysCopyArea < 0) m_ysCopyArea = 0;
  m_xeCopyArea = 0;
  m_yeCopyArea = 0;
  m_bSelCopyArea = true;
  SetCapture();
  // CScrollView::OnLButtonDown(nFlags, point);
}

void CConsoleView::OnLButtonUp(UINT nFlags, CPoint point) 
{
  if(m_bSelCopyArea){
    m_xeCopyArea = point.x / m_xCharSize;
    m_yeCopyArea = point.y / m_yCharSize + m_VscrollPos;
    if(m_xeCopyArea < 0) m_xeCopyArea = 0;
    if(m_yeCopyArea < 0) m_yeCopyArea = 0;
    ReleaseCapture();
    Invalidate(FALSE);
    if(!(m_xsCopyArea == m_xeCopyArea && m_ysCopyArea == m_yeCopyArea)){
      VERIFY(GetDocument()->SelectTextBuffer(m_xsCopyArea, m_ysCopyArea, 
                                             m_xeCopyArea, m_yeCopyArea));
      m_bHasCopyArea = true;
      AfxGetMainWnd()->GetMenu()->EnableMenuItem(ID_EDIT_COPY, MF_ENABLED | MF_BYCOMMAND);
    }
    m_bSelCopyArea = false;
    m_xsCopyArea = 0;
    m_xeCopyArea = 0;
    m_ysCopyArea = 0;
    m_yeCopyArea = 0;
  }
  // CScrollView::OnLButtonUp(nFlags, point);
}


//	Draw text to console view window
void CConsoleView::DrawTextToConsole(int xpos, int ypos, int attr, const char *str, size_t len)
{
  CDC *dc;
  CFont* pOldFont;
  const static char *dummy=" ";
  int x,y;
  size_t i,space(0);
  for(i=0;i<len;i++){
    if(str[i] == '\n'){
      space = len - i; len = i; break;
    }
  }
  dc = GetDC();
  pOldFont = dc->SelectObject( m_font );
  if(attr & TEXT_ATTRIBUTE_SELECT){
    dc->SetTextColor(m_textColor_sel);
    dc->SetBkColor(m_backColor_sel);
  }else{
    if(attr & TEXT_ATTRIBUTE_REVERSE){
      dc->SetTextColor(m_backColor);
      dc->SetBkColor(m_textColor);
    }else{
      dc->SetTextColor(m_textColor);
      dc->SetBkColor(m_backColor);
    }
  }
  x = m_xCharSize * xpos;
  y = m_yCharSize * ypos;
  if(len != 0)dc->TextOut(x, y, str, (int)len);
  for(i=0;i<space;i++){
    dc->TextOut(x + (len + i) * m_xCharSize, y, dummy, 1);
  }
  dc->SelectObject(pOldFont);
  ReleaseDC( dc );
}

///////////////////////////////////////////////////////
//
//		ShellC^[tF[X
//
//
void CConsoleView::PrintText(const char *linebuf, size_t len)
{
  bool do_nextline;
  size_t i,j;
  size_t lest,oneline;
  CConsoleDoc* pDoc = GetDocument();

  SetCaretVisible(false);

  i=0;
  while(i<len){
    do_nextline = false;
    oneline = m_xMax - m_xCurPos; /* printable online char */
    lest    = len - i;               /* lest char length */
    lest    = min(lest, oneline);    /* trim lest char to oneline */
    for(j=0;j<lest;j++){             /* check \n char */
      if(linebuf[i+j] == '\n'){
        do_nextline = true;
        lest = j;
      }
    }
    if(lest > 0){
      /* do print string */
      DrawTextToConsole(m_xCurPos, m_yCurPos - m_yHiden, m_attr, &linebuf[i], lest);
      /* do save text buffer */
      if(do_nextline){
        VERIFY(pDoc->SaveTextBuffer(m_xCurPos, m_yCurPos, m_attr, &linebuf[i], lest+1));
      }else{
        VERIFY(pDoc->SaveTextBuffer(m_xCurPos, m_yCurPos, m_attr, &linebuf[i], lest));
      }
    }else{
      if(!do_nextline) break;
    }
    m_xCurPos += (int)lest;
    if(m_IsAutoNL && oneline == lest) /* check end of line */
      do_nextline = true;
    if(do_nextline){
       m_xCurPos = 0; m_yCurPos++;   /* increment cursor y position */
      if(m_yCurPos - m_yHiden == m_yMax){
        m_yHiden++;
        if(m_yMax + m_yHiden > MAX_CONSOLE_LINE){
          m_yCurPos--; m_yHiden--;
          VERIFY(pDoc->ScrollTextBuffer(true));
          Invalidate(FALSE);
        }else{
          VERIFY(pDoc->ResizeTextBuffer(m_xMax, m_yMax + m_yHiden));
          m_VscrollPos = m_yHiden;
          SetScrollSizes(MM_TEXT, CSize(m_xCharSize * m_xMax,
                                        m_yCharSize * (m_yMax + m_yHiden)));
          SetConsoleScrollPos(true);
        }
        UpdateWindow();
      }
    }
    i += j; /* counter increment to next char */
  }

  SetCaretVisible(true);
}

void CConsoleView::SendTextPrint(const char *buf, size_t len){
  HANDLE  hMutex;
  hMutex = OpenMutex(MUTEX_ALL_ACCESS, FALSE, TTY_MUTEX_NAME);
  WaitForSingleObject(hMutex, INFINITE);
  SendMessage(ID_TTY_TEXT_PRINT,(WPARAM)buf,len);
  ReleaseMutex(hMutex);
  CloseHandle(hMutex);
}

void CConsoleView::SendTextColor(int target, unsigned int color){
  int mode(-1);
  switch(target){
    case 0: /* foreground */
      mode = SENDMESS_TEXTMODE_FGCOLOR;
      break;
    case 1: /* background */
      mode = SENDMESS_TEXTMODE_BGCOLOR;
      break;
  }
  if(mode!=-1){
    SendMessage(ID_TTY_TEXT_MODE, mode, color);
  }
}

void CConsoleView::SendTextAttribute(int attr){
  static const int mode = SENDMESS_TEXTMODE_ATTRIBUTE;
  if(attr == TEXT_ATTRIBUTE_NORMAL ||
     attr == TEXT_ATTRIBUTE_REVERSE){
     SendMessage(ID_TTY_TEXT_MODE, mode, attr);
  }
}

void CConsoleView::SendTextEditMode(int flag){
  SendMessage(ID_TTY_TEXT_MODE, SENDMESS_TEXTMODE_EDIT, flag);
}

void CConsoleView::SendCursorGetMax(int *x, int *y){
  DWORD pos;
  pos = (unsigned int)SendMessage(ID_TTY_CURSOR, SENDMESS_CURSOR_GETMAX, 0L);
  *x = (int)HIWORD(pos);
  *y = (int)LOWORD(pos);
}

//
LRESULT CConsoleView::OnTtyTextPrint(WPARAM wParam, LPARAM lParam)
{
  const char *buf;
  size_t len;
  buf = (const char*)wParam;
  len = (size_t)lParam;
  PrintText(buf,len);
  return 0L;
}


LRESULT CConsoleView::OnTtyCursor(WPARAM wParam, LPARAM lParam)
{
  CConsoleDoc *pDoc;
  int mode;
  DWORD pos;
  WORD x, y;
  mode = (int)wParam;
  switch(mode){
  case SENDMESS_CURSOR_GETMAX:
    y = (WORD)m_yMax;
    x = (WORD)m_xMax;
    pos = (x << 16) + y;
    return pos;
  case SENDMESS_CURSOR_DOWN:
    m_yCurPos += (int)lParam;
    if(m_yCurPos - m_yHiden >= m_yMax) m_yCurPos = m_yHiden + m_yMax - 1;
    break;
  case SENDMESS_CURSOR_UP:
    m_yCurPos -= (int)lParam;
    if(m_yCurPos < m_yHiden) m_yCurPos = m_yHiden;
    break;
  case SENDMESS_CURSOR_RIGHT:
    m_xCurPos += (int)lParam;
    if(m_xCurPos > m_xMax) m_xCurPos = m_xMax;
    break;
  case SENDMESS_CURSOR_LEFT:
    m_xCurPos -= (int)lParam;
    if(m_xCurPos < 0) m_xCurPos = 0;
    break;
  case SENDMESS_CURSOR_BOL:
    m_xCurPos = 0;
    break;
  case SENDMESS_CURSOR_NEWLINE:
    pDoc = GetDocument();
    m_xCurPos = 0; m_yCurPos++;   /* increment cursor y position */
    if(m_yCurPos - m_yHiden == m_yMax){
      m_yHiden++;
      if(m_yMax + m_yHiden > MAX_CONSOLE_LINE){
        m_yCurPos--; m_yHiden--;
        VERIFY(pDoc->ScrollTextBuffer(true));
          Invalidate(FALSE);
          UpdateWindow();
        }else{
          VERIFY(pDoc->ResizeTextBuffer(m_xMax, m_yMax + m_yHiden));
          m_VscrollPos = m_yHiden;
          SetScrollSizes(MM_TEXT, CSize(m_xCharSize * m_xMax,
                                        m_yCharSize * (m_yMax + m_yHiden)));
          SetConsoleScrollPos(true);
        }
      }
    break;
  }
  SetCaretVisible(false); SetCaretVisible(true); // reprint caret
  return 0L;
}

//
// shellC^[tF[XI
//
/////////////////////////////////////////////////////////////////////
void CConsoleView::PasteClipBoardToTextBuffer()
{
  int	ch;
  DWORD size;
  size_t i,len;
  const char *str;
  COleDataObject clipboardData;
  HGLOBAL hText;
  
  clipboardData.AttachClipboard();
  if (!clipboardData.IsDataAvailable(CF_TEXT)) {
    ::MessageBeep(MB_ICONEXCLAMATION);
    return;
  }

  hText = clipboardData.GetGlobalData(CF_TEXT);
  if(hText == NULL) return;
  str = (const char *)::GlobalLock(hText);
  len = strlen(str);
  for(i=0; i<len; i++){
    ch = (unsigned char)str[i];
    WriteFile(m_hStdinW, &ch, sizeof(int), &size, NULL);
  }

  // release resource data
  ::GlobalUnlock(hText);
  ::GlobalFree(hText);
}


void CConsoleView::OnEditPaste() 
{
  PasteClipBoardToTextBuffer();
}

void CConsoleView::OnUpdateEditPaste(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable (::IsClipboardFormatAvailable (CF_TEXT));
}


LRESULT CConsoleView::OnTtyTextMode(WPARAM wParam, LPARAM lParam)
{
  int mode;
  mode = (int)wParam;
  if(mode == SENDMESS_TEXTMODE_FGCOLOR ||
    mode == SENDMESS_TEXTMODE_BGCOLOR){
    unsigned int color,red,green,blue;
    color = (unsigned int)lParam;
    red   = (0xff0000 & color) >> 16;
    green = (0x00ff00 & color) >> 8;
    blue  = (0x0000ff & color);
    switch(mode){
    case SENDMESS_TEXTMODE_FGCOLOR: /* foreground */
      m_textColor = RGB(red,green,blue);
      break;
    case SENDMESS_TEXTMODE_BGCOLOR: /* background */
      m_backColor = RGB(red,green,blue);
      break;
    }
  }else if(mode == SENDMESS_TEXTMODE_ATTRIBUTE){
    int attr;
    attr = (int)lParam;
    if(attr == TEXT_ATTRIBUTE_NORMAL)
      m_attr = attr;
    else
      m_attr |= attr;
  }else if(mode == SENDMESS_TEXTMODE_EDIT){
    int editmode;
    editmode = (int)lParam;
    if(editmode == 0){
      SetCaretVisible(false);
      m_IsAutoNL = true;
    }else{
      m_IsAutoNL = false;
      SetCaretVisible(true);
    }
  }
  return 0L;
}

LRESULT CConsoleView::OnTtyExit(WPARAM wParam, LPARAM lParam)
{
  CWnd * pWnd;
  pWnd = AfxGetApp()->m_pMainWnd;
  pWnd->PostMessage(WM_CLOSE);
  return 0L;
}

void CConsoleView::OnUpdateLineNumber(CCmdUI* pCmdUI)
{
  CString string;
  string.Format (_T ("Line %d"), m_yCurPos + 1);
  pCmdUI->Enable (TRUE);
  pCmdUI->SetText (string);
}

void CConsoleView::OnAppExit() 
{
  if(m_hShellThread != INVALID_HANDLE_VALUE){
    setsigint_win32();
    CloseHandle(m_hStdinW);
    m_hStdinW = INVALID_HANDLE_VALUE;
  }
}

BOOL CConsoleView::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt)
{
  static const int nline = 3;
  static short total = 0;
  short vsize;
  BOOL horz,vart;
#ifdef _VC6
  CheckScrollBars(&horz,&vart);
#else
  CheckScrollBars(horz,vart);
#endif
  if(!vart) /* console window has not scroll bar */
    return FALSE;
  total += zDelta;
  vsize = total / WHEEL_DELTA;
  total = total % WHEEL_DELTA;
  if(vsize == 0) return FALSE;
   m_VscrollPos -= nline * vsize;
  if(vsize < 0){
    if(m_VscrollPos > m_yHiden) m_VscrollPos = m_yHiden;
  }else{
    if(m_VscrollPos < 0) m_VscrollPos = 0;
  }
  SetConsoleScrollPos(true);
  return TRUE;
  // return CScrollView::OnMouseWheel(nFlags, zDelta, pt);
}

void CConsoleView::OnSetFocus(CWnd* pOldWnd)
{
  CScrollView::OnSetFocus(pOldWnd);
  
  CreateSolidCaret(2,m_yCharSize);
  m_HasCaret = true;
  SetCaretVisible(true);
}

void CConsoleView::OnKillFocus(CWnd* pNewWnd)
{
  CScrollView::OnKillFocus(pNewWnd);

  SetCaretVisible(false);
  ::DestroyCaret();
  m_HasCaret = false;
}

void CConsoleView::OnEditCopy()
{
  VERIFY(GetDocument()->SetTextBufferToClipboard());
}

void CConsoleView::OnUpdateEditCopy(CCmdUI *pCmdUI)
{
  pCmdUI->Enable(m_bHasCopyArea);
}

BOOL CConsoleView::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message)
{
  LPSTR cursor_id;
  if(nHitTest == HTVSCROLL)
    cursor_id = IDC_ARROW;
  else
    cursor_id = IDC_IBEAM;

  // TODO : ɃbZ[W nh R[hǉ邩ȀĂяo܂B
  SetCursor( AfxGetApp()->LoadStandardCursor( cursor_id ));
  return TRUE;
  // return CScrollView::OnSetCursor(pWnd, nHitTest, message);
}

#ifdef _VC6
void CConsoleView::CheckScrollBars(BOOL* horz,BOOL* vert){
	int minPos, maxPos;
	this->GetScrollRange(SB_HORZ, &minPos, &maxPos);
	if(minPos == 0 && maxPos == 0)
		*horz = FALSE;
	else
		*horz = TRUE;
	this->GetScrollRange(SB_VERT, &minPos, &maxPos);
	if(minPos == 0 && maxPos == 0)
		*vert = FALSE;
	else
		*vert = TRUE;

}
#endif
