/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: ConsoleView.cpp,v 1.6 2004/08/05 05:41:36 yoshihiko Exp $ */

#include "stdafx.h"
#include "win_shell.h"
#include "shell/satellite4.h"

#include "ConsoleDoc.h"
#include "ConsoleView.h"
#include "ChildFrm.h"
#include "WinThread.h"
#include "shell/satellite4.h"
#include <process.h>
#include <commctrl.h>   // includes the common control header
#include <direct.h>
#include "config.h"
#include "MainFrm.h"

using namespace std;
#define printf          syscom->console->tty_printf

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif
#define TTY_MUTEX_NAME "lightview_tty_mutex"

//global
extern	CConsoleView* g_pConsole;
HANDLE	gCharEvent;
extern	bool interupt;
HANDLE	hThread = NULL;
extern	sl_shell* g_pShell;
extern	int		g_bChdirFlag;
//extern CMultiDocTemplate* g_pWorkDocTemplate;
//extern CWorkDoc*	g_pWorkDoc;
//extern CMultiDocTemplate* g_pDriveDocTemplate;
//extern CDriveDoc* g_pDriveDoc;
//extern CCmdDoc* g_pCmdDoc;
extern char g_charScriptText[_MAX_PATH];

#ifdef __cplusplus
extern "C" {
#endif
__declspec(dllimport) void ExtncsInit(CWinApp* pApp);
__declspec(dllimport) BOOL ExtncsExec(CWinApp* pApp, CView* pView, HANDLE hStdinW);
#ifdef __cplusplus
}
#endif

/////////////////////////////////////////////////////////////////////////////
// CConsoleView

IMPLEMENT_DYNCREATE(CConsoleView, CScrollView)

CConsoleView::CConsoleView()
{
	m_IsAutoNL  = true;
	m_HasCaret  = false;
	m_ShowCaret = false;
	m_xCurPos = 0;
	m_yCurPos = 0;
	m_yHiden  = 0;
	m_font = NULL;
	m_attr = TEXT_ATTRIBUTE_NORMAL;
	m_textColor = RGB(0,0,0);
	m_backColor = RGB(255,255,255);
	m_textColor_sel = SELECTED_FGCOLOR;
	m_backColor_sel = SELECTED_BGCOLOR;

	// properties for scroll bar;
	m_VscrollPos    = 0;
	m_yHiden        = 0;

	// properties for copy and paste
	m_bHasCopyArea = false;
	m_bSelCopyArea = false;
	m_xsCopyArea = 0;
	m_xeCopyArea = 0;
	m_ysCopyArea = 0;
	m_yeCopyArea = 0;

	// properties for status bar
	m_nLine = 1;

	// SATELLITE shell handling properties
	m_hShellThread = INVALID_HANDLE_VALUE; /* shell thread handle */
	m_hStdinW      = INVALID_HANDLE_VALUE; /* TTY input handle */

	// TTY mutex
	hTTYMutex = CreateMutex(NULL, FALSE, TTY_MUTEX_NAME);

	// TTY input handle
	m_hStdinW = INVALID_HANDLE_VALUE;
	//Input mode
	m_bInputMode = FALSE;
	//excecution state
	m_bScriptRun = FALSE;
	//support drag
	m_bDriveDrag = FALSE;
	m_pDragImage = NULL;
	//extend SIGMA
	m_bExtendSigma = FALSE;
	//OnSize
	m_OnSize = FALSE;
}

CConsoleView::~CConsoleView()
{
	if(m_hShellThread != INVALID_HANDLE_VALUE){
		if(m_hStdinW != INVALID_HANDLE_VALUE){
			CloseHandle(m_hStdinW);
		}
		WaitForSingleObject(m_hShellThread, INFINITE);
		CloseHandle(m_hShellThread);
	}
	CloseHandle(hTTYMutex);
	if(m_font != NULL){
		m_font->DeleteObject();
		delete m_font;
	}
	/*m_cmdArray and m_varArray is deleted.*/
	int nCount;
	nCount = (int)m_cmdArray.GetSize();
	if(nCount > 0) {
		for (int i=0; i<nCount; i++)
			delete (CMDINFO*) m_cmdArray.GetAt (i);
		m_cmdArray.RemoveAll();
	}
	nCount = (int)m_varArray.GetSize();
	if(nCount > 0) {
		for (int i=0; i<nCount; i++)
			delete (VARINFO*) m_varArray.GetAt (i);
		m_varArray.RemoveAll();
	}
}


BEGIN_MESSAGE_MAP(CConsoleView, CScrollView)
	//{{AFX_MSG_MAP(CConsoleView)
	ON_WM_CREATE()
	ON_WM_CHAR()
	ON_WM_VSCROLL()
	ON_WM_KEYDOWN()
	ON_WM_SIZE()
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE, OnUpdateFileSave)
	ON_UPDATE_COMMAND_UI(ID_SCRIPT_RUN, OnUpdateScriptRun)
	ON_WM_SETCURSOR()
	ON_WM_MOUSEMOVE()
	ON_COMMAND(ID_EDIT_PASTE, OnEditPaste)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PASTE, OnUpdateEditPaste)
	ON_UPDATE_COMMAND_UI(ID_CONSOLE, OnUpdateConsole)
	ON_UPDATE_COMMAND_UI(ID_FILE_CLOSE, OnUpdateFileClose)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_DESTROY()
	ON_COMMAND(ID_WIN_SIGMA, OnWinSigma)
	ON_UPDATE_COMMAND_UI(ID_WIN_SIGMA, OnUpdateWinSigma)
	ON_WM_MOUSEWHEEL()
	ON_WM_SETFOCUS()
	ON_WM_KILLFOCUS()
	ON_COMMAND(ID_APP_EXIT, OnAppExit)
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateEditCopy)
	ON_UPDATE_COMMAND_UI(ID_FILE_ALL, OnUpdateFileAll)
	ON_UPDATE_COMMAND_UI(ID_FILE_MDL, OnUpdateFileMdl)
	ON_UPDATE_COMMAND_UI(ID_FILE_SL, OnUpdateFileSl)
	ON_UPDATE_COMMAND_UI(ID_FILE_TXT, OnUpdateFileTxt)
	ON_COMMAND(ID_FILE_ALL, OnFileAll)
	ON_COMMAND(ID_FILE_MDL, OnFileMdl)
	ON_COMMAND(ID_FILE_SL, OnFileSl)
	ON_COMMAND(ID_FILE_TXT, OnFileTxt)
	ON_COMMAND(ID_SCRIPT_RUN, OnScriptRun)
	ON_COMMAND(ID_SCRIPT_STOP, OnScriptStop)
	ON_UPDATE_COMMAND_UI(ID_SCRIPT_STOP, OnUpdateScriptStop)
	//}}AFX_MSG_MAP
	ON_MESSAGE(ID_TTY_CURSOR, OnTtyCursor)
	ON_UPDATE_COMMAND_UI(ID_INDICATOR_LINE, OnUpdateLineNumber)
  	ON_MESSAGE(ID_TTY_TEXT_PRINT, OnTtyTextPrint)
	ON_MESSAGE(ID_TTY_EXIT, OnTtyExit)
    ON_MESSAGE(ID_TTY_TEXT_MODE, OnTtyTextMode)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CConsoleView `

void CConsoleView::OnInitialUpdate()
{
  int rtval;
  DWORD  dwShellThreadID;
  CConsoleDoc* pDoc;
  CScrollView::OnInitialUpdate();

  // get window properties
  GetWindowProperties();

  // initialize text buffers
  pDoc = GetDocument();
  VERIFY(pDoc->InitializeTextBuffer(m_xMax, MAX_CONSOLE_LINE));

  m_yHiden     = 0;
  m_VscrollPos = 0;
  SetScrollSizes(MM_TEXT, CSize(m_xCharSize * m_xMax, 
                                m_yCharSize * (m_yMax + m_yHiden)));

  // create satellite shell
  m_hShellThread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)Satellite4Shell,
                                this, 0, &dwShellThreadID);
  rtval = GetThreadPriority( m_hShellThread );
  if(rtval == THREAD_PRIORITY_ERROR_RETURN) {
    AfxMessageBox("GetThreadPriority error.",MB_OK,NULL);
    SendMessage(WM_CLOSE);
  }
  if(!SetThreadPriority(m_hShellThread, THREAD_PRIORITY_BELOW_NORMAL)) {
    AfxMessageBox("SetThreadPriority error.",MB_OK,NULL);
    SendMessage(WM_CLOSE);
  }
  //A g_pConsole variable is set up. 
  g_pConsole = this;

  // Set title
  char   *title;
  size_t  tlen;
  tlen = /*strlen(PACKAGE_STRING) + */strlen("ConsoleViewer");
  title = (char*)malloc(sizeof(char) * (tlen + 1));
  if(title){
    //for(i=0; i<tlen; i++)
    //  title[i]=toupper(PACKAGE_STRING[i]);
    //title[tlen] = '\0';
	strcpy(title, "ConsoleViewer");
	this->GetParent()->SendMessage(WM_SETTEXT,0,(LPARAM)title);
    free(title);
  }
	// SC_CLOSE{^֎~
	CMenu*	pMenu;
	pMenu = this->GetParent()->GetSystemMenu( NULL );
	pMenu->EnableMenuItem( SC_CLOSE, MF_DISABLED );

}

void CConsoleView::OnDraw(CDC* pDC)
{
	int x,y;
	int asave,xsave;
	const char *ptr;
	size_t len;
	CString str;
	CConsoleDoc* pDoc;
	pDoc = GetDocument();
	for(y=0; y<m_yMax; y++){
		len   = 1;
		xsave =  0;
		ptr   = pDoc->GetTextBuffer(xsave,y+m_VscrollPos);
		ASSERT(ptr != NULL);
		asave = pDoc->GetTextAttribute(xsave,y+m_VscrollPos);
		for(x=1; x<m_xMax; x++){
			if(asave == pDoc->GetTextAttribute(x,y+m_VscrollPos)){
				len++;
			}else{
				DrawTextToConsole(xsave, y, asave, ptr, len);
				len   = 1;
				xsave = x;
				ptr   = pDoc->GetTextBuffer(xsave,y+m_VscrollPos);
				ASSERT(ptr != NULL);
				asave = pDoc->GetTextAttribute(xsave,y+m_VscrollPos);
			}
		}
		DrawTextToConsole(xsave, y, asave, ptr, len);
	}
}

void CConsoleView::GetWindowProperties(){
  static const int padding = 2;
  BOOL horz, vart;
  int xtmp,ytmp;
  int xclient,yclient;
  CRect rcClient;
  TEXTMETRIC tm;
  CFont* pOldFont;
  CDC* pdc;

  /* get font size */
  pdc = GetDC();
  pOldFont = pdc->SelectObject(m_font);
  pdc->GetTextMetrics( &tm );
  pdc->SelectObject(pOldFont);
  ReleaseDC(pdc);
  m_yCharSize = tm.tmHeight;
  m_xCharSize = tm.tmAveCharWidth;

  /* get window size */
  GetClientRect(rcClient);
  xclient = rcClient.Width();
  yclient = rcClient.Height();
  xtmp = xclient;
  CheckScrollBars(&horz,&vart);
  if(!vart) /* console window has no scrollbar */
    xtmp -= GetSystemMetrics(SM_CXVSCROLL);
  else
	xtmp = xclient;
  m_xMax = (xtmp - padding) / m_xCharSize;

  ytmp = yclient;
  //if(horz) /* console window has no scrollbar */
  //  ytmp += GetSystemMetrics(SM_CXHSCROLL);
  //else
  //	ytmp = yclient;
  m_yMax = (ytmp/*yclient*/ - padding) / m_yCharSize;
if(m_yMax == 25)
  m_yMax = 25;
}

void CConsoleView::SetCaretVisible(bool flag){
  if(!m_IsAutoNL && m_HasCaret){
    if(flag){
      POINT pos;
      pos.x = m_xCurPos;
      pos.y = m_yCurPos - m_VscrollPos;
      if(pos.x < m_xMax && pos.y < m_yMax){
        ShowCaret();
        pos.x = m_xCurPos * m_xCharSize;;
        pos.y = (m_yCurPos - m_VscrollPos) * m_yCharSize;
        SetCaretPos( pos );
        m_ShowCaret = true;
      }
    }else{
      if(m_ShowCaret){
        HideCaret();
        m_ShowCaret = false;
      }
    }
  }
}

void CConsoleView::SetConsoleScrollPos(bool do_update){
  POINT pt;
  pt.x = 0;
  pt.y = m_yCharSize * m_VscrollPos;
  ScrollToPosition(pt);
  if(do_update){
    Invalidate();
    //UpdateWindow();
    SetCaretVisible(false);
    SetCaretVisible(true); 
  }
}

/////////////////////////////////////////////////////////////////////////////
// CConsoleView ff

#ifdef _DEBUG
void CConsoleView::AssertValid() const
{
	CScrollView::AssertValid();
}

void CConsoleView::Dump(CDumpContext& dc) const
{
	CScrollView::Dump(dc);
}

CConsoleDoc* CConsoleView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CConsoleDoc)));
	return (CConsoleDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CConsoleView bZ[W nh

int CConsoleView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CScrollView::OnCreate(lpCreateStruct) == -1)
		return -1;

	// set font
    m_font = new CFont;
    m_font->CreateFont(
    14,                        // nHeight
    0,                         // nWidth
    0,                         // nEscapement
    0,                         // nOrientation
    FW_NORMAL,                 // nWeight
    FALSE,                     // bItalic
    FALSE,                     // bUnderline
    0,                         // cStrikeOut
    SHIFTJIS_CHARSET,          // nCharSet
    OUT_CHARACTER_PRECIS,      // nOutPrecision
    CLIP_DEFAULT_PRECIS,       // nClipPrecision
    DEFAULT_QUALITY,           // nQuality
    FIXED_PITCH | FF_MODERN ,  // nPitchAndFamily
    "lr@o");           // lpszFacename
	SetFont( m_font );

	// Change icon
	HICON hIcon = AfxGetApp()->LoadIcon(IDI_TTY);
	ASSERT(hIcon);
	this->GetParent()->SendMessage(WM_SETICON, TRUE, (LPARAM)hIcon);

	return 0;
}

//
//ǌ^
//
void CConsoleView::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
  /* Control Key and normal keys */
  unsigned int i;
  int keycode;
  DWORD dwLen;
  keycode = (int)nChar;
  /* binary input mode hook */
  // if(m_bInputMode && keycode == 0x0d) return;
  switch(keycode){
  case 0x03: /* control C */
    setsigint_win32();
    break;
  case 0x20: /* control space hook */
    if(GetKeyState(VK_CONTROL) & 0xFF00) keycode = 0x00;
    break;
  }
  /* restore scroll bar position */
  if(m_VscrollPos != m_yHiden){
    m_VscrollPos = m_yHiden;
//if(m_VscrollPos == 278)
//	m_VscrollPos = 278;
    ScrollToPosition(CPoint(0,m_yCharSize * m_VscrollPos));
    UpdateWindow();
  }
  for(i=0; i<nRepCnt; i++){
    WriteFile(m_hStdinW, &keycode, sizeof(int), &dwLen, NULL);
  }
  // CScrollView::OnChar(nChar, nRepCnt, nFlags);

}

void CConsoleView::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  int vscroll_prev;
  vscroll_prev = m_VscrollPos;
  switch( nSBCode ) {
  case SB_TOP:           m_VscrollPos = 0;                    break;
  case SB_BOTTOM:        m_VscrollPos = m_yHiden;             break;
  case SB_LINEUP:        m_VscrollPos--;                      break;
  case SB_LINEDOWN:      m_VscrollPos++;                      break;
  case SB_PAGEUP:        m_VscrollPos -= m_yMax;              break;
  case SB_PAGEDOWN:      m_VscrollPos += m_yMax;              break;
  case SB_THUMBTRACK:
  case SB_THUMBPOSITION:
    if(nPos != 0) nPos += m_yCharSize;
    m_VscrollPos = nPos/m_yCharSize;
    break;
  }
  if(vscroll_prev != m_VscrollPos){
    if(m_VscrollPos < 0)
      m_VscrollPos=0;
    if(m_VscrollPos > m_yHiden)
      m_VscrollPos = m_yHiden;
    SetConsoleScrollPos(true);
  }
}

// SymbolAdd
void CConsoleView::AddVar(symbol_t *sym)
{
	if(!sym->object)
		return;

	VARINFO* pItem;
	int nCount,i;
	Base_Buffer* buffer;
	double* dptr;
	char val[_MAX_PATH];
	const char *cptr;
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;

	//
	try {
		pItem = new VARINFO;
	}
	catch (CMemoryException* e) {
		e->Delete ();
		return;
	}
	nCount = (int)m_varArray.GetSize();
	pItem->strName = _T(sym->name);
	pItem->type = sym->object->TypeofOBJ();
	buffer = sym->object->GetBufferPointer();
	pItem->dim = buffer->GetDim();
	for(i = 0; i < pItem->dim; i++)
		pItem->index[i] = buffer->GetIndex(i);
	if(pItem->type == SCALAR_OBJECT) {
		dptr = (double *)buffer->GetDataPointer();
		sprintf(val,"%f",*dptr);
		pItem->strValue = _T(val);
	}
	else if(pItem->type == STRING_OBJECT) {
		if(pItem->index[0] == 1) {
			// TODO
			cptr = ((string *)buffer->GetDataPointer())[0].c_str();
			pItem->strValue = _T(cptr);
		}
		else
			pItem->strValue = _T("");
	}
	else
		pItem->strValue = _T("");
	/*An item is registered. */
	m_varArray.InsertAt( nCount, pItem );
	if(pWnd->m_wndWorkBar.IsVisible()) {
		pWnd->m_wndWorkBar.AddVar(pItem);
	}
}

// SymbolDelete
void CConsoleView::DeleteVar(symbol_t *sym)
{
	VARINFO* pItem;
	int index,i,nCount,objtype;
	CString strName;
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;

	/*The item to delete is searched. */
	if(!sym->object)
		return;
	strName = _T(sym->name);
	objtype = sym->object->TypeofOBJ();
	nCount = (int)m_varArray.GetSize();
	index = -1;
	if(nCount > 0) {
		for (i=0; i<nCount; i++) {
			pItem = (VARINFO *) m_varArray.GetAt (i);
			if(objtype == pItem->type &&
				strName.Compare(pItem->strName) == 0) {
				index = i;
				break;
			}
		}
	}

	if(index != -1) {
		/*The item specified by index is deleted. */
		if(pWnd->m_wndWorkBar.IsVisible()) {
			pWnd->m_wndWorkBar.DeleteVar(pItem);
		}
		delete (VARINFO*) m_varArray.GetAt (index);
		m_varArray.RemoveAt(index);
	}
}

// SymbolDelete
void CConsoleView::UpdateVar(symbol_t *sym)
{
	VARINFO* pItem;
	int index,i,nCount,objtype;
	CString strName;
	Base_Buffer* buf;

	/*The item to delete is searched. */
	if(!sym->object)
		return;
	strName = _T(sym->name);
	objtype = sym->object->TypeofOBJ();
	nCount = (int)m_varArray.GetSize();
	index = -1;
	if(nCount > 0) {
		for (i=0; i<nCount; i++) {
			pItem = (VARINFO *) m_varArray.GetAt (i);
			if(objtype == pItem->type &&
				strName.Compare(pItem->strName) == 0) {
				index = i;
				break;
			}
		}
	}

	if(index != -1) {
		/*The item specified by index is changed. */
		pItem = (VARINFO *) m_varArray.GetAt (index);
		buf = sym->object->GetBufferPointer();
		pItem->dim = buf->GetDim();
		for(i = 0; i < pItem->dim; i++)
			pItem->index[i] = buf->GetIndex(i);
	}
}

///////////////////////////////////////////////////////
//
//		ShellC^[tF[X
//
//
void CConsoleView::PrintText(const char *linebuf, size_t len)
{
  bool do_nextline;
  size_t i,j;
  size_t lest,oneline;
  CConsoleDoc* pDoc = GetDocument();


  SetCaretVisible(false);

  i=0;
  while(i<len){
    do_nextline = false;
    oneline = m_xMax - m_xCurPos; /* printable online char */
    lest    = len - i;               /* lest char length */
    lest    = min(lest, oneline);    /* trim lest char to oneline */
    for(j=0;j<lest;j++){             /* check \n char */
      if(linebuf[i+j] == '\n'){
        do_nextline = true;
        lest = j;
      }
    }
    if(lest > 0){
      /* do print string */
      DrawTextToConsole(m_xCurPos, m_yCurPos - m_yHiden, m_attr, &linebuf[i], lest);
      /* do save text buffer */
      if(do_nextline){
        VERIFY(pDoc->SaveTextBuffer(m_xCurPos, m_yCurPos, m_attr, &linebuf[i], lest+1));
      }else{
        VERIFY(pDoc->SaveTextBuffer(m_xCurPos, m_yCurPos, m_attr, &linebuf[i], lest));
      }
    }else{
      if(!do_nextline) break;
    }
    m_xCurPos += (int)lest;
    if(m_IsAutoNL && oneline == lest) /* check end of line */
      do_nextline = true;
    if(do_nextline){
       m_xCurPos = 0; m_yCurPos++;   /* increment cursor y position */
      if(m_yCurPos - m_yHiden == m_yMax){
        m_yHiden++;
        if(m_yMax + m_yHiden > MAX_CONSOLE_LINE){
          m_yCurPos--; m_yHiden--;
          VERIFY(pDoc->ScrollTextBuffer(true));
          Invalidate(FALSE);
        }else{
#if 1
          VERIFY(pDoc->ResizeTextBuffer(m_xMax, m_yMax + m_yHiden));
#else
		//݂̃hLg̕ۑ
		VERIFY(pDoc->InitializeSaveTextBuffer(m_xMax, m_yCurPos+1));
		//TCYύXobt@
		VERIFY(pDoc->InitializeTextBuffer(m_xMax, MAX_CONSOLE_LINE));
		//m_TextBufferɃRs[AsԂ
		pDoc->CopyTextBuffer(m_yCurPos+1);
#endif
          m_VscrollPos = m_yHiden;
          SetScrollSizes(MM_TEXT, CSize(m_xCharSize * m_xMax,
                                        m_yCharSize * (m_yMax + m_yHiden)));
          SetConsoleScrollPos(true);
        }
        UpdateWindow();
      }
    }
    i += j; /* counter increment to next char */
  }

  SetCaretVisible(true);
}

void CConsoleView::SendTextPrint(const char *buf, size_t len){
  HANDLE  hMutex;
  hMutex = OpenMutex(MUTEX_ALL_ACCESS, FALSE, TTY_MUTEX_NAME);
  WaitForSingleObject(hMutex, INFINITE);
  SendMessage(ID_TTY_TEXT_PRINT,(WPARAM)buf,len);
  ReleaseMutex(hMutex);
  CloseHandle(hMutex);
}

void CConsoleView::SendTextColor(int target, unsigned int color){
  int mode(-1);
  switch(target){
    case 0: /* foreground */
      mode = SENDMESS_TEXTMODE_FGCOLOR;
      break;
    case 1: /* background */
      mode = SENDMESS_TEXTMODE_BGCOLOR;
      break;
  }
  if(mode!=-1){
    SendMessage(ID_TTY_TEXT_MODE, mode, color);
  }
}

void CConsoleView::SendTextAttribute(int attr){
  static const int mode = SENDMESS_TEXTMODE_ATTRIBUTE;
  if(attr == TEXT_ATTRIBUTE_NORMAL ||
     attr == TEXT_ATTRIBUTE_REVERSE){
     SendMessage(ID_TTY_TEXT_MODE, mode, attr);
  }
}

void CConsoleView::SendTextEditMode(int flag){
  SendMessage(ID_TTY_TEXT_MODE, SENDMESS_TEXTMODE_EDIT, flag);
}

void CConsoleView::SendCursorGetMax(int *x, int *y){
  DWORD pos;
  pos = (unsigned int)SendMessage(ID_TTY_CURSOR, SENDMESS_CURSOR_GETMAX, 0L);
  *x = (int)HIWORD(pos);
  *y = (int)LOWORD(pos);
}

//
LRESULT CConsoleView::OnTtyTextPrint(WPARAM wParam, LPARAM lParam)
{
  const char *buf;
  size_t len;
  buf = (const char*)wParam;
  len = (size_t)lParam;
  PrintText(buf,len);
  return 0L;
}


LRESULT CConsoleView::OnTtyCursor(WPARAM wParam, LPARAM lParam)
{
  CConsoleDoc *pDoc;
  int mode;
  DWORD pos;
  WORD x, y;
  mode = (int)wParam;
  switch(mode){
  case SENDMESS_CURSOR_GETMAX:
    y = (WORD)m_yMax;
    x = (WORD)m_xMax;
    pos = (x << 16) + y;
    return pos;
  case SENDMESS_CURSOR_DOWN:
    m_yCurPos += (int)lParam;
    if(m_yCurPos - m_yHiden >= m_yMax) m_yCurPos = m_yHiden + m_yMax - 1;
    break;
  case SENDMESS_CURSOR_UP:
    m_yCurPos -= (int)lParam;
    if(m_yCurPos < m_yHiden) m_yCurPos = m_yHiden;
    break;
  case SENDMESS_CURSOR_RIGHT:
    m_xCurPos += (int)lParam;
    if(m_xCurPos > m_xMax) m_xCurPos = m_xMax;
    break;
  case SENDMESS_CURSOR_LEFT:
    m_xCurPos -= (int)lParam;
    if(m_xCurPos < 0) m_xCurPos = 0;
    break;
  case SENDMESS_CURSOR_BOL:
    m_xCurPos = 0;
    break;
  case SENDMESS_CURSOR_NEWLINE:
    pDoc = GetDocument();
    m_xCurPos = 0; m_yCurPos++;   /* increment cursor y position */
    if(m_yCurPos - m_yHiden == m_yMax){
      m_yHiden++;
      if(m_yMax + m_yHiden > MAX_CONSOLE_LINE){
        m_yCurPos--; m_yHiden--;
        VERIFY(pDoc->ScrollTextBuffer(true));
          Invalidate(FALSE);
          UpdateWindow();
        }else{
#if 1
          VERIFY(pDoc->ResizeTextBuffer(m_xMax, m_yMax + m_yHiden));
          m_VscrollPos = m_yHiden;
          SetScrollSizes(MM_TEXT, CSize(m_xCharSize * m_xMax,
                                        m_yCharSize * (m_yMax + m_yHiden)));
          SetConsoleScrollPos(true);
#else
		//݂̃hLg̕ۑ
		VERIFY(pDoc->InitializeSaveTextBuffer(m_xMax, m_yCurPos+1));
		//TCYύXobt@
		VERIFY(pDoc->InitializeTextBuffer(m_xMax, MAX_CONSOLE_LINE));
		//m_TextBufferɃRs[AsԂ
		pDoc->CopyTextBuffer(m_yCurPos+1);
        m_VscrollPos = m_yHiden;
        SetScrollSizes(MM_TEXT, CSize(m_xCharSize * m_xMax,
                                        m_yCharSize * (m_yMax + m_yHiden)));
        SetConsoleScrollPos(true);
#endif
        }
      }
    break;
  }
  SetCaretVisible(false); SetCaretVisible(true); // reprint caret
  return 0L;
}

//
// shellC^[tF[XI
//
//////////////////////////////////////////////////////////////////////

void CConsoleView::PasteClipBoardToTextBuffer()
{
  DWORD size;
  size_t i,len;
  const char *str;
  COleDataObject clipboardData;
  HGLOBAL hText;
  int *cmdbuf;
  
  clipboardData.AttachClipboard();
  if (!clipboardData.IsDataAvailable(CF_TEXT)) {
    ::MessageBeep(MB_ICONEXCLAMATION);
    return;
  }

  hText = clipboardData.GetGlobalData(CF_TEXT);
  if(hText == NULL) return;
  str = (const char *)::GlobalLock(hText);
  len = strlen(str);
  //for(i=0; i<len; i++){
  //  ch = (unsigned char)str[i];
  //	WriteFile(m_hStdinW, &ch, sizeof(int), &size, NULL);
  //}
  cmdbuf = (int *)malloc(len*sizeof(int)+1);
  for(i = 0; i < len; i++) {
	cmdbuf[i] = (unsigned char)str[i];
  }
  WriteFile(m_hStdinW, cmdbuf, (DWORD)(sizeof(int)*len), &size, NULL);
  free(cmdbuf);

  // release resource data
  ::GlobalUnlock(hText);
  ::GlobalFree(hText);
}

void CConsoleView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
  /* extended key */
  bool do_input;
  int keycode;
  DWORD dwLen;
  do_input = false;
  switch(nChar){
  case VK_DELETE: keycode = 0x7f;   do_input = true; break;
  case VK_F1:     keycode = SL_TTY::F1KEY    << 8; do_input = true; break;
  case VK_F2:     keycode = SL_TTY::F2KEY    << 8; do_input = true; break;
  case VK_F3:     keycode = SL_TTY::F3KEY    << 8; do_input = true; break;
  case VK_F4:     keycode = SL_TTY::F4KEY    << 8; do_input = true; break;
  case VK_F5:     keycode = SL_TTY::F5KEY    << 8; do_input = true; break;
  case VK_F6:     keycode = SL_TTY::F6KEY    << 8; do_input = true; break;
  case VK_F7:     keycode = SL_TTY::F7KEY    << 8; do_input = true; break;
  case VK_F8:     keycode = SL_TTY::F8KEY    << 8; do_input = true; break;
  case VK_F9:     keycode = SL_TTY::F9KEY    << 8; do_input = true; break;
  case VK_F10:    keycode = SL_TTY::F10KEY   << 8; do_input = true; break;
  case VK_F11:    keycode = SL_TTY::F11KEY   << 8; do_input = true; break;
  case VK_F12:    keycode = SL_TTY::F12KEY   << 8; do_input = true; break;
  case VK_HOME:   keycode = SL_TTY::HOMEKEY  << 8; do_input = true; break;
  case VK_END:    keycode = SL_TTY::ENDKEY   << 8; do_input = true; break;
  case VK_NEXT:   keycode = SL_TTY::PGUPKEY  << 8; do_input = true; break;
  case VK_PRIOR:  keycode = SL_TTY::PGDNKEY  << 8; do_input = true; break;
  case VK_UP:     keycode = SL_TTY::UPKEY    << 8; do_input = true; break;
  case VK_DOWN:   keycode = SL_TTY::DOWNKEY  << 8; do_input = true; break;
  case VK_RIGHT:  keycode = SL_TTY::RIGHTKEY << 8; do_input = true; break;
  case VK_LEFT:   keycode = SL_TTY::LEFTKEY  << 8; do_input = true; break;
  case VK_SCROLL: keycode = SL_TTY::SCRLKKEY << 8; do_input = true; break;
  }
  if(do_input){
    unsigned int i;
    /* restore scroll bar position */
    if(m_VscrollPos != m_yHiden){
      m_VscrollPos = m_yHiden;
//if(m_VscrollPos == 278)
//	m_VscrollPos = 278;
      SetConsoleScrollPos(true);
    }
    for(i=0; i<nRepCnt; i++){
      WriteFile(m_hStdinW, &keycode, sizeof(int), &dwLen, NULL);
    }
    return;
  }

  CScrollView::OnKeyDown(nChar, nRepCnt, nFlags);

}

void CConsoleView::OnSize(UINT nType, int cx, int cy) 
{
  //OĂȂƁAQdCxgɂȂ
  //CScrollView::OnSize(nType, cx, cy);

  if(IsWindowVisible()){
    CConsoleDoc *pDoc;
    int fix;
    int xmax_pre, ymax_pre;
    int yhiden_pre, yhiden_post;

    pDoc = GetDocument();

    xmax_pre = m_xMax;
    ymax_pre = m_yMax;
    yhiden_pre = m_yHiden;
    GetWindowProperties();

    if(xmax_pre == m_xMax && ymax_pre == m_yMax)
		return;
	//n߂ɁATCY
	if(xmax_pre != m_xMax) {
		//hLgt@C
		pDoc->SaveTextBuffer("test1.lst");
		//݂̃hLg̕ۑ
		VERIFY(pDoc->InitializeSaveTextBuffer(xmax_pre, m_yCurPos+1));
		//TCYύXobt@
		VERIFY(pDoc->InitializeTextBuffer(m_xMax, MAX_CONSOLE_LINE));
		//m_TextBufferɃRs[AsԂ
		yhiden_post = pDoc->CopyTextBuffer(m_yCurPos+1);
		//hLgt@C
		pDoc->SaveTextBuffer("test2.lst");

		//TCYݒ
		//m_yHiden     = yhiden_pre + yhiden_post;
		m_yCurPos += yhiden_post;
		if(m_yCurPos >= MAX_CONSOLE_LINE)
			m_yCurPos = MAX_CONSOLE_LINE -1;
		m_yHiden = m_yCurPos - m_yMax + 1;	// TODO:+1͉̂
		if(m_yHiden < 0)
			m_yHiden = 0;
	}
	if(xmax_pre == m_xMax && ymax_pre != m_yMax) {
		if(ymax_pre > m_yMax) {
			fix = ymax_pre - m_yMax;
			if((m_yCurPos+fix) > m_yMax)
				m_yHiden += fix;
		}
		else {
			fix = m_yMax - ymax_pre;
			m_yHiden -= fix;
			if(m_yHiden < 0)
				m_yHiden = 0;
		}
	}

	//hLgTCY
	SetScrollSizes(MM_TEXT, CSize(m_xCharSize * m_xMax, 
                                m_yCharSize * (m_yMax + m_yHiden)));
	//TODO:XN[ʒúABĂ̈敪ł
	m_VscrollPos = m_yHiden;
	//if(m_VscrollPos == 278)
	//	m_VscrollPos = 278;
	//if(m_yCurPos == MAX_CONSOLE_LINE-1)
	//	m_VscrollPos -= 1;
	//printf("%d %d %d\n",m_yCurPos, m_yHiden, m_VscrollPos);
	if(m_VscrollPos < 0)
		m_VscrollPos = 0;
    SetConsoleScrollPos(true);
  }
}

void CConsoleView::FixedSelectedCopyArea()
{
  int tmp;
  if(m_xsCopyArea < 0) m_xsCopyArea = 0;
  if(m_ysCopyArea < 0) m_ysCopyArea = 0;
  if(m_xeCopyArea < 0) m_xeCopyArea = 0;
  if(m_yeCopyArea < 0) m_yeCopyArea = 0;

  /* swap y pos and x pos */
  if(m_ysCopyArea > m_yeCopyArea){
    tmp = m_yeCopyArea; m_yeCopyArea = m_ysCopyArea; m_ysCopyArea = tmp;
    tmp = m_xeCopyArea; m_xeCopyArea = m_xsCopyArea; m_xsCopyArea = tmp;
  }else if(m_ysCopyArea == m_yeCopyArea && m_xsCopyArea > m_xeCopyArea){
    tmp = m_xeCopyArea; m_xeCopyArea = m_xsCopyArea; m_xsCopyArea = tmp;
  }

  /* check buffer size */
  if(m_yeCopyArea >= m_yMax + m_yHiden) m_yeCopyArea = m_yMax + m_yHiden - 1;
  if(m_xeCopyArea > m_xMax)             m_xeCopyArea = m_xMax;
}


BOOL CConsoleView::PreCreateWindow(CREATESTRUCT& cs) 
{
	// TODO: ̈ʒuɌŗL̏ǉ邩A܂͊{NXĂяoĂ

	return CScrollView::PreCreateWindow(cs);
}

void CConsoleView::OnUpdateFileSave(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	pCmdUI->Enable(FALSE);	
}

BOOL CConsoleView::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message) 
{
	//TODO
	//FileViewer̃hbOhbv
	//if(g_pDriveDoc) {
	//	//check m_pDragImge
	//	if(m_pDragImage) {
	//		//
	//		ShowCursor(FALSE);
	//		//hbOJn
	//		m_pDragImage->BeginDrag(0,CPoint(0,0));
	//		//NULLw
	//		m_pDragImage->DragEnter(this, CPoint(0,0));
	//		//J[\Lv`
	//		::SetCapture(this->GetSafeHwnd());
	//	}
	//}

	return CScrollView::OnSetCursor(pWnd, nHitTest, message);
}

void CConsoleView::OnMouseMove(UINT nFlags, CPoint point) 
{
  if(m_bSelCopyArea){
    m_xeCopyArea = point.x / m_xCharSize;
    m_yeCopyArea = point.y / m_yCharSize + m_VscrollPos;
    if(m_xeCopyArea < 0) m_xeCopyArea = 0;
    if(m_yeCopyArea < 0) m_yeCopyArea = 0;
    GetDocument()->ClearSelectedBuffer();
    VERIFY(GetDocument()->SelectTextBuffer(m_xsCopyArea, m_ysCopyArea, 
                                           m_xeCopyArea, m_yeCopyArea));
    Invalidate(FALSE);
  }
  /*When an image exists */
  //CClientDC dc (this);
  //OnPrepareDC (&dc);
  //dc.DPtoLP (&point);
  if(m_pDragImage) {
	m_pDragImage->DragEnter(NULL, point);
	m_pDragImage->DragMove(point);
  }

  CScrollView::OnMouseMove(nFlags, point);

}

void CConsoleView::OnEditPaste() 
{
  PasteClipBoardToTextBuffer();
}

void CConsoleView::OnUpdateEditPaste(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable (::IsClipboardFormatAvailable (CF_TEXT));
}

LRESULT CConsoleView::OnTtyTextMode(WPARAM wParam, LPARAM lParam)
{
  int mode;
  mode = (int)wParam;
  if(mode == SENDMESS_TEXTMODE_FGCOLOR ||
    mode == SENDMESS_TEXTMODE_BGCOLOR){
    unsigned int color,red,green,blue;
    color = (unsigned int)lParam;
    red   = (0xff0000 & color) >> 16;
    green = (0x00ff00 & color) >> 8;
    blue  = (0x0000ff & color);
    switch(mode){
    case SENDMESS_TEXTMODE_FGCOLOR: /* foreground */
      m_textColor = RGB(red,green,blue);
      break;
    case SENDMESS_TEXTMODE_BGCOLOR: /* background */
      m_backColor = RGB(red,green,blue);
      break;
    }
  }else if(mode == SENDMESS_TEXTMODE_ATTRIBUTE){
    int attr;
    attr = (int)lParam;
    if(attr == TEXT_ATTRIBUTE_NORMAL)
      m_attr = attr;
    else
      m_attr |= attr;
  }else if(mode == SENDMESS_TEXTMODE_EDIT){
    int editmode;
    editmode = (int)lParam;
    if(editmode == 0){
      SetCaretVisible(false);
      m_IsAutoNL = true;
    }else{
      m_IsAutoNL = false;
      SetCaretVisible(true);
    }
  }
  return 0L;
}


void CConsoleView::OnUpdateLineNumber(CCmdUI* pCmdUI)
{
    int nLine = m_yCurPos;

    CString string;
    string.Format (_T ("Line %d"), nLine);
    pCmdUI->Enable (TRUE);
    pCmdUI->SetText (string);
}

void CConsoleView::OnUpdateConsole(CCmdUI* pCmdUI) 
{
	// ɊJĂ̂
	pCmdUI->Enable(FALSE);	
}

void CConsoleView::OnUpdateFileClose(CCmdUI* pCmdUI) 
{
	// TODO: ̈ʒu command update UI nhp̃R[hǉĂ
	pCmdUI->Enable(FALSE);	
}

void CConsoleView::OnLButtonDown(UINT nFlags, CPoint point) 
{
  // if already selected copy area then clear selected flag
  m_bHasCopyArea = false;
  GetDocument()->ClearSelectedBuffer();
  Invalidate(TRUE);
  UpdateWindow();
  m_xsCopyArea = point.x / m_xCharSize;
  m_ysCopyArea = point.y / m_yCharSize + m_VscrollPos;
  if(m_xsCopyArea < 0) m_xsCopyArea = 0;
  if(m_ysCopyArea < 0) m_ysCopyArea = 0;
  m_xeCopyArea = 0;
  m_yeCopyArea = 0;
  m_bSelCopyArea = true;
  SetCapture();
  // CScrollView::OnLButtonDown(nFlags, point);

}

void CConsoleView::OnLButtonUp(UINT nFlags, CPoint point) 
{
  CMainFrame * pWnd;
  pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;

  if(m_bSelCopyArea){
    m_xeCopyArea = point.x / m_xCharSize;
    m_yeCopyArea = point.y / m_yCharSize + m_VscrollPos;
    if(m_xeCopyArea < 0) m_xeCopyArea = 0;
    if(m_yeCopyArea < 0) m_yeCopyArea = 0;
    ReleaseCapture();
    Invalidate(FALSE);
    if(!(m_xsCopyArea == m_xeCopyArea && m_ysCopyArea == m_yeCopyArea)){
      VERIFY(GetDocument()->SelectTextBuffer(m_xsCopyArea, m_ysCopyArea, 
                                             m_xeCopyArea, m_yeCopyArea));
      m_bHasCopyArea = true;
      pWnd->GetMenu()->EnableMenuItem(ID_EDIT_COPY, MF_ENABLED | MF_BYCOMMAND);
    }
    m_bSelCopyArea = false;
    m_xsCopyArea = 0;
    m_xeCopyArea = 0;
    m_ysCopyArea = 0;
    m_yeCopyArea = 0;
  }
  //When an image exists 
  if(m_pDragImage) {
	m_pDragImage->EndDrag();
	delete m_pDragImage;
	::ReleaseCapture();
	m_pDragImage = NULL;
	ShowCursor(TRUE);

	// ChildFrameŎs
	strcpy(g_charScriptText, m_strDragText);
	this->GetParent()->PostMessage(ID_SCRIPT_EXE, 0, 0L);
	// Active
	g_pConsole->SetFocus();
  }

  // CScrollView::OnLButtonUp(nFlags, point);

}

//	Draw text to console view window
void CConsoleView::DrawTextToConsole(int xpos, int ypos, int attr, const char *str, size_t len)
{
  CDC *dc;
  CFont* pOldFont;
  const static char *dummy=" ";
  int x,y;
  size_t i,space(0);
  for(i=0;i<len;i++){
    if(str[i] == '\n'){
      space = len - i; len = i; break;
    }
  }
  dc = GetDC();
  pOldFont = dc->SelectObject( m_font );
  if(attr & TEXT_ATTRIBUTE_SELECT){
    dc->SetTextColor(m_textColor_sel);
    dc->SetBkColor(m_backColor_sel);
  }else{
    if(attr & TEXT_ATTRIBUTE_REVERSE){
      dc->SetTextColor(m_backColor);
      dc->SetBkColor(m_textColor);
    }else{
      dc->SetTextColor(m_textColor);
      dc->SetBkColor(m_backColor);
    }
  }
  x = m_xCharSize * xpos;
  y = m_yCharSize * ypos;
  if(len != 0)dc->TextOut(x, y, str, (int)len);
  for(i=0;i<space;i++){
    dc->TextOut(x + (int)(len + i) * m_xCharSize, y, dummy, 1);
  }
  dc->SelectObject(pOldFont);
  ReleaseDC( dc );
}

LRESULT CConsoleView::OnTtyExit(WPARAM wParam, LPARAM lParam)
{
  CWnd * pWnd;
  pWnd = AfxGetApp()->m_pMainWnd;
  pWnd->PostMessage(WM_CLOSE);
  return 0L;
}

void CConsoleView::OnDestroy() 
{

	CScrollView::OnDestroy();
	
	// TODO: ̈ʒuɃbZ[W nhp̃R[hǉĂ
	
}

void CConsoleView::OnWinSigma() 
{
	OpenWinSigma();
}

void CConsoleView::OpenWinSigma()
{
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	//MainFrameToolbarBAm_bShowToolbarFALSEɂ̂
	//((CMainFrame *)AfxGetMainWnd())->ShowToolbar(FALSE);
	pWnd->m_bShowToolbar = FALSE;
	//SIGMA
	ExtncsInit(AfxGetApp());
	//SIGMAR[
	ExtncsExec( AfxGetApp(), this, m_hStdinW );
	m_bExtendSigma = TRUE;
}

void CConsoleView::OnUpdateWinSigma(CCmdUI* pCmdUI) 
{
	//if(m_bExtendSigma)
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	if(pWnd->m_bShowToolbar)
		pCmdUI->SetCheck(0);
	else
		pCmdUI->SetCheck(1);
	
}

BOOL CConsoleView::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) 
{
  static const int nline = 3;
  static short total = 0;
  short vsize;
  BOOL horz,vart; 
  CheckScrollBars(&horz,&vart);
  if(!vart) /* console window has not scroll bar */
    return FALSE;
  total += zDelta;
  vsize = total / WHEEL_DELTA;
  total = total % WHEEL_DELTA;
  if(vsize == 0) return FALSE;
   m_VscrollPos -= nline * vsize;
  if(vsize < 0){
    if(m_VscrollPos > m_yHiden) m_VscrollPos = m_yHiden;
  }else{
    if(m_VscrollPos < 0) m_VscrollPos = 0;
  }
  SetConsoleScrollPos(true);
  return TRUE;
  // return CScrollView::OnMouseWheel(nFlags, zDelta, pt);
}

void CConsoleView::OnSetFocus(CWnd* pOldWnd) 
{
	CScrollView::OnSetFocus(pOldWnd);
	
	CreateSolidCaret(2,m_yCharSize);
	m_HasCaret = true;
	SetCaretVisible(true);
	
}

void CConsoleView::OnKillFocus(CWnd* pNewWnd) 
{
	CScrollView::OnKillFocus(pNewWnd);
	
	SetCaretVisible(false);
	::DestroyCaret();
	m_HasCaret = false;
	
}

void CConsoleView::OnAppExit() 
{
  //if(m_hShellThread != INVALID_HANDLE_VALUE){
  //  setsigint_win32();
  //  CloseHandle(m_hStdinW);
  //  m_hStdinW = INVALID_HANDLE_VALUE;
  //}
	char cmdline[_MAX_PATH];
	strcpy(cmdline, "exit\n");
	int i,ch,len;
	DWORD size;

	//A command is transmitted. 
  if(m_hShellThread != INVALID_HANDLE_VALUE){
	len = (int)strlen(cmdline);
	for(i=0; i<len; i++){
		ch = (unsigned char)cmdline[i];
		WriteFile(m_hStdinW, &ch, sizeof(int), &size, NULL);
	}
  }
}

void CConsoleView::OnEditCopy() 
{
  VERIFY(GetDocument()->SetTextBufferToClipboard());
	
}

void CConsoleView::OnUpdateEditCopy(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_bHasCopyArea);
	
}

void CConsoleView::CheckScrollBars(BOOL* horz,BOOL* vert){
#if 0
	CPoint pt;
	pt = GetScrollPosition();
	if(pt.x == 0)
		*horz = FALSE;
	else
		*horz = TRUE;
	if(pt.y == 0)
		*vert = FALSE;
	else
		*vert = TRUE;
#else
	int minPos, maxPos;
	this->GetScrollRange(SB_HORZ, &minPos, &maxPos);
	if(minPos == 0 && maxPos == 0)
		*horz = FALSE;
	else
		*horz = TRUE;
	this->GetScrollRange(SB_VERT, &minPos, &maxPos);
	if(minPos == 0 && maxPos == 0)
		*vert = FALSE;
	else
		*vert = TRUE;
#endif
}

/*Module information is set as m_cmdArray. */
void CConsoleView::ModuleLoad(char *module_name, char *module_file)
{
	CString msgArray[MAX_ARG];
	/*A modular node item is set up.*/
	SetCommandItem(0, "", module_name, "", 0, "", msgArray);
	/*module_file is read and a child item is set up. */
	ModuleList(module_name, module_file);
	/*It will transmit, if CommandViewer is open. */
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	if(pWnd->m_wndCmdBar.IsVisible())
		pWnd->m_wndCmdBar.ModuleLoad(module_name, &m_cmdArray);
}

BOOL CConsoleView::ModuleList(char *module_name, char *module_file)
{
	FILE *fp;
	char openfile[_MAX_PATH];
	char name[_MAX_PATH];
	char func[_MAX_PATH];
	char msg[_MAX_PATH];
	BOOL cmdfg;
	CString cmdStr,msgStr,msgArray[MAX_ARG];
	char pargs[256];
	int argnum;

	strcpy(openfile, module_file);
	if((fp = fopen(openfile, "r")) == NULL) {
		return FALSE;
	}
	for(cmdfg = FALSE;;) {
		strcpy(name, module_name);
		if(fgets(func, _MAX_PATH, fp) == NULL)
			break;
		if(!cmdfg && strstr(func,"Command {") != NULL)
			cmdfg = TRUE;
		else if(cmdfg) {
			if(strstr(func,"}") != NULL)
				break;
			else {
				/*A command name is extracted and it is set as func. */
				strcpy(pargs, "");
				cmdStr = GetCommandName(module_file, func, &argnum, pargs, msgArray);
				if(cmdStr.GetLength() > 0) {
					strcpy(func, cmdStr);
					/*Additional information is set up. */
					msgStr = GetCommandMsg(module_file, func);
					strcpy(msg, msgStr);
					/*An item is registered. */
					SetCommandItem(1,name,func, msg, argnum, pargs, msgArray);
				}
			}
		}
	}
	fclose(fp);
	return TRUE;
}

CString CConsoleView::GetCommandMsg(char *module, char *name)
{
	FILE *fp;
	char msg[_MAX_PATH], linebuf[_MAX_PATH], openfile[_MAX_PATH],*cptr;
	BOOL cmdfg;
	CString msgStr = _T("");

	sprintf(openfile,"%s2", module);
	if((fp = fopen(openfile, "r")) == NULL) {
		return msgStr;
	}
	for(cmdfg = FALSE;;) {
		sprintf(msg, "\"%s\"", name);
		if(fgets(linebuf, _MAX_PATH, fp) == NULL)
			break;
		if(!cmdfg && strstr(linebuf,"Explain {") != NULL)
			cmdfg = TRUE;
		else if(cmdfg) {
			if(strstr(linebuf,"}") != NULL)
				break;
			else {
				if((cptr = strstr(linebuf, msg)) != NULL) {
					cptr += strlen(msg);
					msgStr = GetMsg( cptr );
					break;
				}
			}
		}
	}
	fclose(fp);
	return msgStr;
}

CString CConsoleView::GetMsg(char *buf)
{
	char text[_MAX_PATH],*cptr;
	int addr;
	CString strMsg = _T("");

	cptr = buf;
	addr = -1;
	while(*cptr != '\0') {
		if(addr == -1 && *cptr == '\"') {
			addr = 0;
		} else if(addr >= 0 && *cptr == '\"')
			break;
		else {
			if(addr >= 0) {
				text[addr++] = *cptr;
			}
		}
		cptr++;
	}
	if(addr >= 0) {
		text[addr] = '\0';
		strMsg = _T(text);
	}
	return strMsg;

}

CString CConsoleView::GetCommandName(char *file, char *linebuf,
					int *argn, char *args, CString msgArray[])
{
	CString str,strRight,strLeft,strCmd,strArg;
	int pos, pos1,argnum=0,arg,i,msgnum;
	char argname[64];
	CString strArgs;

	//initialize
	str = _T(linebuf);
	strArgs = _T("");
	//"܂
	pos = str.Find(_T("\""));
	pos1 = str.Find(_T("#"));
	if(pos1 == -1 && pos != -1 ) {
		//name
		strRight = str.Mid(pos+1);
		pos = strRight.Find(_T("\""));
		strLeft = strRight.Left(pos);
		strCmd = strLeft;
		//function
		strRight = strRight.Mid(pos+1);
		pos = strRight.Find(_T("\""));
		strRight = strRight.Mid(pos+1);
		pos = strRight.Find(_T("\""));
		strLeft = strRight.Left(pos);	//function
		//number
		strRight = strRight.Mid(pos+1);
		pos = strRight.Find(_T(","));
		strRight = strRight.Mid(pos+1);
		pos = strRight.Find(_T(","));
		if(pos == -1) {
			//Ȃ
			argnum = 0;
		}
		else {
			strArg = strRight.Left(pos);	//argnum
			argnum = atoi(strArg);
			strRight = strRight.Mid(pos+1);
			//strRightXg쐬
			for(arg = 0; arg < argnum; arg++) {
				pos = strRight.Find(_T("\""));
				strRight = strRight.Mid(pos+1);
				pos = strRight.Find(_T(","));
				strLeft = strRight.Left(pos-1);	//args
				strRight = strRight.Mid(pos+1);
				//delete "\"
				pos=0;
				for(i = 0; i < strLeft.GetLength(); i++) {
					if(strLeft.GetAt(i) != '\\')
						argname[pos++] = strLeft.GetAt(i);
				}
				argname[pos] = '\0';
				strArgs += _T(argname);
				if(arg != argnum-1)
					strArgs += _T(",");
			}
			//bZ[Wԍ擾
			for(arg = 0; arg < argnum; arg++) {
				pos = strRight.Find(_T(","));
				if(pos == -1) {
					/*bZ[W͂P*/
					msgnum = atoi(strRight);
					msgArray[arg] = GetArgMessage(file, msgnum);
					break;
				}
				else {
					strLeft = strRight.Left(pos);
					msgnum =atoi(strLeft);
					msgArray[arg] = GetArgMessage(file, msgnum);
					strRight = strRight.Mid(pos+1);
				}
			}
		}
	}
	else {
		strCmd = _T("");
	}
	//alloc memory
	if(strArgs.GetLength() >0) {
		strcpy(args, strArgs);
	}
	*argn = argnum;
	return strCmd;
}

CString CConsoleView::GetArgMessage(char *file, int msgnum)
{
	FILE *fp;
	char linebuf[_MAX_PATH], openfile[_MAX_PATH];
	BOOL cmdfg;
	CString tmpStr,numStr;
	int pos,pos1,num,i,len;
	CString msgStr = _T("");

	sprintf(openfile,"%s", file);
	if((fp = fopen(openfile, "r")) == NULL) {
		return msgStr;
	}
	for(cmdfg = FALSE;;) {
		if(fgets(linebuf, _MAX_PATH, fp) == NULL)
			break;
		if(!cmdfg && strstr(linebuf,"Message {") != NULL)
			cmdfg = TRUE;
		else if(cmdfg) {
			if(strstr(linebuf,"}") != NULL)
				break;
			else {
				/* check number */
				tmpStr = _T(linebuf);
				pos = tmpStr.Find(",");
				if(pos != -1) {
					numStr = tmpStr.Left(pos);
					pos1 = numStr.Find("#");
					if(pos1 == -1) {
						//RgłȂ
						num = atoi(numStr);
						if(num == msgnum) {
							msgStr = tmpStr.Mid(pos+1);
							break;
						}
					}
				}
			}
		}
	}
	fclose(fp);
	if(msgStr.GetLength() > 0) {
		/*`*/
		tmpStr = msgStr;
		pos = tmpStr.Find("\"");
		if(pos != -1) {
			numStr = tmpStr.Mid(pos);
			len = numStr.GetLength();
			for(i=len-1;i>0; i--) {
				if(numStr.GetAt(i) == '\"') {
					//numStr.SetAt(i+1, '\0');
					break;
				}
			}
			if(i > 0)
				msgStr = numStr.Left(i+1);

			//pos1 = numStr.Find("\"",1);
			//if(pos1 != -1) {
			//	/*ꂪ*/
			//	msgStr = numStr.Left(pos1+1);
			//}
		}
	}
	return msgStr;
}

/*
 * parent -> module
 * func -> command
 * msg -> explain
 */
BOOL CConsoleView::SetCommandItem(int node, char *parent, char *func, char *msg,
					int argnum, char *args, CString msgs[])
{
	CMDINFO* pItem;
	int nCount,i;

	/**/
	try {
		pItem = new CMDINFO;
	}
	catch (CMemoryException* e) {
		e->Delete ();
		return NULL;
	}
	pItem->node = node;
	pItem->strParent = _T(parent);
	pItem->strName = _T(func);
	pItem->strExplain = _T(msg);
	pItem->argnum = argnum;
	pItem->strArgs = _T(args);
	for(i = 0; i < argnum; i++)
		pItem->strMsg[i] = msgs[i];
	//o^
	nCount = (int)m_cmdArray.GetSize();
	m_cmdArray.InsertAt( nCount, pItem );
	return TRUE;	
}


void CConsoleView::OnUpdateFileAll(CCmdUI* pCmdUI) 
{
	int ftype;
	CMainFrame * pWnd;

	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	if(!pWnd->m_wndDriveBar.IsVisible())
		pCmdUI->Enable(FALSE);
	ftype = pWnd->m_wndDriveBar.GetFtype();
	if(ftype == FILE_ALL)
		pCmdUI->SetCheck(TRUE);
	else
		pCmdUI->SetCheck(FALSE);
	
}

void CConsoleView::OnUpdateFileMdl(CCmdUI* pCmdUI) 
{
	int ftype;
	CMainFrame * pWnd;

	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	if(!pWnd->m_wndDriveBar.IsVisible())
		pCmdUI->Enable(FALSE);
	ftype = pWnd->m_wndDriveBar.GetFtype();
	if(ftype == FILE_MDL)
		pCmdUI->SetCheck(TRUE);
	else
		pCmdUI->SetCheck(FALSE);
	
}

void CConsoleView::OnUpdateFileSl(CCmdUI* pCmdUI) 
{
	int ftype;
	CMainFrame * pWnd;

	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	if(!pWnd->m_wndDriveBar.IsVisible())
		pCmdUI->Enable(FALSE);
	ftype = pWnd->m_wndDriveBar.GetFtype();
	if(ftype == FILE_SL)
		pCmdUI->SetCheck(TRUE);
	else
		pCmdUI->SetCheck(FALSE);
	
}

void CConsoleView::OnUpdateFileTxt(CCmdUI* pCmdUI) 
{
	int ftype;
	CMainFrame * pWnd;

	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	if(!pWnd->m_wndDriveBar.IsVisible())
		pCmdUI->Enable(FALSE);
	ftype = pWnd->m_wndDriveBar.GetFtype();
	if(ftype == FILE_TXT)
		pCmdUI->SetCheck(TRUE);
	else
		pCmdUI->SetCheck(FALSE);
	
}

void CConsoleView::OnFileAll() 
{
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	pWnd->m_wndDriveBar.SetFtype(FILE_ALL);
	
}

void CConsoleView::OnFileMdl() 
{
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	pWnd->m_wndDriveBar.SetFtype(FILE_MDL);
	
}

void CConsoleView::OnFileSl() 
{
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	pWnd->m_wndDriveBar.SetFtype(FILE_SL);
	
}

void CConsoleView::OnFileTxt() 
{
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	pWnd->m_wndDriveBar.SetFtype(FILE_TXT);
	
}

void CConsoleView::OnScriptRun() 
{
	BOOL enable;
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	enable = pWnd->m_wndDriveBar.SetScriptFile();
	if(enable)
		ScriptRun(INLINE_MODE);
}

void CConsoleView::OnUpdateScriptRun(CCmdUI* pCmdUI) 
{
	//IĂt@CslȂL	
	BOOL enable;
	CMainFrame * pWnd;
	pWnd = (CMainFrame*)AfxGetApp()->m_pMainWnd;
	enable = pWnd->m_wndDriveBar.IsScriptFile();
	if(enable) {
		if(m_bScriptRun)
			pCmdUI->Enable(FALSE);
		else
			pCmdUI->Enable(TRUE);
	}
	else
		pCmdUI->Enable(FALSE);
}

void CConsoleView::OnScriptStop() 
{
	ScriptStop();	
}

void CConsoleView::OnUpdateScriptStop(CCmdUI* pCmdUI) 
{
	//s
	if(m_bScriptRun)
		pCmdUI->Enable(TRUE);
	else
		pCmdUI->Enable(FALSE);
	
}

void CConsoleView::ScriptRun(int mode)
{
	this->GetParent()->PostMessage(ID_SCRIPT_EXE, mode, 0L);
}

void CConsoleView::ScriptStop()
{
	this->GetParent()->PostMessage(ID_SCRIPT_QUIT, 0, 0L);
}
