/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: win32_console.cpp,v 1.8 2005/03/17 14:10:30 orrisroot Exp $ */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <windows.h>
#include "SL_header.h"


#include <stdio.h>
#include <conio.h>

#include "shell/satellite4.h"
#include "win32_console.h"

using namespace std;


win32_console::win32_console() : is_buffered(false), oflag(true), isReverse(false){
  hCurProcess=GetCurrentProcess();
  hStdin  = GetStdHandle( STD_INPUT_HANDLE );
  hStdout = GetStdHandle( STD_OUTPUT_HANDLE );
  hStderr = GetStdHandle( STD_ERROR_HANDLE );

  GetConsoleMode( hStdin, &dwNormalModeIn );
  GetConsoleMode( hStdout, &dwNormalModeOut );

  GetConsoleMode( hStdout, &dwNormalModeErr );
  dwRawModeIn  = ENABLE_WINDOW_INPUT | ENABLE_MOUSE_INPUT;
  dwRawModeOut = ENABLE_PROCESSED_OUTPUT;
  dwRawModeErr = ENABLE_PROCESSED_OUTPUT;

  GetConsoleScreenBufferInfo( hStdout, &cInfo );

  TextAttrib = cInfo.wAttributes;
  tty_normalmode();
  term_set_fgcolor(SL_TTY::WHITE);
  term_set_bgcolor(SL_TTY::BLACK);
  term_clear_screen();

  ifd.handle = hStdin;
  ofd.handle = hStdout;
  efd.handle = hStderr;
}

win32_console::~win32_console(){
  SetConsoleTextAttribute( hStdout, cInfo.wAttributes );
  tty_normalmode();
}

void win32_console::tty_rawmode(){
  SetConsoleMode( hStdin, dwRawModeIn );
  SetConsoleMode( hStdout, dwRawModeOut );
  SetConsoleMode( hStderr, dwRawModeErr );

  // SetConsoleTextAttribute( hStdout, TextAttrib );
}

void win32_console::tty_normalmode(){
  SetConsoleMode( hStdin, dwNormalModeIn );
  SetConsoleMode( hStdout, dwNormalModeOut );

  SetConsoleMode( hStderr, dwNormalModeErr );
}

void win32_console::term_set_fgcolor(SL_TTY::COLOR color){
  if( isReverse ){
    isReverse = false;
    SL_TTY::COLOR tmp;
    tmp = fg;
    term_set_fgcolor(bg);
    term_set_bgcolor(tmp);
  }
  if( color == SL_TTY::BLACK ){
    TextAttrib = (TextAttrib & ~FOREGROUND_INTENSITY) &
      ~FOREGROUND_RED   & ~FOREGROUND_BLUE  & ~FOREGROUND_GREEN;
  } else if ( color == SL_TTY::RED ) {
    TextAttrib = (TextAttrib |  FOREGROUND_INTENSITY) &
      ~FOREGROUND_GREEN & ~FOREGROUND_BLUE  |  FOREGROUND_RED;
  } else if ( color == SL_TTY::GREEN ) {
    TextAttrib = (TextAttrib |  FOREGROUND_INTENSITY) &
      ~FOREGROUND_RED   & ~FOREGROUND_BLUE  |  FOREGROUND_GREEN;
  } else if ( color == SL_TTY::YELLOW ) {
    TextAttrib = (TextAttrib |  FOREGROUND_INTENSITY) &
      ~FOREGROUND_BLUE  |  FOREGROUND_RED   |  FOREGROUND_GREEN ;
  } else if ( color == SL_TTY::BLUE ) {
    TextAttrib = (TextAttrib |  FOREGROUND_INTENSITY) &
      ~FOREGROUND_RED   & ~FOREGROUND_GREEN |  FOREGROUND_BLUE;
  } else if ( color == SL_TTY::MAGENTA ) {
    TextAttrib = (TextAttrib |  FOREGROUND_INTENSITY) &
      ~FOREGROUND_GREEN |  FOREGROUND_RED   |  FOREGROUND_BLUE;
  } else if ( color == SL_TTY::CYAN ) {
    TextAttrib = (TextAttrib |  FOREGROUND_INTENSITY) &
      ~FOREGROUND_RED   |  FOREGROUND_GREEN |  FOREGROUND_BLUE;
  } else if ( color == SL_TTY::WHITE ) {
    TextAttrib = (TextAttrib |  FOREGROUND_INTENSITY) |
       FOREGROUND_RED   |  FOREGROUND_GREEN |  FOREGROUND_BLUE;
  } else  { return; }
  fg=color;
  SetConsoleTextAttribute( hStdout, TextAttrib );
}

void win32_console::term_set_bgcolor(SL_TTY::COLOR color){
  if( isReverse ){
    isReverse = false;
    SL_TTY::COLOR tmp;
    tmp = fg;
    term_set_fgcolor(bg);
    term_set_bgcolor(tmp);
  }
  if( color == SL_TTY::BLACK ){
    TextAttrib = (TextAttrib & ~BACKGROUND_INTENSITY) &
      ~BACKGROUND_RED   & ~BACKGROUND_BLUE  & ~BACKGROUND_GREEN;
  } else if ( color == SL_TTY::RED ) {
    TextAttrib = (TextAttrib |  BACKGROUND_INTENSITY) &
      ~BACKGROUND_GREEN & ~BACKGROUND_BLUE  |  BACKGROUND_RED;
  } else if ( color == SL_TTY::GREEN ) {
    TextAttrib = (TextAttrib |  BACKGROUND_INTENSITY) &
      ~BACKGROUND_RED   & ~BACKGROUND_BLUE  |  BACKGROUND_GREEN;
  } else if ( color == SL_TTY::YELLOW ) {
    TextAttrib = (TextAttrib |  BACKGROUND_INTENSITY) &
      ~BACKGROUND_BLUE  |  BACKGROUND_RED   |  BACKGROUND_GREEN ;
  } else if ( color == SL_TTY::BLUE ) {
    TextAttrib = (TextAttrib |  BACKGROUND_INTENSITY) &
      ~BACKGROUND_RED   & ~BACKGROUND_GREEN |  BACKGROUND_BLUE;
  } else if ( color == SL_TTY::MAGENTA ) {
    TextAttrib = (TextAttrib |  BACKGROUND_INTENSITY) &
      ~BACKGROUND_GREEN |  BACKGROUND_RED   |  BACKGROUND_BLUE;
  } else if ( color == SL_TTY::CYAN ) {
    TextAttrib = (TextAttrib |  BACKGROUND_INTENSITY) &
      ~BACKGROUND_RED   |  BACKGROUND_GREEN |  BACKGROUND_BLUE;
  } else if ( color == SL_TTY::WHITE ) {
    TextAttrib = (TextAttrib |  BACKGROUND_INTENSITY) |
       BACKGROUND_RED   |  BACKGROUND_GREEN |  BACKGROUND_BLUE;
  } else  { return; }
  bg=color;
  SetConsoleTextAttribute( hStdout, TextAttrib );
}

void win32_console::term_getyx(int *y, int *x){
  CONSOLE_SCREEN_BUFFER_INFO info;
  GetConsoleScreenBufferInfo( hStdout, &info );
  *y = (int)info.dwCursorPosition.Y;
  *x = (int)info.dwCursorPosition.X;
}

void win32_console::term_move(int y, int x){
  cInfo.dwCursorPosition.X = (SHORT)x;
  cInfo.dwCursorPosition.Y = (SHORT)y;
  SetConsoleCursorPosition( hStdout,cInfo.dwCursorPosition );
}

void win32_console::term_move_down(int y){
  int cx,cy,mx,my,to;
  term_getyx(&cy,&cx);
  term_getmaxyx(&my,&mx);
  to = (cy+y >= my)? my-1 : cy+y;
  term_move(to,cx);
}

void win32_console::term_move_up(int y){
  int cx,cy,to;
  term_getyx(&cy,&cx);
  to = (cy-y < 0)? 0 : cy-y;
  term_move(to,cx);
}

void win32_console::term_move_right(int x){
  int cx,cy,mx,my,to;
  term_getyx(&cy,&cx);
  term_getmaxyx(&my,&mx);
  to = (cx+x >= mx)? mx-1 : cx+x;
  term_move(cy,to);
}

void win32_console::term_move_left(int x){
  int cx,cy,to;
  term_getyx(&cy,&cx);
  to = (cx-x < 0)? 0 : cx-x;
  term_move(cy,to);
}

void win32_console::term_move_bol(){
  int cx,cy;
  term_getyx(&cy,&cx);
  term_move(cy,0);
}

void win32_console::term_move_newline(){
  term_putc('\r');
  term_putc('\n');
}

void win32_console::term_clear_screen(){
  int mx,my;
  COORD coord;
  DWORD noaw;
  coord.X=0; coord.Y=0;

  term_getmaxyx(&my,&mx);

  FillConsoleOutputCharacter(hStdout, ' ', mx*my, coord, &noaw);

  term_move(0, 0);
}

void win32_console::term_clear_eol(int x){
  int dis,i,mx,my;
  term_getmaxyx(&my,&mx);
  dis=mx-x;
  for(i=0;i<dis;i++) term_putc(' ');
  term_move_left(dis-1);
}

void win32_console::term_flush(sl4_fd_t fd){
  if(fd.handle != INVALID_HANDLE_VALUE)
    FlushFileBuffers(fd.handle);
}

void win32_console::term_getmaxyx(int *y, int *x){
  CONSOLE_SCREEN_BUFFER_INFO info;
  GetConsoleScreenBufferInfo( hStdout, &info );
  *y = (int)info.dwSize.Y;
  *x = (int)info.dwSize.X;
}

void win32_console::term_set_attr_normal(){
  if( isReverse ){
    SL_TTY::COLOR tmp;
    tmp=fg;
    term_set_fgcolor(bg);
    term_set_bgcolor(tmp);
    isReverse = false;
  }
  SetConsoleTextAttribute( hStdout, TextAttrib );
}

void win32_console::term_set_attr_reverse(){
  if( !isReverse ){
	SL_TTY::COLOR tmp;
	tmp=fg;
	term_set_fgcolor(bg);
	term_set_bgcolor(tmp);
	isReverse = true;
  }
  SetConsoleTextAttribute( hStdout, TextAttrib );
}

int win32_console::term_getc(){
  DWORD size;
  if(is_buffered){
    is_buffered = false;
    return (int)cbuffer[1];
  }
  ReadFile( hStdin, &cbuffer, 1, &size, 0);
  if(size == 2){
    is_buffered = true;
  }
  return (int)cbuffer[0];
}

char *win32_console::term_gets(char *buf, int c){
  DWORD size;
  ReadFile( hStdin, buf, c, &size, 0);
  buf[size]='\0';
  return buf;
}

void win32_console::term_putc(int c){
  unsigned char ch;
  DWORD size;
  ch = (unsigned char)c;
  WriteConsole( hStdout, &ch, 1, &size, 0 );
}

void win32_console::term_print(const char *str){
  DWORD size;
  WriteConsole( hStdout, str, (DWORD)strlen(str), &size, 0 );
}


int win32_console::term_keypad_getc(){
  int ret = -1;
  INPUT_RECORD buf;
  DWORD size;
  static bool ctrl_lock = false;
  if(is_buffered){
    is_buffered = false;
    return cbuffer[1];
  }
  while(1){
    if(ReadConsoleInput( hStdin, &buf, 1, &size ) == FALSE)
      return EOF; /* error */
    if(buf.EventType == KEY_EVENT){
      if(buf.Event.KeyEvent.bKeyDown){
        switch(buf.Event.KeyEvent.wVirtualKeyCode){
          case VK_UP:     ret = SL_TTY::UPKEY    << 8; break;
          case VK_DOWN:   ret = SL_TTY::DOWNKEY  << 8; break;
          case VK_RIGHT:  ret = SL_TTY::RIGHTKEY << 8; break;
          case VK_LEFT:   ret = SL_TTY::LEFTKEY  << 8; break;
          case VK_DELETE: ret = 0x7f; break;
          case VK_CONTROL: ctrl_lock = true; break;
          default: ret = (unsigned char)buf.Event.KeyEvent.uChar.AsciiChar;
        }
        if(ret == 0x20)
          if(ctrl_lock)
            ret = 0;
        if(ret != -1)
          break;
      }else{
        if(buf.Event.KeyEvent.wVirtualKeyCode == VK_CONTROL)
          ctrl_lock = false;
      }
    }
  }
  return ret;
}

