/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: terminal.c,v 1.2 2005/03/01 14:30:58 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#ifndef WIN32

#if defined(HAVE_CURSES_H) && defined(HAVE_TERM_H)
# include <curses.h>
# include <term.h>
#else
# ifdef HAVE_TERMCAP_H
#  include <termcap.h>
# endif
#endif
#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#include "terminal.h"

#define TC_ENV_SIZE 2048
#define TC_STR_IS_SET(x) (term.tcstr[(x)] != NULL)
#define TC_KEY_IS_SET(x) (term.tckey[(x)] != NULL)
#define TC_STR(x)         term.tcstr[(x)]
#define TC_KEY(x)         term.tckey[(x)]

/*
#define TC_co    0
#define TC_li    1
*/

#ifdef __cplusplus
extern "C" {
#endif

static struct term_keypad_t {
  char *key;
  int  code;
} term_keypad[] = {
  { "\x1b[A",  SL4_KEYPAD_UP    },
  { "\x1b[B",  SL4_KEYPAD_DOWN  },
  { "\x1b[C",  SL4_KEYPAD_RIGHT },
  { "\x1b[D",  SL4_KEYPAD_LEFT  },
  { "\x1b[H",  SL4_KEYPAD_HOME  },
  { "\x1b[F",  SL4_KEYPAD_END   },
  { "\x1bOA",  SL4_KEYPAD_UP    },
  { "\x1bOB",  SL4_KEYPAD_DOWN  },
  { "\x1bOC",  SL4_KEYPAD_RIGHT },
  { "\x1bOD",  SL4_KEYPAD_LEFT  },
  { "\x1bOH",  SL4_KEYPAD_HOME  },
  { "\x1bOF",  SL4_KEYPAD_END   } 
};

static char *term_val_names[]={
#define TC_km         0
  "km", /* Has Meta Key */
#define TC_MT         1
  "MT", /* Has Meta Key ? */
#define TC_VAL_SIZE   2
  NULL
};

static char *term_key_names[]={
#define TC_kd   0
  "kd", /* sends cursor down */
#define TC_kl   1
  "kl", /* sends cursor left */
#define TC_kr   2
  "kr", /* sends cursor right */
#define TC_ku   3
  "ku", /* sends cursor up */
#define TC_kh   4
  "kh", /* sends cursor home */
#define TC_at7  5
  "@7", /* sends cursor end */
#define TC_KEY_SIZE 6
  NULL
};

static char *term_str_names[]={
#define TC_cr    0
  "cr", /* carrige return */
#define TC_cl    1
  "cl", /* clear screen */
#define TC_ho    2
  "ho", /* home cursor */
#define TC_cd    3
  "cd", /* clear to bottom */
#define TC_ce    4
  "ce", /* clear to end of line */
#define TC_nw    5
  "nw", /* new line */
#define TC_mb    6
  "mb", /* begin blink mode */
#define TC_md    7
  "md", /* begin bold mode */
#define TC_se    8
  "se", /* end standout mode */
#define TC_so    9
  "so", /* begin standout mode */
#define TC_us   10
  "us", /* begin underline mode */
#define TC_ue   11
  "ue", /* end underline mode */
#define TC_me   12
  "me", /* end attributes */
#define TC_bl   13
  "bl", /* audible bell */
#define TC_vb   14
  "vb", /* visible bell */
#define TC_do   15
  "do", /* cursor down one */
#define TC_le   16
  "le", /* cursor left one */
#define TC_nd   17
  "nd", /* cursor right one (non destructive space) */
#define TC_up   18
  "up", /* cursor up one */
#define TC_ip   19
  "ip", /* insert padding */
#define TC_dm   20
  "dm", /* begin delete mode */
#define TC_ed   21
  "ed", /* end delete mode */
#define TC_im   22
  "im", /* begin insert mode */
#define TC_ei   23
  "ei", /* end insert mode */
#define TC_dc   24
  "dc", /* delete a character */
#define TC_ic   25
  "ic", /* insert a character */
#define TC_DC   26
  "DC", /* delete multiple chars */
#define TC_IC   27
  "IC", /* insert multiple chars */
#define TC_DO   28
  "DO", /* cursor down multiple */
#define TC_LE   29
  "LE", /* cursor left multiple */
#define TC_RI   30
  "RI", /* cursor right multiple */
#define TC_UP   31
  "UP", /* cursor up multiple */
#define TC_ve   32
  "ve", /* make cursor appear normal */
#define TC_vi   33
  "vi", /* make cursor invisible */
#define TC_STR_SIZE  34
  NULL
};

/*  "co", */ /* number of columns */
/*  "li", */ /* number of lines */

typedef struct _sl4_term_t {
  int ttyin, ttyout, ttyerr;
  FILE *fin, *fout, *ferr;
  /* tty information */
  struct termios tty_normal, tty_raw, tty_cook, tty_quote;
  char           tty_mode; /* TTY_MODE */
  /* terminal capability information */
  char          *env;
  char           tcap[TC_ENV_SIZE];
  char         **tcstr;
  char         **tckey;
  int           *tcval;
  /* terminal information */
  int            col, lin;
  /* sigwinch */
  int            got_sigwinch;
} sl4_term_t;

static sl4_term_t term;
static void   sigwinch_handler(int signo);
static int   _sl4_tty_init();
static int   _sl4_tty_quit();
static int   _sl4_term_refresh_wsize();
static int   _sl4_tcstr_set(int n, const char *str);
static int   _sl4_tckey_set(int n, const char *str);

static void sigwinch_handler(int signo){
  term.got_sigwinch = 1;
}

static int _sl4_tty_init(){
  /* get current terminal capability attributes */
  tcgetattr(term.ttyin, &term.tty_normal);
  term.tty_raw = term.tty_normal;
  /* make raw mode -  cfmakeraw(&tty_raw); */
  term.tty_raw.c_iflag &= ~(BRKINT|ISTRIP|INLCR|IGNCR|ICRNL|IXON|IXOFF);
  term.tty_raw.c_oflag &= ~OPOST;
  term.tty_raw.c_lflag &= ~(ICANON|IEXTEN|ECHO|ECHOE|ECHOK|ECHONL|ISIG);
  term.tty_raw.c_cc[VMIN]  = 1;
  term.tty_raw.c_cc[VTIME] = 0;
  /* make cooked mode */
  /* TODO */
  term.tty_cook = term.tty_normal;
  /* make quoted mode */
  /* TODO */
  term.tty_quote = term.tty_raw;
  /* set now mode */
  term.tty_mode = TTY_MODE_NORMAL;
  return 0;
}

static int _sl4_tty_quit(){
  sl4_tty_setmode(TTY_MODE_NORMAL);
  return 0;
}

static int _sl4_term_refresh_wsize(){
#ifdef TIOCGWINSZ
  struct winsize ws;
#endif /* TIOCGWINSZ */
#ifdef TIOCGSIZE
  struct ttysize ts;
#endif /* TIOCGSIZE */
#ifdef TIOCGWINSZ
  if (ioctl(0, TIOCGWINSZ, &ws) != -1) {
    if(ws.ws_col) term.col = ws.ws_col;
    if(ws.ws_row) term.lin = ws.ws_row;
  }
#endif /* TIOCGWINSZ */
#ifdef TIOCGSIZE
  if(ioctl(0, TIOCGSIZE, &ts) != -1) {
    if(ts.ts_cols)  term.col = ts.ts_cols;
    if(ts.ts_lines) term.lin = ts.ts_lines;
  }
#endif /* TIOCGSIZE */
  return 0;
}

static int _sl4_tcstr_set(int n, const char *str){
  if(term.tcstr[n] != NULL){
    free(term.tcstr[n]);
    term.tcstr[n] = NULL;
  }
  if(str != NULL && *str != '\0'){
    term.tcstr[n] = (char*)malloc(sizeof(char)*(strlen(str)+1));
    if(term.tcstr[n] == NULL)
      return -1;
    strcpy(term.tcstr[n], str); /* safe */
  }
  return 0;
}

static int _sl4_tckey_set(int n, const char *str){
  if(term.tckey[n] != NULL){
    free(term.tckey[n]);
    term.tckey[n] = NULL;
  }
  if(str != NULL && *str != '\0'){
    term.tckey[n] = (char*)malloc(sizeof(char)*(strlen(str)+1));
    if(term.tckey[n] == NULL)
      return -1;
    strcpy(term.tckey[n], str); /* safe */
  }
  return 0;
}


int sl4_tty_setmode(int mode){
  int ret = 0;
  switch(mode){
  case TTY_MODE_NORMAL:
    if(term.tty_mode != TTY_MODE_NORMAL){
      tcsetattr(0,TCSADRAIN,&term.tty_normal);
      term.tty_mode = TTY_MODE_NORMAL;
    }
    break;
  case TTY_MODE_RAW:
    if(term.tty_mode != TTY_MODE_RAW){
      tcsetattr(0,TCSADRAIN,&term.tty_raw);
      term.tty_mode = TTY_MODE_RAW;
    }
    break;
  case TTY_MODE_COOK:
    if(term.tty_mode != TTY_MODE_COOK){
      tcsetattr(0,TCSADRAIN,&term.tty_cook);
      term.tty_mode = TTY_MODE_COOK;
    }
    break;
  default:
    ret = -1;
  }
  return ret;
}

int sl4_term_init(FILE *fin, FILE *fout, FILE *ferr){
  /* term */
  int  i,ent;
  char *area,*tmp;
  char buf[TC_ENV_SIZE];
  sigset_t oset, nset;
  struct sigaction sigact;

  /* set file pointer */
  term.fin  = fin;
  term.fout = fout;
  term.ferr = ferr;
  /* set file discripter */
  term.ttyin  = fileno(fin);
  term.ttyout = fileno(fout);
  term.ttyerr = fileno(ferr);

  /* initialize tty */
  _sl4_tty_init();

  /* block SIGWINCH signal */
  sigemptyset(&nset);
  sigaddset(&nset, SIGWINCH);
  sigprocmask(SIG_BLOCK, &nset, &oset);

  area = buf;
  tmp = getenv("TERM");
  if(tmp == NULL || *tmp == '\0')
    tmp = "dumb";
  term.env = (char*)malloc(sizeof(char)*(strlen(tmp)+1));
  if(term.env){
    strcpy(term.env, tmp); /* safe */
  }

  memset(term.tcap, 0, TC_ENV_SIZE);
  term.tcstr = (char**)malloc(sizeof(char*)*TC_STR_SIZE);
  if(term.tcstr)
    for(i=0;i<TC_STR_SIZE;i++) term.tcstr[i] = NULL;
  term.tckey = (char**)malloc(sizeof(char*)*TC_KEY_SIZE);
  if(term.tckey)
    for(i=0;i<TC_KEY_SIZE;i++) term.tckey[i] = NULL;
  term.tcval = (int*)malloc(sizeof(int)*TC_VAL_SIZE);
  if(term.tcval)
    for(i=0;i<TC_VAL_SIZE;i++) term.tcval[i] = 0;

  /* memory allocation error */
  if(term.env == NULL || term.tcval == NULL || 
     term.tcstr == NULL || term.tckey == NULL){
    sigprocmask(SIG_SETMASK, &oset, NULL);
    return -1;
  }

  ent = tgetent(term.tcap, term.env);
  if(ent<=0){
    if(ent==-1)
      fprintf(term.ferr,"Cannot read termcap database;\n");
    else if(ent==0)
      fprintf(term.ferr,"No entry for terminal type \"%s\";\n",term.env);
    fprintf(term.ferr,"using dumb terminal settings.\n");
    /* dumb terminal */
    term.col = 80;
    term.lin = 0;
  } else {
    term.col = tgetnum("co");
    term.lin = tgetnum("li");
    for(i=0; term_val_names[i]!=NULL; i++)
      term.tcval[i] = tgetnum(term_val_names[i]);
    for(i=0; term_key_names[i]!=NULL; i++){
      tmp = tgetstr(term_key_names[i], &area);
      if(tmp != 0)
        _sl4_tckey_set(i, tmp);
    }
    for(i=0; term_str_names[i]!=NULL; i++){
      tmp = tgetstr(term_str_names[i], &area);
      if(tmp != 0)
        _sl4_tcstr_set(i, tmp);
    }
  }
  if(term.col < 2) term.col = 80;
  if(term.lin < 1) term.lin = 24;

  term.got_sigwinch = 0;
  _sl4_term_refresh_wsize();
  /* restore SIGWINCH signal */
  sigprocmask(SIG_SETMASK, &oset, NULL);

  /* register SIGWINCH event hander */
  sigact.sa_handler = sigwinch_handler;
  sigemptyset(&sigact.sa_mask);
  sigact.sa_flags = SA_RESTART;
  sigaction(SIGWINCH, &sigact, NULL);
  return 0;
}

int sl4_term_quit(){
  int i;
  _sl4_tty_quit();
  free(term.env);   term.env = NULL;
  for(i=0;i<TC_STR_SIZE;i++) _sl4_tcstr_set(i, NULL);
  free(term.tcstr); term.tcstr = NULL;
  for(i=0;i<TC_KEY_SIZE;i++) _sl4_tckey_set(i, NULL);
  free(term.tckey); term.tckey = NULL;
  /* for(i=0;i<TC_VAL_SIZE;i++) term.tcval[i] = 0; */
  free(term.tcval); term.tcval = NULL;
  return 0;
}

int sl4_term_putc(int c){
  return fputc(c, term.fout);
}

int sl4_term_puts(const char *str){
  return fputs(str, term.fout);
}

int sl4_term_getc(){
  return fgetc(term.fin);
}

int sl4_term_keypad_getc(){
  int i,j,ret,ch,ch2,kp,n;
  const char *p;
  char buf[128];
  ch = fgetc(term.fin);
  ret = ch;
  for(i=0;i<12;i++){
    kp = -1; n = 0;
    p = term_keypad[i].key;
    if(*p == ch){
      for(p++; *p!='\0'; p++){
        ch2 = fgetc(term.fin);
        buf[n++] = ch2;
        if(*p != ch2 || n == 128){
          for(j=n-1;j>=0;j--) ungetc(buf[j], term.fin);
          break;
        }
      }
      if(*p=='\0') kp=term_keypad[i].code;
    }
    if(kp!=-1){ ret = kp; break; }
  }
  if(kp==-1){
    for(i=0; i<TC_KEY_SIZE; i++){
      n=0;
      if(TC_KEY_IS_SET(i)) p=TC_KEY(TC_kd);
      else continue;
      if(*p == ch){
        for(p++; *p!='\0'; p++){
          ch2 = fgetc(term.fin);
          buf[n++] = ch2;
          if(*p != ch2 || n == 128){
            for(j=n-1; j>=0; j--) ungetc(buf[j], term.fin);
            break;
          }
        }
        if(*p=='\0'){
          switch(i){
          case TC_kd:  kp=SL4_KEYPAD_DOWN;  break;
          case TC_kl:  kp=SL4_KEYPAD_LEFT;  break;
          case TC_kr:  kp=SL4_KEYPAD_RIGHT; break;
          case TC_ku:  kp=SL4_KEYPAD_UP;    break;
          case TC_kh:  kp=SL4_KEYPAD_HOME;  break;
          case TC_at7: kp=SL4_KEYPAD_END;   break;
          }
        }
      }
      if(kp!=-1){ ret = kp; break; }
    }
  }
  return ret;
}

char *sl4_term_gets(char *buf, int size){
  return fgets(buf, size, term.fin);
}

int sl4_term_cursor_up(int y){
  int i;
  if(y <= 0) return -1;
  if(TC_STR_IS_SET(TC_UP)){
    tputs(tgoto(TC_STR(TC_UP),y,y), y, sl4_term_putc);
  }else if(TC_STR_IS_SET(TC_up)){
    for(i=0;i<y;i++)
      tputs(TC_STR(TC_up), 1, sl4_term_putc);
  }
  return 0;
}

int sl4_term_cursor_down(int y){
  int i;
  if(y <= 0) return -1;
  if(TC_STR_IS_SET(TC_DO)){
    tputs(tgoto(TC_STR(TC_DO),y,y), y, sl4_term_putc);
  }else if(TC_STR_IS_SET(TC_do)){
    for(i=0;i<y;i++)
      tputs(TC_STR(TC_do), 1, sl4_term_putc);
  }
  return 0;
}

int sl4_term_cursor_right(int x){
  int i;
  if(x <= 0) return -1;
  if(TC_STR_IS_SET(TC_RI)){
    tputs(tgoto(TC_STR(TC_RI),x,x), x, sl4_term_putc);
  }else if(TC_STR_IS_SET(TC_nd)){
    for(i=0;i<x;i++)
      tputs(TC_STR(TC_nd), 1, sl4_term_putc);
  }
  return 0;
}

int sl4_term_cursor_left(int x){
  int i;
  if(x <= 0) return -1;
  if(TC_STR_IS_SET(TC_LE)){
    tputs(tgoto(TC_STR(TC_LE),x,x), x, sl4_term_putc);
  }else if(TC_STR_IS_SET(TC_le)){
    for(i=0;i<x;i++)
      tputs(TC_STR(TC_le), 1, sl4_term_putc);
  }
  return 0;
}

int sl4_term_cursor_bol(){
  if(TC_STR_IS_SET(TC_cr)){
    tputs(TC_STR(TC_cr), 1, sl4_term_putc);
  }else{
    sl4_term_putc('\r');
  }
  return 0;
}

int sl4_term_cursor_newline(){
  sl4_term_cursor_bol();
  sl4_term_cursor_right(term.col-1);
  if(TC_STR_IS_SET(TC_nw)){
    tputs(TC_STR(TC_nw), 1, sl4_term_putc);
  }else{
    sl4_term_putc('\r');
    sl4_term_putc('\n');
  }
  return 0;
}

int sl4_term_cursor_invisible(){
  if(TC_STR_IS_SET(TC_vi)){
    tputs(TC_STR(TC_vi), 1, sl4_term_putc);
  }
  return 0;
}

int sl4_term_cursor_normal(){
  if(TC_STR_IS_SET(TC_ve)){
    tputs(TC_STR(TC_ve), 1, sl4_term_putc);
  }
  return 0;
}

int sl4_term_bell(){
  if(TC_STR_IS_SET(TC_bl)){
    tputs(TC_STR(TC_bl), 1, sl4_term_putc);
  }
  return 0;
}

int sl4_term_vbell(){
  if(TC_STR_IS_SET(TC_vb)){
    tputs(TC_STR(TC_vb), 1, sl4_term_putc);
  }
  return 0;
}

int sl4_term_attr_normal(){
  char se,ue,me;
  se=ue=me=0;
  if(TC_STR_IS_SET(TC_se)) se=1;
  if(TC_STR_IS_SET(TC_ue)) ue=1;
  if(TC_STR_IS_SET(TC_me)) me=1;
  if(se && me) if(!strcmp(TC_STR(TC_se), TC_STR(TC_me))) se=0;
  if(ue && me) if(!strcmp(TC_STR(TC_ue), TC_STR(TC_me))) ue=0;
  if(se) tputs(TC_STR(TC_se), 1, sl4_term_putc);
  if(ue) tputs(TC_STR(TC_ue), 1, sl4_term_putc);
  if(me) tputs(TC_STR(TC_me), 1, sl4_term_putc);
  return 0;
}

int sl4_term_attr_bold(){
  if(TC_STR_IS_SET(TC_md))
    tputs(TC_STR(TC_md), 1, sl4_term_putc);
  return 0;
}

int sl4_term_attr_underline(){
  if(TC_STR_IS_SET(TC_us))
    tputs(TC_STR(TC_us), 1, sl4_term_putc);
  return 0;
}

int sl4_term_attr_reverse(){
  if(TC_STR_IS_SET(TC_so))
    tputs(TC_STR(TC_so), 1, sl4_term_putc);
  return 0;
}

int sl4_term_clear_screen(){
  if(TC_STR_IS_SET(TC_cl)){
    tputs(TC_STR(TC_cl), term.lin, sl4_term_putc);
  }else if(TC_STR_IS_SET(TC_ho) && (TC_STR_IS_SET(TC_cd))){
    tputs(TC_STR(TC_ho), term.lin, sl4_term_putc);
    tputs(TC_STR(TC_cd), term.lin, sl4_term_putc);
  }else{
    sl4_term_putc('\r');
    sl4_term_putc('\n');
  }
  return 0;
}

int sl4_term_clear_eol(int xpos){
  int i,dis;
  if(TC_STR_IS_SET(TC_ce)){
    tputs(TC_STR(TC_ce), 1, sl4_term_putc);
  }else{
    dis=term.col - xpos;
    for(i=0;i<dis;i++) sl4_term_putc(' ');
    sl4_term_cursor_left(dis-1);
  }
  return 0;
}

int sl4_term_getmaxyx(int *y, int *x){
  if(term.got_sigwinch){
    _sl4_term_refresh_wsize();
    term.got_sigwinch = 0;
  }
  if(y != NULL) *y=term.lin;
  if(x != NULL) *x=term.col;
  return 0;
}


#ifdef __cplusplus
}
#endif

#endif /* WIN32 */

