/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: libusr.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/********************************************************************
**  There are Library functions that are used by "USR" Model(C)    **
**  *errorFunc  : Error Function for Least Square                  **
**  *dataStore  : stores C Model Outputs                           **
**                (called by storeParamHist)                       **
********************************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include <math.h>

#include "libsatellite.h"

#include "libnpec.h"
#include "libnpee.h"

#ifdef __cplusplus
extern "C" {
#endif

/* User defined Model Function */
#ifdef WIN32
typedef int    (*model_func_t)(int, double*, double*);
typedef double (*penalty_func_t)(double*);

extern DLLIMPORT model_func_t   model;
extern DLLIMPORT penalty_func_t penalty;
DLLIMPORT void open_dlls();
DLLIMPORT void close_dlls();
#else
int    model(int t, double *param, double *output);
double penalty(double *param);
#endif

/* private functions */
static double err_norm_one(npe_param_t *AllParam, double *disp);
static double err_norm_two(npe_param_t *AllParam, double *disp);
static double err_norm_inf(npe_param_t *AllParam, double *disp);

double  errorFunc_user(npe_param_t *AllParam);
void    dataStore_user(npe_param_t *Param);
int     readParam_user();

/*******************************************************************
**  Main Function                                                **
********************************************************************/
int main(){
  int  code;

  load_syscom_file();

  code = npe_estimator_init();
  if(code != 0) return code;

  /* set functions */
#ifdef WIN32
  open_dlls();
#endif
  npe_errorFunc     = errorFunc_user;
  npe_dataStore     = dataStore_user;
  npe_readParam     = readParam_user;
  /* optimization method call */
  code = npe_optimazation();
#ifdef WIN32
  close_dlls();
#endif
  if(code != NULL)
    return code;
  return npe_estimator_final();
}
  
/*******************************************************************
**  Error Function                                                **
********************************************************************/
double  errorFunc_user(npe_param_t  *AllParam)
{
  double  error, disp;

  switch(npe_common->norm_type){
  case NPE_NORM_ONE: error = err_norm_one(AllParam, &disp); break;
  case NPE_NORM_TWO: error = err_norm_two(AllParam, &disp); break;
  case NPE_NORM_INF: error = err_norm_inf(AllParam, &disp); break;
  default:           error = err_norm_two(AllParam, &disp); break;
  }

  if(npe_common->display_type == NPE_DISPLAY_NO_WEIGHT)
    tmpDisp = disp;
  else
    tmpDisp = error;

  tmpPena = penalty(ModelParam);
  error += tmpPena;
  return error;
}


static double err_norm_two(npe_param_t *AllParam, double *disp)
{
  double  error = 0.0, tmp;
  unsigned int i;
  int t, k;

  *disp = 0.0;
  for(i=0; i < npe_common->init_paramnum; i++)
    ModelParam[i] = getValue(AllParam[i]);

  for(t=0; t < npe_common->point_value; t++) {
    model(t, ModelParam, Output);

    for(k=0; k < npe_common->number_value; k++) {
      tmp = WaveData[k][t]-Output[k];
      error += tmp * tmp * WeightData[k][t];
      *disp += tmp * tmp;
    }
  }
  return error;
}


static double err_norm_one(npe_param_t *AllParam, double *disp)
{
  double  error = 0.0, tmp;
  unsigned int i;
  int t, k;

  *disp = 0.0;
  for(i=0; i < npe_common->init_paramnum; i++)
    ModelParam[i] = getValue(AllParam[i]);

  for(t=0; t < npe_common->point_value; t++) {
    model(t, ModelParam, Output);
    for(k=0; k < npe_common->number_value; k++) {
      tmp = fabs(WaveData[k][t]-Output[k]);
      error += tmp * WeightData[k][t];
      *disp += tmp;
    }
  }
  return error;
}


static double err_norm_inf(npe_param_t *AllParam, double *disp)
{
  double  error = 0.0,tmp, tmperr;
  unsigned int i;
  int t, k;

  *disp = 0.0;
  for(i=0; i < npe_common->init_paramnum; i++)
    ModelParam[i] = getValue(AllParam[i]);

  for(t=0; t < npe_common->point_value; t++) {
    model(t, ModelParam, Output);
    for(k=0; k < npe_common->number_value; k++) {
      tmp = fabs(WaveData[k][t]-Output[k]);
      tmperr = tmp * WeightData[k][t];
      if(error < tmperr){
        error = tmperr;
        *disp = tmp;
      }
    }
  }
  return error;
}

/*******************************************************************
**  'dataStore()' stores Model Output                             **
********************************************************************/
void  dataStore_user(npe_param_t  *Param){
  unsigned int i;
  int     t, k, dpt;
  int     Dimension, local_index[10];
  double  *output;

  Dimension = 2;
  local_index[0] = npe_common->point_value;
  local_index[1] = npe_common->number_value;

  output = (double*)malloc(sizeof(double)*IndexSize(Dimension, local_index));
  if(output == NULL)
    exit(101); /* out of memory */

 /*
  ** calculation of model output
  */
  for(i = 0; i < npe_common->init_paramnum; i++){
      ModelParam[i] = (double)getValue(Param[i]);
  }

  dpt = 0;
  for(t = 0; t < npe_common->point_value; t++){
    model(t, ModelParam, Output);
    for(k = 0; k < npe_common->number_value; k++){
      output[dpt+k] = (double)Output[k];
    }
    dpt += npe_common->number_value;
  }

  /* must be 8 byte blocksize */
  _WriteFile4(npe_common->result_file, Dimension, local_index, (char*)output);
  free(output);
}


/******************************************************************
**  'readParam()' reads Initial Parameter                        **
******************************************************************/
int  readParam_user(){
  unsigned int i, j;
  npe_estparam_t *e;

  if(npe_common->init_paramnum > UPPER_LIMIT)
    exit(104); /* too meny data */

  /*
  **  allocating array
  */
  AllParam   = (npe_param_t*)malloc2Dim(0, npe_common->init_paramnum, 'P');
  ModelParam = (double*)malloc2Dim(0, npe_common->init_paramnum, 'D');

  /*
  **  copying each Parameters
  */
  NumVarParam = 0;
  for(i=0,e=npe_common->estparam; i<npe_common->init_paramnum; i++,e++){
    AllParam[i].value = e->value;
    AllParam[i].flag  = e->flag;
    if(AllParam[i].flag == NPE_PARAM_FLAG_VAR)
      NumVarParam++;
    AllParam[i].scale = 1.0;
    AllParam[i].span  = e->span;
  }
  
  /*
  **  setting Variable Parameters
  */
  VarParam = (npe_param_t**)malloc(sizeof(npe_param_t*)*NumVarParam);
  if(VarParam == NULL)
    exit(101); /* out of memory */

  for(i=j=0; i<npe_common->init_paramnum; i++){
    if(AllParam[i].flag == NPE_PARAM_FLAG_VAR){
      VarParam[j++] = &AllParam[i];
    }
  }
  return 0;
}

#ifdef __cplusplus
}
#endif

