/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: libncs.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/*******************************************************************
**  There are Library functions that are used by "NCS" model      **
**	*errorFunc	: Error Function for NCS model            **
**	*dataStore	: stores NCS Model Outputs                **
**			  (called by storeParamHist)              **
**	*readParam	: reads Parameters for NCS Model          **
********************************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>

#include "libsatellite.h"

#include "libncsc.h"
#include "libncss.h"
#include "ncsmodel.h"

#include "libnpec.h"
#include "libnpee.h"

#include "ncswrap.h"

#ifdef __cplusplus
extern "C" {
#endif

/* external function */
#ifdef WIN32
typedef double (*penalty_func_t)(double*);

extern DLLIMPORT penalty_func_t penalty;
DLLIMPORT void open_dlls();
DLLIMPORT void close_dlls();
#else
double penalty(double *param);
#endif

/* private functions */
static double err_norm_one(double *disp);
static double err_norm_two(double *disp);
static double err_norm_inf(double *disp);

double  errorFunc_ncs(npe_param_t *AllParam);
void    dataStore_ncs(npe_param_t *Param);
int     readParam_ncs();

/* global vars in this file */
double   **NcsParamPtr;

/*******************************************************************
**  Main Function                                                **
********************************************************************/
int main(){
  int  code;
  
  load_syscom_file();

  code = npe_estimator_init();
  if(code != 0) return code;

  /* set functions */
#ifdef WIN32
  open_dlls();
#endif

  npe_errorFunc = errorFunc_ncs;
  npe_dataStore = dataStore_ncs;
  npe_readParam = readParam_ncs;

  /* call optimazation method */
  code = npe_optimazation();

#ifdef WIN32
  close_dlls();
#endif

  if(code != 0)
    return code;

  return npe_estimator_final();
}

/*******************************************************************
**  Error Function                                                **
********************************************************************/
static double err_norm_one(double *disp)
{
  double error = 0.0, tmp;
  int i,t;

  *disp = 0.0;
  for(t=0; t < npe_common->point_value; t++){
    for(i=0; i < npe_common->number_value; i++){
      tmp = fabs(WaveData[i][t]-NcsOutput[i][t]);
      error += tmp * WeightData[i][t];
      *disp += tmp;
    }
  }
  return(error);
}


static double err_norm_two(double *disp)
{
  double error = 0.0, tmp;
  int i,t;

  *disp = 0.0;
  for(t=0; t < npe_common->point_value; t++){
    for(i=0; i < npe_common->number_value; i++){
      tmp = fabs(WaveData[i][t]-NcsOutput[i][t]);
      error += tmp*tmp * WeightData[i][t];
      *disp += tmp*tmp;
    }
  }
  return(error);
}


static double err_norm_inf(double *disp)
{
  double error = 0.0, tmp, tmperr;
  int i,t;

  *disp = 0.0;
  for(t=0; t < npe_common->point_value; t++){
    for(i=0; i < npe_common->number_value; i++){
      tmp = fabs(WaveData[i][t]-NcsOutput[i][t]);
      tmperr = tmp * WeightData[i][t];
      if(error < tmperr){
        error = tmperr;
        *disp = tmp;
      }
    }
  }
  return(error);
}


double  errorFunc_ncs(npe_param_t  *AllParam)
{
  double  error, disp;
  unsigned int  i;

  for(i=0 ; i < npe_common->init_paramnum ; i++){
    ModelParam[i] = *NcsParamPtr[i] = getValue(AllParam[i]);
  }
  ncsxstart1();

  switch(npe_common->norm_type){
  case NPE_NORM_ONE: error = err_norm_one(&disp); break;
  case NPE_NORM_TWO: error = err_norm_two(&disp); break;
  case NPE_NORM_INF: error = err_norm_inf(&disp); break;
  default:           error = err_norm_two(&disp); break;
  }

  if(npe_common->display_type == NPE_DISPLAY_NO_WEIGHT)
    tmpDisp = disp;
  else
    tmpDisp = error;

  tmpPena = penalty(ModelParam);
  error += tmpPena;
  return(error);
}

/*******************************************************************
**  'dataStore()' stores Model Output                             **
********************************************************************/
void  dataStore_ncs(npe_param_t  *Param){
  static double **output = NULL;
  double *outbuf;
  unsigned int i;
  int t, k;
  int     local_index[MAX_INDEX], local_index2[MAX_INDEX];

  if(output == NULL)
    output = (double**)malloc2Dim(npe_common->number_value, 
                                  npe_common->point_value, 'D');
  /*
  ** calculation of model output
  */
  for(i = 0; i < npe_common->init_paramnum; i++)
    *NcsParamPtr[i] = getValue(AllParam[i]);
  
  ncsxstart1();
  
  for(t=0; t < npe_common->point_value; t++)
    for(k=0; k < npe_common->number_value; k++)
      output[k][t] = NcsOutput[k][t];
  
  local_index[0] = npe_common->number_value;
  local_index[1] = npe_common->point_value;
  outbuf = (double*)malloc(sizeof(double) * IndexSize(2, local_index));
  if(outbuf == NULL){
    exit(101); /* out of memory */
  }
  for(local_index2[0]=0;
      local_index2[0] < npe_common->number_value; 
      local_index2[0]++){
    for(local_index2[1]=0;
        local_index2[1] < npe_common->point_value;
        local_index2[1]++){
      outbuf[_Index(local_index2,2,local_index)] 
        = output[local_index2[0]][local_index2[1]];
    }
  }
  /* must be 8 byte blocksize */
  _WriteFile4(npe_common->result_file, 2, local_index, (char*)outbuf);
  free(outbuf);
}

/*******************************************************************
**  'readParam()' reads Initial Parameter                         **
********************************************************************/
int  readParam_ncs(){
  unsigned int i,j;
  char *tail;
  char *cp, str2[NPE_MAXPARAMNAME], str3[NPE_MAXPARAMNAME];
  npe_estparam_t *e;

  ncsReadConditionFile();

  if(npe_common->point_value < data_point){
    fprintf(stderr, "Error: DataPoint is Wrong.  NPE: %d <--> NCS: %d\n",
      npe_common->point_value, data_point);
    
    fprintf(stderr,"Abort\n");
    exit(102); /* fatal error */
  }

  if(npe_common->init_paramnum > UPPER_LIMIT){
    exit(104); /* too many data */
  }

  /*
  **  allocating array
  */
  AllParam    = (npe_param_t*)malloc2Dim(0, npe_common->init_paramnum, 'P');
  ModelParam  = (double *)    malloc2Dim(0, npe_common->init_paramnum, 'D');
  NcsOutput   = (double**)    malloc2Dim(npe_common->number_value, 
                                         npe_common->point_value, 'D');
  NcsParamPtr = (double **)malloc(sizeof(double*) * npe_common->init_paramnum);
  if(NcsParamPtr == NULL)
    exit(101); /* out of memory */

  /*
  **  reading each Parameters
  */
  NumVarParam = 0;
  for(i=0,e=npe_common->estparam; i<npe_common->init_paramnum; i++,e++){ 
    AllParam[i].value = e->value;
    AllParam[i].flag  = e->flag;
    if(AllParam[i].flag == NPE_PARAM_FLAG_VAR)
      NumVarParam++;
    AllParam[i].scale = 1.0;
    AllParam[i].span  = e->span;
  
    /*
    ** Pointer of Blockdata (SigmaParamPtr)
    */

    cp = strutil_strtok_r(e->name, "@\t ", &tail);
    strcpy(str3, cp);
    cp = strutil_strtok_r(NULL, "\n", &tail);
    strcpy(str2, cp);

    if((NcsParamPtr[i] = GetPrmPtr(str2, str3)) == NULL)
      exit(112); /* can't get parameter of NCS model */

    /* replace parameter name */
    sprintf(npe_common->estparam[i].name, "%s %s", str2, str3); /* TODO check! */
  }

  /*
  **  setting Variable Parameters
  */
  VarParam = (npe_param_t **)malloc(sizeof(npe_param_t*)*NumVarParam);
  if(VarParam == NULL)
    exit(101); /* out of memory */

  for(i=j=0; i<npe_common->init_paramnum; i++){
    if(AllParam[i].flag == NPE_PARAM_FLAG_VAR){
      VarParam[j++] = &AllParam[i];
    }
  }

 return(0);
}

#ifdef __cplusplus
}
#endif
