/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: golden.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/**************************************************
**  golden.cpp                                   **
**  linear search method                         **
**************************************************/
#define  MOD_NPE_LIBNPEE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "libsatellite.h"
#include "libnpec.h"
#include "libnpee.h"
#include "npeepriv.h"

#ifdef __cplusplus
extern "C" {
#endif

#define  GOLD    1.618034
#define  GLIMIT  100.0
#define  TINY    1.0e-20
#define  R       0.61803399
#define  C       (1.0 - R)
#define  Max_It  100
#define  TOL     1.0e-12
#define  ZEPS    1.0e-10

/* function prototype */
static void  bracketing(double *ax, double *bx, double *cx, double *Xk);

/* function body */
void  npe_lsearch_golden(){
  int  i, iteration=0;
  double  x0, x1, x2, x3, f1, f2,ftmp;
  double  ax, bx, cx;
  double  *Xk;

  Xk = (double*)malloc2Dim(0, NumVarParam, 'D');

  ax = 0.0;
  bx = npe_common->lsearch_value;
  for(i=0;i<(int)NumVarParam;i++)  Xk[i] = VarParam[i]->value;

  bracketing(&ax,&bx,&cx,Xk);

  x0 = ax;
  x3 = cx;
  if(fabs(cx - bx) > fabs(bx - ax)){
    x1 = bx;
    x2 = bx + C*(cx - bx);  /* new point */
  } else {
    x2 = bx;
    x1 = bx - C*(bx - ax);
  }

  for (i=0;i<(int)NumVarParam;i++)
    VarParam[i]->value = Xk[i] + x1*Direction[i];
  f1 = npe_errorFunc( AllParam );

  for (i=0;i<(int)NumVarParam;i++)
    VarParam[i]->value = Xk[i] + x2*Direction[i];
  f2 = npe_errorFunc( AllParam );

  while( fabs(x3 - x0) > TOL*( fabs(x1)+fabs(x2) )/2.0 + ZEPS){
    printf("."); fflush( stdout );
    if( f2 < f1 ){
      x0 = x1; x1 = x2; x2 = R*x1 + C*x3;

      for (i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + x2*Direction[i];
        ftmp = npe_errorFunc( AllParam );
      f1 = f2; f2 = ftmp;
    } else {
      x3 = x2; x2 = x1; x1 = R*x2+C*x0;

      for (i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + x1*Direction[i];
      ftmp = npe_errorFunc( AllParam );
      f2 = f1; f1 = ftmp;
    }
    if(iteration++ >= Max_It){
      printf("\nWarrning: Over Max iteration[%d] ... why ?\n", Max_It);
      break;
    }
  }
  printf("\n");

  if(f1 < f2) {
    for (i=0;i<(int)NumVarParam;i++)
      VarParam[i]->value = Xk[i] + x1*Direction[i];
     ErrorValue = f1;
  } else {
    for (i=0;i<(int)NumVarParam;i++)
      VarParam[i]->value = Xk[i] + x2*Direction[i];
     ErrorValue = f2;
  }
  free2Dim(Xk, 0, 'D');
} 


static void bracketing(double *ax, double *bx, double *cx, double *Xk)
{
  int  i;
  double  fa, fb, fc;
  double  ulim, u,r,q,fu,dum,tmp;


  for(i=0;i<(int)NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*ax)*Direction[i];
  fa = npe_errorFunc( AllParam );  /* fa = func(ax) */

  for(i=0;i<(int)NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*bx)*Direction[i];
  fb = npe_errorFunc( AllParam ); /* fb = func(bx) */

  if( fb > fa ){
      dum = *ax;  *ax = *bx;
      *bx = dum;
      dum = fb; fb = fa;
      fa = dum;  
  }

  *cx = (*bx) + GOLD*(*bx - *ax); /* initial estimated value of cx */

  for(i=0;i<(int)NumVarParam;i++)
    VarParam[i]->value = Xk[i] + (*cx)*Direction[i];
  fc = npe_errorFunc( AllParam );     /* fc = func(cx) */

  while( fb > fc ){ /* iterate until enclose */

    /* cal. u from a,b,c */
    r = (*bx - *ax)*(fb - fc);
    q = (*bx - *cx)*(fb - fa);
    if(fabs(q-r) > TINY) tmp = fabs(q-r);
    else tmp = TINY;
    if( q-r > 0.0 ) dum = fabs(tmp);
    else dum = -fabs(tmp);
    dum *= 2.0;
    u = (*bx) - ((*bx - *cx)*q - (*bx - *ax)*r)/dum;
    /* avoid to divide by zero */
    ulim = (*bx) + GLIMIT*(*cx - *bx);
    if( (*bx - u)*(u - *cx) > 0.0 ){ /* b < u < c */
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );    /* fu = func(u) */
      if( fu < fc ){          /* found a minimum value between b and c  */
        *ax = (*bx);
        *bx = u;
        fa = fb;
        fb = fu;
        return;
      } else if( fu > fb ) {  /* found a minimum value between a and u  */
        *cx = u;
        fc = fu;
        return;
      }
      u = (*cx) + GOLD*(*cx - *bx);
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );
    } else if((*cx - u)*(u - ulim) > 0.0){
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );

      if( fu < fc ){
        *bx = *cx; *cx = u;
        u = (*cx + GOLD*(*cx - *bx));

        for(i=0;i<(int)NumVarParam;i++)
          VarParam[i]->value = Xk[i] + u*Direction[i];
        tmp = npe_errorFunc( AllParam );
        fb = fc; fc = fu; fu = tmp;
      }
    } else if((u - ulim)*(ulim - *cx) >= 0.0) {
      u = ulim;
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );
    } else {
      u = (*cx) + GOLD*(*cx - *bx);
      for(i=0;i<(int)NumVarParam;i++)
        VarParam[i]->value = Xk[i] + u*Direction[i];
      fu = npe_errorFunc( AllParam );
    }
    *ax = *bx; *bx = *cx; *cx = u;
    fa = fb; fb = fc; fc = fu;
  }
}

#ifdef __cplusplus
}
#endif
