/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: conjfr.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/**************************************************************
**  conjfr.cpp                                               **
**  conjugate gradient method using fletcher-reeves formula  **
**************************************************************/
#define  MOD_NPE_LIBNPEE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "libsatellite.h"
#include "libnpec.h"
#include "libnpee.h"
#include "npeepriv.h"

#ifdef __cplusplus
extern "C" {
#endif

#define MAXITERATION	(10000)

/* usually RESTART sets to n, n+1, 2n as n:parameter num. */
#define	RESTART		(NumVarParam)

/* private variables */
/* for beta, gamma */
static double Beta, *Gamma;

/* private functions */
static void  conjGradientInit();
static void  calcDirection();
static void  linearSearch();
static void  renewBeta();

/*********************************************************************
**  Main routine of quasi-Newton Method                             **
**********************************************************************/
int npe_method_conjfr(){
  /* start up message */
  printf("[ NPE ] CONJFR\n");
  fflush(stdout);

  conjGradientInit();

  for(Step=0; Step < MAXITERATION; Step++){
    npe_display_parameters(AllParam, DispValue);

    npe_storeParamHist(AllParam, NPE_FALSE);

    /* whether stop condition has been satisfied or not */
    if(npe_term()){
      fprintf(stdout,"Esitimation is finishied.\n");
      break;
    }

    npe_scaleTrans(VarParam);

    calcDirection();
    linearSearch();
    renewBeta();
  }
  npe_storeParamHist(AllParam, NPE_TRUE);
  npe_cleanUp();

  return 0; /* success */
}

/*********************************************************************
**  Subroutine of Initializition                                    **
**********************************************************************/
static void  conjGradientInit(){
  int  i;

  npe_readData();
  npe_readParam();
  npe_readTermCriterion();
  npe_readStoreFile();

  Beta = 0.0;
  nablaF    = (double *) malloc2Dim(0, NumVarParam, 'D');
  Gamma     = (double *) malloc2Dim(0, NumVarParam, 'D');
  Direction = (double *) malloc2Dim(0, NumVarParam, 'D');

  ErrorValue = npe_errorFunc(AllParam);
  DispValue = tmpDisp;
  PenaltyValue = tmpPena;
  npe_differentiate(VarParam, nablaF, NULL);
  for(i=0; i<(int)NumVarParam; i++)
    Direction[i] =  0.0;
}

/*********************************************************************
**  Subroutine of Calculation of Direction                          **
**********************************************************************/
static void  calcDirection(){
  int  i;
  double  D=0.0;

  for(i=0; i<(int)NumVarParam; i++){
    Direction[i] = - nablaF[i] + Beta * Direction[i];
    D += square(Direction[i]);
  }

  if(D != 0.0){
    D = 1.0 / sqrt(D);
    for(i=0; i<(int)NumVarParam; i++)
      Direction[i] *= D;     /* normalization */
  }
}

/*********************************************************************
**  Subroutine of LinearSearch                                      **
**********************************************************************/
static void  linearSearch(){
  int  i;

/*  kepp a gradient at X_k-1. Use in the cal. of beta */
  for(i=0; i<(int)NumVarParam; i++)
    Gamma[i] = nablaF[i];    /* Gamma = dF(Xk-1) */

  /* optimization by means of linearity */
  npe_interpolation();
  DispValue = tmpDisp;
  PenaltyValue = tmpPena;
}


/*********************************************************************
**  Subroutine of Calculation of Beta : Fletcher-Reeves             **
**********************************************************************/
static void  renewBeta(){
  int  i;
  double  norm2 = 0.0; /* keep before norm^2 */

  Beta = 0.0;

  npe_differentiate(VarParam, nablaF, NULL); 

  if((Step+1) % RESTART == 0){
    /* restart as \beta_0 = 0.0 */
    Beta = 0.0;
  } else {
/*-------------------------------------------------------------------
             || nablaF(X[k]) ||^2
  Beta = --------------------------
            || nablaF(X[k-1]) ||^2
--------------------------------------------------------------------*/
    for(i=0; i<(int)NumVarParam; i++)
      norm2 += square(Gamma[i]);

    if(norm2 == 0.0){ /* infrequently gradient = 0 */
      fprintf(stderr,"Derivates are zero.\n");
      fprintf(stderr,"I guess the estimation is done. Now storing results\n");
      npe_storeParamHist(AllParam, NPE_TRUE);
      exit(0); /* success */
    }

    for(i=0; i<(int)NumVarParam; i++)
      Beta += square(nablaF[i]);

    Beta /= norm2;
  }
}

#ifdef __cplusplus
}
#endif
