/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: set.c,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "libnpec.h"
#include "npecpriv.h"

#ifdef __cplusplus
extern "C" {
#endif

/* private function */
static int   _npe_estparam_init(npe_common_t *nc, unsigned int size);
static char *_npe_string_trim(const char *fname, int *status);

/***************************************************************/
/* set method element                                          */
/* return value:                                               */
/*               1: illegal type                               */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/*              -2: out of memory                              */
/***************************************************************/
int npe_method_set(npe_common_t *nc, int type){
  char *name = NULL;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(type < NPE_METHOD_NONE || type >=NPE_METHOD_TYPESIZE)
    return  1;
  if(type != NPE_METHOD_NONE){
    name = strdup(npec_method_str[type]);
    if(name == NULL)
      return -2; /* out of memory */
  }
  nc->method_type = type;
  nc->method_name = name;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set lsearch element                                         */
/* return value:                                               */
/*               2: illegal value                              */
/*               1: illegal type                               */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/*              -2: out of memory                              */
/***************************************************************/
int npe_lsearch_set(npe_common_t *nc, int type, double value){
  char *name = NULL;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(type < NPE_LSEARCH_NONE || type >=NPE_LSEARCH_TYPESIZE)
    return  1;
  if(type == NPE_LSEARCH_NONE){
    value = 0.0;
  }else{
    if(value < 0.0)
      return 2;
    name = strdup(npec_lsearch_str[type]);
    if(name == NULL)
      return -2; /* out of memory */
  }
  if(nc->lsearch_name != NULL)
    free(nc->lsearch_name);
  nc->lsearch_type  = type;
  nc->lsearch_value = value;
  nc->lsearch_name  = name;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set model element                                           */
/* return value:                                               */
/*               2: illegal file name                          */
/*               1: illegal type                               */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/*              -2: out of memory                              */
/***************************************************************/
int npe_model_set(npe_common_t *nc, int type, const char *fname){
  int   err;
  char *file = NULL;
  char *name = NULL;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(type < NPE_MODEL_NONE || type >=NPE_MODEL_TYPESIZE)
    return 1; /* illegal type */
  if(type != NPE_MODEL_NONE){
    if(fname == NULL)
      return 2; /* illegal file name */
    if(*fname != '\0'){
      file = _npe_string_trim(fname, &err);
      switch(err){
      case -1: return -2; /* out of memory */
      case -2: return  2; /* illegal file name */
      }
      name = strdup(npec_model_str[type]);
      if(name == NULL){
        free(file);
        return -2; /* out of memory */
      }
    }
  }
  if(nc->model_name)
    free(nc->model_name);
  if(nc->model_file)
    free(nc->model_file);
  nc->model_type = type;
  nc->model_name = name;
  nc->model_file = file;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set penalty function element                                */
/* return value:                                               */
/*               1: illegal file name                          */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/*              -2: out of memory                              */
/***************************************************************/
int npe_penalty_set(npe_common_t *nc, const char *fname){
  int   err;
  char *file = NULL;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(fname == NULL)
    return 1; /* illegal file name */
  if(*fname != '\0'){
    file = _npe_string_trim(fname, &err);
    switch(err){
    case -1: return -2; /* out of memory */
    case -2: return  1; /* illegal file name */
    }
  }
  if(nc->penalty_file)
    free(nc->penalty_file);
  nc->penalty_file = file;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set init element                                            */
/* return value:                                               */
/*               3: illegal parameter name                     */
/*               2: illegal type flag                          */
/*               1: illegal number of parameter                */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/*              -2: out of memory                              */
/***************************************************************/
int npe_init_set(npe_common_t *nc, unsigned int size, unsigned int inum, 
                 double value, int flag, const char *name, double span){
  int   err, num;
  npe_estparam_t e;
  e.value     = 0.0;
  e.flag      = NPE_PARAM_FLAG_VAR;
  e.flag_name = NULL;
  e.name      = NULL;
  e.span      = 1.0;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(size > NPE_LIMIT){ /* limit check */
    return 1; /* illegal number of parameter */
  }else if(size == 0){
    inum = 0;
  }else if(size < inum || inum <= 0){
    return 1; /* illegal number of parameter */
  }else{
    if(flag < NPE_PARAM_FLAG_NONE || flag >= NPE_PARAM_FLAG_TYPESIZE)
      return 2; /* illegal type flag */
    if(flag == NPE_PARAM_FLAG_NONE){
      e.flag_name = strdup(npec_param_flag_str[NPE_PARAM_FLAG_VAR]);
      if(e.flag_name == NULL)
        return -2; /* out of memory */
      e.name      = strdup("N");
      if(e.name == NULL){
        free(e.flag_name);
        return -2; /* out of memory */
      }
    }else{
      e.value     = value;
      e.flag      = flag;
      e.span      = span;
      e.name      = _npe_string_trim(name, &err);
      switch(err){
      case -1: return -2; /* out of memory */
      case -2: return 3;  /* illegal parameter name */
      }
      e.flag_name = strdup(npec_param_flag_str[flag]);
      if(e.flag_name == NULL){
        free(e.name);
        return -2; /* out of memory */
      }
    }
  }
  /* check size */
  if(nc->init_paramnum != size){
    /* allocate or initialize */
    err = _npe_estparam_init(nc, size);
    /* if error occured then cause was 'out of memory' only */
    if(err != 0){
      if(e.name != NULL) free(e.name);
      if(e.flag_name != NULL) free(e.flag_name);
      return -2; /* out of memory */
    }
  }
  num = inum - 1; /* it is specification of cinit() */
  if(num < 0){ /* for case of size == 0 */
    return 0; /* success */
  }
  if(nc->estparam[num].name)
    free(nc->estparam[num].name);
  if(nc->estparam[num].flag_name)
    free(nc->estparam[num].flag_name);
  nc->estparam[num].value     = e.value;
  nc->estparam[num].flag      = e.flag;
  nc->estparam[num].flag_name = e.flag_name;
  nc->estparam[num].name      = e.name;
  nc->estparam[num].span      = e.span;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}

/***************************************************************/
/* set scale element                                           */
/* return value:                                               */
/*               1: illegal type                               */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_scale_set(npe_common_t *nc, int type){
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(type < NPE_SCALE_NONE || type >=NPE_SCALE_TYPESIZE)
    return  1;
  nc->scale_type = type;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}

/***************************************************************/
/* set logic of term element                                   */
/* return value:                                               */
/*               1: illegal logic string                       */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/*              -2: out of memory                              */
/***************************************************************/
int npe_logic_set(npe_common_t *nc, const char *logic){
  int   i;
  int   err;
  char *logic_dup = NULL;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(logic == NULL)
    return 1; /* illegal logic string */
  if(*logic != '\0'){
    logic_dup = _npe_string_trim(logic, &err);
    switch(err){
    case -1:  return -2; /* out of memory */
    case -2:  return 1;  /* illegal logic string */
    }
  }
  if(nc->term_logic)
    free(nc->term_logic);
  /* initialize term values.  it is specification of NPE. */
  for(i=0; i<NPE_TERM_CRITESIZE; i++){
    /* i think, this is not necessary */
    npe_term_set(nc, i, 0.0, NPE_FALSE);
  }
  nc->term_logic = logic_dup;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set term element                                            */
/* return value:                                               */
/*               1: illegal terminate number                   */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_term_set(npe_common_t *nc, int num, double value, npe_bool_t flag){
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(num < 0 || num >= NPE_TERM_CRITESIZE)
    return 1; /* illegal terminate number */
  if(flag == NPE_FALSE)
    value = 0.0;
  nc->term_setbit[num] = flag;
  nc->term_crite[num]  = value;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set number element                                          */
/* return value:                                               */
/*               1: illegal number of wave                     */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_number_set(npe_common_t *nc, int num){
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(num < 0 || num > NPE_LIMIT) /* limit check */
    return 1; /* illegal number of wave */
  nc->number_value = num;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set point element                                           */
/* return value:                                               */
/*               1: illegal number of point                    */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_point_set(npe_common_t *nc, int num){
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(num < 0 || num > NPE_LIMIT) /* limit check */
    return 1; /* illegal point of wave */
  nc->point_value = num;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set data element                                            */
/* return value:                                               */
/*               1: illegal file name                          */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_data_set(npe_common_t *nc, const char *fname){
  int   err;
  char *file = NULL;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(fname == NULL){
    return 1;
  }
  if(*fname != '\0'){
    file = _npe_string_trim(fname, &err);
    switch(err){
    case -1: return -2; /* out of memory */
    case -2: return  1; /* illegal file name */
    }
  }
  /* release old memory */
  if(nc->data_file)
    free(nc->data_file);
  nc->data_file = file;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}

/***************************************************************/
/* set weight element                                          */
/* return value:                                               */
/*               1: illegal file name                          */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_weight_set(npe_common_t *nc, const char *fname){
  int   err;
  char *file = NULL;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(fname == NULL){
    return 1;
  }
  if(*fname != '\0'){
    file = _npe_string_trim(fname, &err);
    switch(err){
    case -1: return -2; /* out of memory */
    case -2: return  1; /* illegal file name */
    }
  }
  /* release old memory */
  if(nc->weight_file)
    free(nc->weight_file);
  nc->weight_file = file;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}

/***************************************************************/
/* set result element                                          */
/* return value:                                               */
/*               1: illegal file name                          */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_result_set(npe_common_t *nc, const char *fname){
  int   err;
  char *file = NULL;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(fname == NULL){
    return 1;
  }
  if(*fname != '\0'){
    file = _npe_string_trim(fname, &err);
    switch(err){
    case -1: return -2; /* out of memory */
    case -2: return  1; /* illegal file name */
    }
  }
  /* release old memory */
  if(nc->result_file)
    free(nc->result_file);
  nc->result_file = file;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}

/***************************************************************/
/* set history element                                         */
/* return value:                                               */
/*               2: illegal interval                           */
/*               1: illegal file name                          */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_history_set(npe_common_t *nc, const char *fname, int interval){
  int   err;
  char *file = NULL;
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(fname == NULL){
    return 1;
  }
  if(*fname != '\0'){
    if(interval < 0)
      return 2; /* illegal interval */
    else if(interval == 0)
      interval = 1;
    file = _npe_string_trim(fname, &err);
    switch(err){
    case -1: return -2; /* out of memory */
    case -2: return  1; /* illegal file name */
    }
  }else{
    interval = 0;
  }
  /* release old memory */
  if(nc->history_file)
    free(nc->history_file);
  nc->history_interval = interval;
  nc->history_file = file;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set display element                                         */
/* return value:                                               */
/*               1: illegal type                               */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_display_set(npe_common_t *nc, int type){
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(type < NPE_DISPLAY_NONE || type >=NPE_DISPLAY_TYPESIZE)
    return  1;
  nc->display_type = type;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}


/***************************************************************/
/* set norm element                                            */
/* return value:                                               */
/*               1: illegal type                               */
/*               0: success                                    */
/*              -1: npe common not initialized                 */
/***************************************************************/
int npe_norm_set(npe_common_t *nc, int type){
  if(nc == NULL)
    return -1; /* npe common not initialized */
  if(type < NPE_NORM_NONE || type >=NPE_NORM_TYPESIZE)
    return  1;
  nc->norm_type = type;
  nc->store_flag = NPE_FALSE;
  return 0; /* success */
}

/* private function */
/***************************************************************/
/* allocate & initialize init element of estimation parameters */
/* return value:                                               */
/*               0: success                                    */
/*              -2: out of memory                              */
/***************************************************************/
static int _npe_estparam_init(npe_common_t *nc, unsigned int size){
  unsigned int i, oldsize;
  npe_estparam_t *ret;
  oldsize = nc->init_paramnum;
  if(size <= 0){
    ret = NULL;
    size = 0;
  }else{
    ret = (npe_estparam_t*)malloc(sizeof(npe_estparam_t)*size);
    if(ret == NULL)
      return -2; /* out of memory */
    for(i=0; i<size; i++){
      ret[i].value     = 0.0;
      ret[i].flag      = NPE_PARAM_FLAG_VAR;
      ret[i].flag_name = strdup(npec_param_flag_str[NPE_PARAM_FLAG_VAR]);
      ret[i].name      = strdup("N");
      ret[i].span      = 1.0;
    }
    /* check memory allocate error */
    for(i=0; i<size; i++){
      if(ret[i].name == NULL || ret[i].flag_name == NULL){
        for(i=0; i<size; i++){
          if(nc->estparam[i].flag_name)
            free(nc->estparam[i].flag_name);
          if(nc->estparam[i].name)
            free(nc->estparam[i].name);
        }
        free(ret);
        return -2; /* out of memory */
      }
    }
  }
  if(nc->estparam){
    for(i=0; i<oldsize; i++){
      if(nc->estparam[i].flag_name)
        free(nc->estparam[i].flag_name);
      if(nc->estparam[i].name)
        free(nc->estparam[i].name);
    }
    free(nc->estparam);
  }
  nc->estparam = ret;
  nc->init_paramnum = size;
  return 0; /* success */
}

/***************************************************************/
/* duplicate trimmed file name                                 */
/* return value:  non NULL : allocated memory                  */
/*                    NULL : an error occured                  */
/* error number:                                               */
/*               0: success                                    */
/*              -1: out of memory                              */
/*              -2: trimmed file name is null string           */
/***************************************************************/
static char *_npe_string_trim(const char *fname, int *err){
  size_t i, len, rspace, lspace, retlen;
  const char *p;
  char *ret;
  /* check the null file name */
  if(fname == NULL){ *err = -2; return NULL; }
  /* get file name length */
  len = strlen(fname);
  /* check the zero length file name */
  if(len == 0){ *err = -2; return NULL; }
  /* count up left side white space */
  for(p=fname, lspace=0; *p != '\0'; p++, lspace++)
    if(*p != ' ' && *p != '\t' && *p != '\n' && *p != '\r') break;
  /* check white spaced only string */
  if(lspace == len){ *err = -2;  return NULL; }
  /* count up right side white space */
  for(p=fname+len-1, rspace=0; p != fname; p--, rspace++)
    if(*p != ' ' && *p != '\t' && *p != '\n' && *p != '\r') break;
  /* calculate returning string length */
  retlen = len - (lspace + rspace);
  /* allocate memory */
  ret = (char*)malloc((retlen + 1) * sizeof(char));
  /* check memory allocation error */
  if(ret == NULL){ *err = -1; return NULL; }
  /* copy string */
  for(i = lspace; i<retlen; i++)
    ret[i] = fname[lspace+i];
  ret[retlen] = '\0';
  /* success */
  *err = 0;
  return ret;
}

#ifdef __cplusplus
}
#endif
