/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: npe.cpp,v 1.3 2004/07/27 11:13:13 yoshihiko Exp $ */
/*******************************
**  NPE main control routine  **
*******************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "libsatellite.h"
#include "libnpec.h"
#include "npepriv.h"

using namespace std;

#ifdef __cplusplus
extern "C" {
#endif

static int _npe_check_filetype(const char *fname);
static int _npe_common_memory_check(npe_condition_t *ncond);
static int _npe_read_data(symbol_t *sym, const char *fname);

DLLEXPORT int mod_npe_npe()
{
  npe_condition_t ncond;
  int   errcode, oldsize;
  symbol_t *sym_res, *sym_his;

  /* initialize condtion */
  if(get_tmpdir() == NULL)
    return 201; /* can not get temporary directory */
  ncond.model_file_type   = NPE_FILETYPE_NONE;
  ncond.penalty_file_type = NPE_FILETYPE_NONE;

  /* get argument 'result'  */
  sym_res = syscom->get_symbol(0);
  if(sym_res == 0)
    return 202; /* argument 'result' is not variable */

  /* get argument 'result'  */
  sym_his = syscom->get_symbol(1);
  if(sym_his == 0)
    return 203; /* argument 'history' is not variable */

  /* check npe common memory */
  if((errcode = _npe_common_memory_check(&ncond)) != 0)
    return errcode;

  /* create parameter estimation program */
  if((errcode = npe_make(&ncond)) != 0)
     return errcode;

  /* execute parameter estimation program */
  /* change data size */
  oldsize = sl2_get_data_size();
  ChangeDataSize(8);
  if((errcode = npe_run(&ncond)) != 0)
    return errcode;

  printf(">> Estimation is done.\n");
  /* copy result data file to result buffer */
  if((errcode = _npe_read_data(sym_res, npe_common->result_file)) != 0)
    return errcode;
  /* read history data file to history buffer */
  if((errcode = _npe_read_data(sym_his, npe_common->history_file)) != 0)
    return errcode;
  /* resotre data size */
  ChangeDataSize(oldsize);
  return 0;
}

static int _npe_check_filetype(const char *fname){
  char *ext;
  int   ret = NPE_FILETYPE_OBJECT;
  ext = strrchr(fname, '.');
  if(ext != NULL){
#ifdef WIN32
    if(npe_strincmp(ext, ".c", 3) == 0){
      ret = NPE_FILETYPE_CSOURCE;
    }else if(npe_strincmp(ext, ".cc", 4) == 0 || 
             npe_strincmp(ext, ".cxx", 5) == 0 ||
             npe_strincmp(ext, ".cpp", 5) == 0){
      ret = NPE_FILETYPE_CXXSOURCE;
    }else if(npe_strincmp(ext, ".dll", 5) == 0){
      ret = NPE_FILETYPE_DLL;
    }
#else
    if(strncmp(ext, ".c", 3) == 0){
      ret = NPE_FILETYPE_CSOURCE;
    }else if(strncmp(ext, ".C", 3) == 0 || 
             npe_strincmp(ext, ".cc", 4) == 0 || 
             npe_strincmp(ext, ".cxx", 5) == 0 ||
             npe_strincmp(ext, ".cpp", 5) == 0){
      ret = NPE_FILETYPE_CXXSOURCE;
    }
#endif
  }
  return ret;
}

static int _npe_common_memory_check(npe_condition_t *ncond){
  FILE *fp;
  const char *p, *str;
  int num;
  char *mdlobj;
  /* method */
  if(npe_common->method_type == NPE_METHOD_NONE)
    return 204; /* optimazation method not defined */
  if(npe_common->method_type != NPE_METHOD_SIMPLEX){
    /* lsearch */
    if(npe_common->lsearch_type == NPE_LSEARCH_NONE)
      return 205; /* linear search method not defined */
  }
  /* model */
  if(npe_common->model_type == NPE_MODEL_NONE)
    return 206; /* model file is not defined */
#ifdef WIN32
  if(npe_common->model_type == NPE_MODEL_USR){
    if(Access(npe_common->model_file, SL_FATTR_FOK) != 0)
      return 207; /* can't fild model file */
  }
  else {
    mdlobj = npe_tmpfile_path(get_tmpdir(), npe_common->model_file);
	if(Access(mdlobj, SL_FATTR_FOK) != 0) {
	  free(mdlobj);
      return 207; /* can't fild model file */
	}
	free(mdlobj);
  }
#else
  if(Access(npe_common->model_file, SL_FATTR_FOK) != 0)
    return 207; /* can't fild model file */
#endif
  if(npe_common->model_type == NPE_MODEL_USR){
    ncond->model_file_type = _npe_check_filetype(npe_common->model_file);
  }else{
    ncond->model_file_type = NPE_FILETYPE_OBJECT;
  }
  /* penalty */
  if(npe_common->penalty_file == NULL)
    return 208; /* penalty file is not defined */
  if(Access(npe_common->penalty_file, SL_FATTR_FOK) != 0)
    return 209; /* can't find penalty file */
  ncond->penalty_file_type = _npe_check_filetype(npe_common->penalty_file);
  /* init */
  if(npe_common->init_paramnum == 0)
    return 210; /* initial estimation parameter is not defined */
  /* scale */
  if(npe_common->scale_type == NPE_SCALE_NONE)
    return 211; /* scale type is not defined */
  /* logic */
  if(npe_common->term_logic == NULL)
    return 212; /* terminate criterion logic is not defined */
  /* term */
  str = npe_common->term_logic;
  num=0;
  for(p=str; *p!='\0'; p++){
    if(isdigit(*p)){
      if(npe_common->term_setbit[*p-'0'] == NPE_FALSE)
        return 213; /* terminate criterion is not defined */
      num++;
    }
  }
  if(num==0)
    return 213; /* terminate criterion is not defined */
  /* number */
  if(npe_common->number_value == -1)
    return 214; /* number of wave is not defined */
  /* point */
  if(npe_common->point_value == -1)
    return 215; /* number of data point is not defined */
  /* data */
  if(npe_common->data_file == NULL)
    return 216; /* experimental data file is not defined */
  if(Access(npe_common->data_file, SL_FATTR_FOK) != 0)
    return 217; /* can't fild experimental data file */
  /* weight */
  if(npe_common->weight_file == NULL)
    return 218; /* weight data file is not defined */
  if(Access(npe_common->weight_file, SL_FATTR_FOK) != 0)
    return 219; /* can't fild weight data file */
  /* result */
  if(npe_common->result_file == NULL)
    return 220; /* result data file is not defined */
  if((fp = fopen(npe_common->result_file, "w")) == NULL)
    return 221; /* can't write result data file */
  fclose(fp);
  /* history */
  if(npe_common->history_file == NULL)
    return 222; /* history data file is not defined */
  if((fp = fopen(npe_common->history_file, "w")) == NULL)
    return 223; /* can't write history data file */
  fclose(fp);
  /* disp - (option element) */
  /* norm - (option element) */
  return 0;
}

static int _npe_read_data(symbol_t *sym, const char *fname){
  SL_Object     *obj;
  Series_Buffer *buf;
  Index          idx;
  Header head;
  double *dpoint, *data;
  int     idxsize;
  /* must be 8 byte blocksize */
  data = (double*)_ReadFile(fname, &head);
  if(data == NULL)
    return 225; /* can't read fileobject */
  idx.SetIndexInfo(head.dim,head.index);
  obj = symbol_get_object(sym);
  try{
    obj->Init(SL_OBJ::SERIES_O,idx);
  }catch(bad_alloc){
    return 2;   /* out of memory */
  }catch(buffer_exception){
    return 226; /* unknown exception */
  }
  buf = (Series_Buffer*)obj->GetBufferPointer();
  idxsize = buf->IndexSize();
  dpoint = (double*)buf->GetDataPointer();
  memcpy(dpoint, data, idxsize*sizeof(double));
  FreeData(data);
  return 0;
}

#ifdef __cplusplus
}
#endif

