/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: clist.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/*************************************************
**  clist.cpp                                   **
**  command  : clist                            **
**  function : print the estimation conditions  **
*************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>
#include  <ctype.h>

#include  "SL_macro.h"
#include  "SL_cmd.h"

#include "libnpec.h"
#include "npepriv.h"

#ifdef __cplusplus
extern "C" {
#endif

static void listAllElement();
static void listMethod();
static void listLsearch();
static void listModel();
static void listPenalty();
static void listInit();
static void listScale();
static void listTerm();
static void listNumber();
static void listPoint();
static void listData();
static void listWeight();
static void listResult();
static void listHistory();
static void listDisplay();
static void listNorm();

/******************** MAIN ********************/
DLLEXPORT int mod_npe_clist(){
  int   type;
  char *str;

  if(GetArgNum() != 0){
    str = GetString(0);
    if(str == NULL)
      return 102; /* illegal element type */
    type = npe_element_type(str);
    switch(type){
    case NPE_ELEMENT_METHOD:  listMethod();     break;
    case NPE_ELEMENT_LSEARCH: listLsearch();    break;
    case NPE_ELEMENT_MODEL:   listModel();      break;
    case NPE_ELEMENT_PENALTY: listPenalty();    break;
    case NPE_ELEMENT_INIT:    listInit();       break;
    case NPE_ELEMENT_SCALE:   listScale();      break;
    case NPE_ELEMENT_TERM:    listTerm();       break;
    case NPE_ELEMENT_NUMBER:  listNumber();     break;
    case NPE_ELEMENT_POINT:   listPoint();      break;
    case NPE_ELEMENT_DATA:    listData();       break;
    case NPE_ELEMENT_WEIGHT:  listWeight();     break;
    case NPE_ELEMENT_RESULT:  listResult();     break;
    case NPE_ELEMENT_HISTORY: listHistory();    break;  
    case NPE_ELEMENT_DISPLAY: listDisplay();    break;
    case NPE_ELEMENT_NORM:    listNorm();       break;
    case NPE_ELEMENT_ALL:     listAllElement(); break;
    default :
      return 102; /* illegal element type */
    }
  }else{
    listAllElement();
  }
  return 0;
}


static void listAllElement(){
  listMethod();
  listLsearch();
  listModel();
  listPenalty();
  listInit();
  listScale();
  listTerm();
  listNumber();
  listPoint();
  listData();
  listWeight();
  listResult();
  listHistory();
  listDisplay();
  listNorm();
}


static void listMethod(){
  if(npe_common->method_type != NPE_METHOD_NONE)
    printf("method\t: %-15s\n", npe_common->method_name);
}


static void listLsearch(){
  if(npe_common->lsearch_type != NPE_LSEARCH_NONE)
    printf("lsearch\t: %-15s %-13.5g\n", npe_common->lsearch_name, 
           npe_common->lsearch_value);
}

static void listModel(){
  if(npe_common->model_type != NPE_MODEL_NONE){
    printf("model\t:");
    printf(" %s     %-15s\n", npe_common->model_name, npe_common->model_file);
  }
}

static void listPenalty(){
  if(npe_common->penalty_file != NULL)
    printf("penalty\t: %-15s\n", npe_common->penalty_file);
}

static void listInit(){
  unsigned int i;
  if(npe_common->init_paramnum != 0){
    printf("init\t: Number of Parameter = %d\n", npe_common->init_paramnum);
    for(i=0; i<npe_common->init_paramnum; i++){
      /* must be success */
      printf("   %2d : %9.5g\t", i+1, npe_common->estparam[i].value);
      printf("%s", (npe_common->estparam[i].flag == NPE_PARAM_FLAG_FIX) ? 
             "Fixed   ": "Variable");
      printf(" %15s\t%9.5g\n", npe_common->estparam[i].name, 
             npe_common->estparam[i].span);
    }
  }
}

static void listScale(){
  if(npe_common->scale_type != NPE_SCALE_NONE)
    printf("scale\t: %d\n", npe_common->scale_type);
}

static void listTerm(){
  int i;
  if(npe_common->term_logic != NULL){
    printf("term\t: %s\n", npe_common->term_logic);
    for(i=0; i<NPE_TERM_CRITESIZE; i++){
      if(npe_common->term_setbit[i] != NPE_FALSE)
        printf("\t TERM%2d = %-13.5g\n", i, npe_common->term_crite[i]);
    }
  }
}

static void listNumber(){
  if(npe_common->number_value != 0)
    printf("number\t: %4d\n", npe_common->number_value);
}

static void listPoint(){
  if(npe_common->point_value != 0)
    printf("point\t: %4d\n", npe_common->point_value);
}


static void listData(){
  if(npe_common->data_file != NULL)
    printf("data\t: %-15s\n", npe_common->data_file);
}


static void listWeight(){
  if(npe_common->weight_file != NULL)
    printf("weight\t: %-15s\n", npe_common->weight_file);
}


static void listResult(){
  if(npe_common->result_file != NULL)
    printf("result\t: %-15s\n", npe_common->result_file);
}

static void listHistory(){
  if(npe_common->history_file != NULL)
    printf("history\t: %-15s, (interval = %d)\n", 
           npe_common->history_file, npe_common->history_interval);
}

static void listDisplay(){
  if(npe_common->display_type != NPE_DISPLAY_NONE){
    printf("display\t: ");
    switch(npe_common->display_type){
    case NPE_DISPLAY_NO_WEIGHT:   printf("no weight\n");   break;
    case NPE_DISPLAY_WITH_WEIGHT: printf("with weight\n"); break;
    default: printf("Unknown type\n");
    }
  }
}


static void listNorm(){
  if(npe_common->norm_type != NPE_NORM_NONE){
    printf("norm\t: ");
    switch(npe_common->norm_type){
    case NPE_NORM_ONE:  printf("1\n");     break;
    case NPE_NORM_TWO:  printf("2\n");     break;
    case NPE_NORM_INF:  printf("infty\n"); break;
    default : printf("unknown type\n");
    }
  }
}

#ifdef __cplusplus
}
#endif

/* --- end of neplist.c ----- */
