/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/*****************************************************************************

         NCS FUNCTIONS FOR READING OF SIMULATION CONDITION FILE


    $Id: ncssrscf.cpp,v 1.2 2004/07/27 11:14:39 yoshihiko Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_SYSTYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif

#define MOD_NCS_LIBNCSS_EXPORTS
#include "libncsc.h"
#include "libncss.h"
#include "ncssdata.h"
#include "ncssdsp.h"

#ifdef __cplusplus 
extern "C" {
#endif

#define FCNTL_ERROR (-1)

#ifndef WIN32
# ifndef O_BINARY
#  define O_BINARY 0
# endif
#endif

/* private functions */
static void checker(double small, double big);
static int  ReadXin(char *xin_name, int *xfdn_max);
static int  ReadXfp(char *xfp_name, int xfdn_max);
static void PrintXfp(struct scf_xfp  scxfp);
static void Mismatch(int n, const char *out_name);
static int  Bsearch(struct moni_buff *vbuff, int n);
static int  SetOutPtr(int fd_out, char *out_name);
static int  GetOutPtr(int fd_out, char *out_name, int buff_max_number);
static int  SetMoniBuff(struct moni_buff *vbuff);
static int  GetMoniInfo(int fd_out);

/*****************************************************************************
FUNCTION  NCSF_RDMAT
******************************************************************************
READ MODE AND TIMER FILE

return( 0 ) : success;
return( 1 ) : error;

MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT int ncsf_rdmat( char scfn[] ){
  struct scf_mat  sc;
  int             fd_mat, r_size;
  char            mat_name[NCS_FNAME_WD];

#ifdef HAVE_SNPRINTF
  snprintf(mat_name, NCS_FNAME_WD, "%s.%s", scfn, SCF_MAT_EXTN);
#else
  sprintf(mat_name, "%s.%s", scfn, SCF_MAT_EXTN);
#endif
  fd_mat = open(mat_name, O_RDONLY | O_BINARY);
  if (fd_mat == FCNTL_ERROR) {
    /* DEBUG(("mat file open error : %s\n",mat_name)); */
    printf(("mat file open error : %s\n",mat_name));
    exit(201);
  }

  r_size = sizeof(struct scf_mat);
  if(read(fd_mat, (char *) &sc, r_size) < r_size) {
    exit(202);
  }

  if (sc.type == '1') {
    ncsg_endtime = sc.timer.sim_timer[0];
    ncsg_calstep = sc.timer.sim_timer[1];
    ncsg_strstep = sc.timer.sim_timer[2];
    ncsg_bufstep = sc.timer.sim_timer[3];

    /*
     * checker(small, big);
     */
#ifdef NCS_DEBUG_CHILD
    checker(ncsg_calstep, ncsg_strstep);
    checker(ncsg_strstep, ncsg_bufstep);
    checker(ncsg_bufstep, ncsg_endtime);
#endif

    if( ncsg_endtime == 0. || ncsg_calstep == 0. ) {
      exit(204);
	}

   if( sc.integral.maxcell < 0 || ncsg_cell < sc.integral.maxcell ) {
      exit(205);
	}

    ncsg_intg_method = sc.integral.method;
    ncsg_cell   = sc.integral.maxcell;
    ncsg_relerr = sc.integral.relerr;

    if( ncsg_relerr < 0. || ncsg_relerr > 1. ) {
      exit(206);
	}


  }

  if (close(fd_mat) == FCNTL_ERROR)
    exit(203);
  return ( NCS_SUCCESS );

}


/*****************************************************************************
FUNCTION  NCSF_RDPRM
******************************************************************************
READ PARAMETER FILE

return( 0 ) : success;
return( 1 ) : error;

MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT int ncsf_rdprm( char *scfn ){
  struct scf_prm  sc;
  int             fd_prm, r_size;
  char            prm_name[NCS_FNAME_WD];
#ifdef DSPTEC_EXTENSION
  int  n=0;
#endif

#ifdef HAVE_SNPRINTF
  snprintf(prm_name, NCS_FNAME_WD, "%s.%s", scfn, SCF_PRM_EXTN);
#else
  sprintf(prm_name, "%s.%s", scfn, SCF_PRM_EXTN);
#endif
  fd_prm = open(prm_name, O_RDONLY | O_BINARY);
  if(fd_prm == FCNTL_ERROR){
    //  DEBUG(("prm file open error : %s\n",prm_name));
    exit(201);
  }

  r_size = sizeof(struct scf_prm);

  while (read(fd_prm, (char *) &sc, r_size) == r_size 
        && sc.type != SCF_EOF) {
    if (ncsf_prmset_func(&sc) != NCS_SUCCESS) {
      exit(221);
    }
#ifdef DSPTEC_EXTENSION
    scData[n++] = sc;
#endif
  }

  if (close(fd_prm) == FCNTL_ERROR) {
    exit(203);
  }
  return ( NCS_SUCCESS );

}


/*****************************************************************************
FUNCTION  NCSF_RDXINFO
******************************************************************************
READ EXTERNAL INFORMATION FILE

return( 0 ) : success;
return( 1 ) : error;

MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT int ncsf_rdxinfo( char *scfn ){
  char xin_name[NCS_FNAME_WD], xfp_name[NCS_FNAME_WD];
  int  xfdn_max = 0;

#ifdef HAVE_SNPRINTF
  snprintf(xin_name, NCS_FNAME_WD, "%s.%s", scfn, SCF_XIN_EXTN);
  snprintf(xfp_name, NCS_FNAME_WD, "%s.%s", scfn, SCF_XFP_EXTN);
#else
  sprintf(xin_name, "%s.%s", scfn, SCF_XIN_EXTN);
  sprintf(xfp_name, "%s.%s", scfn, SCF_XFP_EXTN);
#endif

  ncsg_xinfaddr[0] = ncsl_ramp;
  ncsg_xinfaddr[1] = ncsl_pulse;
  ncsg_xinfaddr[2] = ncsl_interp;
  ncsg_xinfaddr[3] = ncsl_interp;
#ifdef DSPTEC_EXTENSION
  ncsg_xinfaddr[4] = ncsl_ad3163;
  ncsg_xinfaddr[5] = ncsl_sin;
  ncsg_xinfaddr[6] = ncsl_sweep;
#endif

  ReadXin( xin_name, &xfdn_max );
  ReadXfp( xfp_name, xfdn_max );
  return( NCS_SUCCESS );
}


/*****************************************************************************
FUNCTION  NCSF_RDOUT
******************************************************************************
READ OUTPUT INFORMATION FILE

return( 0 ) : success;
return( 1 ) : error;

MSC Ver.5.1                                   Coded by A.Anzai     06/12/1989
                                              Modified by A.I     14,Jun.,'94
*****************************************************************************/

DLLEXPORT int ncsf_rdout( char scfn[] ){
  int    fd_out;
  char   out_name[NCS_FNAME_WD];
  int    buff_max;

#ifdef HAVE_SNPRINTF
  snprintf( out_name, NCS_FNAME_WD, "%s.%s", scfn, SCF_OUT_EXTN );
#else
  sprintf( out_name, "%s.%s", scfn, SCF_OUT_EXTN );
#endif
  fd_out = open(out_name, O_RDONLY | O_BINARY);
  if(fd_out == FCNTL_ERROR ){
    /*  DEBUG(("out file open error : %s\n",out_name)); */
    exit(201);
  }

  if( ( buff_max = SetOutPtr(fd_out, out_name) ) == 0 ){
    exit(221);
  }
  GetOutPtr( fd_out, out_name, buff_max );
  GetMoniInfo( fd_out );

  if( close(fd_out) == FCNTL_ERROR ){
    exit(203);
  }

  return( NCS_SUCCESS );
}


/*****************************************************************************
FUNCTION  NCSF_RDDLY
******************************************************************************
READ DELAY INFORMATION FILE

return( 0 ) : success;
return( 1 ) : error;

on UNIX                                     Coded by S.Hitomi   10/05/1989
*****************************************************************************/

DLLEXPORT int ncsf_rddly( char scfn[] ){
  struct scf_dly  sc;

  int             fd_dly, r_size, id = 0;
  char            dly_name[NCS_FNAME_WD];
  short           delay_flag;

#ifdef HAVE_SNPRINTF
  snprintf( dly_name, NCS_FNAME_WD, "%s.%s", scfn, SCF_DLY_EXTN );
#else
  sprintf( dly_name, "%s.%s", scfn, SCF_DLY_EXTN );
#endif

  fd_dly = open(dly_name, O_RDONLY | O_BINARY);
  if(fd_dly == FCNTL_ERROR ){
    /* DEBUG(("dly file open error : %s\n",dly_name)); */
    exit(201);
  }

  /* Head of .DLY file */
  r_size = sizeof( struct scf_dly );

  delay_flag = 0;
  while (read(fd_dly, (char *) &sc, r_size) == r_size 
         && sc.type != SCF_EOF) {
    if( sc.type == '1' ) {
      if( id < 0 ) { 
        printf( "Module or Input variable Name Mismatch in '%s'\n", 
                dly_name);
        exit(224);
      }
      
      ncsg_dlytime[id] = sc.delay.time;   
      ncsg_dlyinit_flg[id] = sc.delay.init_flag;
      if(sc.delay.init_flag != 'A' ) {
        ncsg_dlyinit[id] = sc.delay.init_out;
      }
      ncsg_qsize[id] = (long)( sc.delay.time / ncsg_calstep );
      if( ncsg_qsize[id] != 0 ){
        delay_flag = 1; 
      }
      if( ncsg_intg_method == 'R' ){
        ncsg_qsize[id] *= 4;
      }
      
      id++;
    }
  }
  
  if( ncsg_intg_method == 'F' && delay_flag ){
    exit(208);
  }

  if( close(fd_dly) == FCNTL_ERROR ){
    exit(203);
  }
  return( NCS_SUCCESS );
}

/*****************************************************************************
FUNCTION  ncsf_FreeOutBuffer
******************************************************************************
for Free Memory

on UNIX                                     Coded by A.I   5,Oct.,'96
*****************************************************************************/
DLLEXPORT void ncsf_FreeOutBuffer(){
  int i;
  free( ncsg_strdptr );
  free( ncsg_out_buffnum );
  free( ncsg_out_buffdim );
  /* TODO append */
  for(i=0;ncsg_out_bufname[i]!=NULL;i++)
    free( ncsg_out_bufname[i] );
  free( ncsg_out_bufname );
}


/* private functions */

/* required by ncsf_rdmat() */
static void checker(double small, double big){
  if(big < small)
    exit(204);
}

/* required by ncsf_rdxinfo() */
static int ReadXin( char *xin_name, int *xfdn_max ){
  struct scf_xin  scxin;
  char           *mdl_name;
  int             fd_xin, xin_mtype, r_size;
  int             comp_number, nc, xfd_number;  
  double         *mem_addr;

  fd_xin = open(xin_name, O_RDONLY | O_BINARY);
  if(fd_xin == FCNTL_ERROR){
    /* DEBUG(("xin file open error : %s\n",xin_name)); */
    exit(201);
  }

  r_size = sizeof(struct scf_xin);

  nc = 0;
  while (read(fd_xin, (char *) &scxin, r_size) == r_size
         && scxin.type != SCF_EOF) {
    if (scxin.type == '1') {
      /**/
      mdl_name = scxin.xin.rec3.mdl_name;
      comp_number = scxin.xin.rec3.comp_number;
      xfd_number = scxin.xin.rec3.xfd_number;
      /**/
      xin_mtype = string_ana( mdl_name, ncsg_xintblptr, NCS_MODE0 );
      if ( xin_mtype == 0 ){
        exit(207);
      }

      mem_addr = ncsm_xinptr[xin_mtype - 1];
      ncsg_xindata[nc].comp_addr = &mem_addr[comp_number];
      ncsg_xindata[nc].comp_number = comp_number;
      ncsg_xindata[nc].xfd_number = xfd_number;
      if (xfd_number > *xfdn_max){
        *xfdn_max = xfd_number;
      }

      nc++;
    }
  }
  ncsg_nxinc = nc;

  if (scxin.type != SCF_EOF){
    exit(202);
  }
  if (close(fd_xin) == FCNTL_ERROR){
    exit(203);
  }
  return( NCS_SUCCESS );
}

static int ReadXfp( char *xfp_name, int xfdn_max ){
  struct scf_xfp  scxfp;
  int             fd_xfp, xfp_cnt, r_size, ret, nc, time_buffid;
  char            time_buff_name[NCS_DATA_FNAME];

  fd_xfp = open(xfp_name, O_RDONLY | O_BINARY);
  if ( fd_xfp == FCNTL_ERROR ){ 
    /* DEBUG(("xfp file open error : %s\n",xfp_name)); */
    exit(201);
  }

  r_size = sizeof(struct scf_xfp);

  for( xfp_cnt = 0; xfp_cnt < xfdn_max; xfp_cnt++ ){
    ret = read(fd_xfp, (char *) &scxfp, r_size);
#ifdef DEBUG_NCS_CHILD
    PrintXfp(scxfp);
#endif
    if (ret == r_size && scxfp.type != SCF_EOF) {
      ncsg_xfpdata[xfp_cnt].xf_number = scxfp.func_number;
      switch( scxfp.type ) {
       case '1':	/* SCF_XFP_TYPE_PULSE */
	   case '3':	/* SCF_XFP_TYPE_AD3163 */
         for (nc = 0; nc < NCS_XFP_MAX; nc++){
           ncsg_xfpdata[xfp_cnt].xf_para[nc] = scxfp.xfp.rec1.func_para[nc];
         }
         break;
       case '2':
         /*****  Change for multiple exinput buffer  *****/
         /*****                        [28,Aug,`94]  *****/
         exbuff_num++;
         ncsg_xfpdata[xfp_cnt].xf_para[0]=(double)exbuff_num;
         ncsg_xfpdata[xfp_cnt].xf_para[1]=(double)scxfp.xfp.rec2.buffdim;
         /* time information of buffer or file */
         if( strlen( scxfp.xfp.rec2.timename ) == 0 ){ /* none */
           time_buffid = 0;
           time_buff_name[0] = NCS_EOS;
         }else{  /* buffer id of time info buffer */
           time_buffid = scxfp.xfp.rec2.timebuff ;
           strcpy( time_buff_name, scxfp.xfp.rec2.timename );
         }
         if( scxfp.func_number == 3 ){
           /* case of "FILE" type */
           ncsg_buff_info_p = add_exbuf_info( ncsg_buff_info, ncsg_buff_info_p,
                                              scxfp.xfp.rec2.buffnum, 
                                              scxfp.xfp.rec2.datf_name,
                                              time_buffid,
                                              time_buff_name );
         }else{
           ncsg_buff_info_p = add_exbuf_info( ncsg_buff_info, ncsg_buff_info_p,
                                              scxfp.xfp.rec2.buffnum,
                                              scxfp.xfp.rec2.datf_name/*NULL*/,
                                              time_buffid,
                                              time_buff_name );
         }
         break;
       default:
         exit(212);
         break;
      }
    }
  }

  if(close(fd_xfp) == FCNTL_ERROR ){
    exit(203);
  }
  return( NCS_SUCCESS );
}

static void PrintXfp(struct scf_xfp  scxfp){
  int i;
  fprintf(stdout,"-->PrintXfp\n");
  fprintf(stdout,"%c\n",scxfp.type);
  fprintf(stdout,"%d\n",scxfp.func_number);
  for(i = 0;i<SCF_XFP_MAX;i++) {
    fprintf(stdout,"%f\n",scxfp.xfp.rec1.func_para[i]);
  }
  fprintf(stdout,"%d\n",scxfp.xfp.rec2.buffnum);
  fprintf(stdout,"%d\n",scxfp.xfp.rec2.buffdim);
  fprintf(stdout,"%s\n",scxfp.xfp.rec2.datf_name); /* 22 bytes */
  fprintf(stdout,"%s\n",scxfp.xfp.rec2.buffname);  /* 32 bytes */
  fprintf(stdout,"%d\n",scxfp.xfp.rec2.timebuff);  /*  4 bytes */
  fprintf(stdout,"%s\n",scxfp.xfp.rec2.timename);  /* 32 bytes */
}


/* required by ncsf_rdout() */
static void  Mismatch( int n, const char *out_name ){
  printf("Module Name Mismatch in '%s' !!%d\n", out_name, n);
  exit(222);
}

static int Bsearch( struct moni_buff *vbuff, int n ){
  register int  i;

  for( i = 0; i < ncsg_nstrd; i++ ){
    if( (vbuff->num[n] == ncsg_out_buffnum[i]) 
        && (vbuff->dim[n] == ncsg_out_buffdim[i] ) ){
      return(i);
    }
  }
  return(-1);
}


/***** dynamic memory allocation   *****/
/***** for output buffer '96/10/15 *****/
static int SetOutPtr( int fd_out, char out_name[] ){
  int     r_size;
  long    offset;
  OutFile sc;
  int     buff_max_number = 0;

  ncsg_nstrd = 0;

  /* Head of RECORD3 -> RECORD2 */
  r_size = sizeof(OutFile);
  offset = 0L;
  if( lseek(fd_out, offset, SEEK_SET) == -1 ){
    fprintf( stderr, "lseek's error\n" );
  }

  while( read( fd_out, (void *)&sc, r_size ) == r_size 
         && sc.type != SCF_EOF) {
    if( sc.type == SCF_OUT_BUFF_SIZE ){
      int i;
      buff_max_number = sc.out.rec4.max_number;
      ncsg_strdptr = (double **)malloc(sizeof(double *)*buff_max_number);
      if(ncsg_strdptr == NULL){
        fprintf( stderr, "GetOutPtr()[ncsg_strdptr]\n" );
        exit( 213 );
      }
      ncsg_out_buffnum = (int *)malloc(sizeof(int)*buff_max_number);
      if(ncsg_out_buffnum == NULL ){
        fprintf( stderr, "GetOutPtr()[ncsg_out_buffnum]\n" );
        exit( 213 );
      }
      ncsg_out_buffdim = (int *)malloc(sizeof(int)*buff_max_number);
      if( ncsg_out_buffdim  == NULL ){
        fprintf( stderr, "GetOutPtr()[ncsg_out_buffdim]\n" );
        exit( 213 );
      }
      /* TODO append */
      ncsg_out_bufname = (char **)malloc(sizeof(char*)*(buff_max_number+1));
      if( ncsg_out_bufname  == NULL ){
        fprintf( stderr, "GetOutPtr()[ncsg_out_bufname]\n" );
        exit( 213 );
      }
      ncsg_out_bufname[buff_max_number] = NULL;
      for(i=0;i<buff_max_number;i++){
        ncsg_out_bufname[i] = (char *)malloc(NCS_FNAME_WD + 1);
        if( ncsg_out_bufname[i]  == NULL ){
          fprintf( stderr, "GetOutPtr()[ncsg_out_bufname(%d)]\n",i );
          exit( 213 );
        }
      }
      break;
    }
  }
  /* printf( "bmax = %d\n", buff_max_number); */
  return( buff_max_number );
}

static int GetOutPtr( int fd_out, char out_name[], int buff_max_number ){
  int      r_size, mtype;
  long     offset;
  char    *mdl_name, obs_name[SCF_NAME_WID];
  double  *mem_addr=NULL;
  OutFile  sc;

  ncsg_nstrd = 0;

  /* Head of RECORD3 -> RECORD2 -> RECORD3 96/10/15 */
  r_size = sizeof(OutFile);
  offset = OUT_RECORD3;
  lseek(fd_out, offset, SEEK_SET);
  while (read(fd_out, (char *) &sc, r_size) == r_size 
         && sc.type != SCF_EOF) {
    mdl_name = sc.out.rec3.mdl_name;

    switch( sc.type ) {
    case SCF_OUT_BUFF_OUT:
      mtype = string_ana( mdl_name, ncsg_outtblptr, NCS_MODE0 );
      if( mtype == 0 ){
        Mismatch( 1, out_name );
      }
      mem_addr = ncsm_outptr[mtype - 1];
      
      break;

    case SCF_OUT_BUFF_EXIN:
      mtype = string_ana( mdl_name, ncsg_xintblptr, NCS_MODE0 );
        if( mtype == 0 ){
          Mismatch( 2, out_name );
        }
        mem_addr = ncsm_xinptr[mtype - 1];
        
        break;
        
    case SCF_OUT_BUFF_OBS:
      mtype = string_ana( mdl_name, ncsg_outtblptr, NCS_MODE0 );
      if( mtype == 0 ){
        Mismatch( 2, out_name );
      }

#ifdef HAVE_SNPRINTF      
      snprintf( obs_name, SCF_NAME_WID, "o%02d", (mtype - 1) );
#else
      sprintf( obs_name, "o%02d", (mtype - 1) );
#endif
      strcat( obs_name, sc.out.rec3.obs_name );
#if 0
      {
        int i;
        fprintf(stdout,"obs_name[%s]\n", obs_name);
        for(i=0; ncsg_obstblptr[i]!=EOS; i++) {
          fprintf(stdout,"obs[%s]\n", ncsg_obstblptr[i]);
        }
        fflush(stdout);
      }
#endif
      mtype = string_ana( obs_name, ncsg_obstblptr, NCS_MODE0 );
      if( mtype == 0 ){
        Mismatch( 3, out_name );
      }
      mem_addr = ncsm_obsptr[mtype - 1];

      break;
      default:
        /* mem_flag = 2;*/
        break;
    }
    /* TODO append */
    strcpy(ncsg_out_bufname[ncsg_nstrd], sc.out.rec3.buf_name);
    ncsg_out_buffnum[ncsg_nstrd] = sc.out.rec3.buff_num;
    ncsg_out_buffdim[ncsg_nstrd] = sc.out.rec3.buff_dim;
    ncsg_strdptr[ncsg_nstrd++] = &mem_addr[sc.out.rec3.comp_number];
    if( buff_max_number < ncsg_nstrd ){
      exit(231);
    }
  }      /* while */
  return( NCS_TRUE );
}

static int SetMoniBuff( struct moni_buff *vbuff ){
  int  nc, buff;
  int *tmp;

  for (nc = 0; nc < NCS_MONI_NVAR; nc++) {
    buff = Bsearch(vbuff, nc);
    if( buff < 0 ){
      ncsg_vmoniptr[nc] = NULL;
    }else{
      ncsg_vmoniptr[nc] = ncsg_strdptr[buff];
      ncsg_nvmoni++;
    } 
    tmp = vbuff->num;
    ncsg_vbuff[nc] = tmp[nc];
    tmp = vbuff->dim;
    ncsg_vbuffdim[nc] = tmp[nc];
  }
  return( NCS_TRUE );
}

static int GetMoniInfo( int fd_out ){
  long  offset;
  int   r_size;
  struct moni_buff vbuff;
  struct scf_out  sc;


  /* Head of RECORD1 */
  r_size = sizeof(struct scf_out);
  offset = 0L;
  lseek( fd_out, offset, SEEK_SET );

  ncsg_nvmoni = 0;
  while(read(fd_out, (char *) &sc, r_size) == r_size 
        && sc.type != SCF_EOF) {
    switch (sc.type) {
    case SCF_OUT_MONITOR:
      ncsg_monimode = sc.out.rec1.monitor_mode;
      
      if (ncsg_monimode != '1'){
        break;
      }  else {
        vbuff.num = sc.out.rec1.vmoni_buffnum;
        vbuff.dim = sc.out.rec1.vmoni_buffdim;
        
        SetMoniBuff(&vbuff);
      }  
      break;
      
    case SCF_OUT_FILE:
      ncsg_strflg = sc.out.rec2.strf_flag;
      strcpy(ncsg_strfile, sc.out.rec2.strf_name);
      break;
      
    }
  }
  return( NCS_TRUE );
}

#ifdef __cplusplus 
}
#endif
