/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/******************************************************************************

                     Integration Library

    $Id: ncssintg.cpp,v 1.3 2005/02/23 03:33:21 orrisroot Exp $

******************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <errno.h>
#ifdef  HAVE_IEEEFP_H
# include <ieeefp.h>
#endif
#ifdef  HAVE_FLOAT_H
# include <float.h>
#endif

#define MOD_NCS_LIBNCSS_EXPORTS
#include "libncsc.h"
#include "libncss.h"
#include "ncssdata.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef short BOOL;

#define ARG relerr, &abserr,&dumre,iflag
#define remin  (1e-20)
#define dmax1(x,y) ( ((x) < (y)) ? (y) : (x) )
#define dmin1(x,y) ( ((x) < (y)) ? (x) : (y) )

static void impcall();
static void toreset(int flg, double *relerr, double *abserr, 
                     double *dumre, int *iflag);
static void rkf45(INTEG_FUNC cf, int cell, int neqn, double **yd, double *t, 
                  double *tout, double *relerr, double *abserr, int *iflag );
static void fehl(INTEG_FUNC cf, int cell, int neqn,  double **yd, double *t,
                 double *h, double **yp, double **f1, double **f2,
                 double **f3, double **f4, double **f5, double **s );

/* global functions */
/*****************************************************************
*                  Adaptive Integration                          *
*                      based on 5th Fehlberg method              *
*                  (    take 1 step mode    )                    *
******************************************************************/

DLLEXPORT void AdapIntgrl(INTEG_FUNC cf, double tstart, double tstep, 
                          int cell, int neqn,double **y, double *relerr, 
                          int *iflag){
  /* cf      -  function that is called in integral function */
  /* tstart  -  time */
  /* tstep   -  step for calculation */
  /* cell    -  number of cell */
  /* neqn    -  number of integrations */

  static double t, tout, dumre, abserr = 1.e-10; /* abserr = 1.e-16 */

  /*********************************************/
  /* stepsaved = NCS_FALSE;                        */
  /*********************************************/
  t = tstart;
  if (*iflag == 1) {
    dumre = *relerr;
    tout = t;
  } else {
    tout = t + tstep;
  }
  while (1) {
    rkf45(cf, cell, neqn, y, &t, &tout, relerr, &abserr, iflag);
    switch (*iflag) {
    case 1:
    case 8:
      impcall();
      exit(218);
      break;
    case 3:
      toreset(1, ARG);
      break;
    case 5:
      toreset(2, ARG);
      break;
    case 6:
      toreset(3, ARG);
      break;
    case 2:
      *relerr = dumre;
      return;
      break;
    default:
      break;
    }
  }
  return;
}

/******************************************************************************
*                                                                             *
*         Euler Integration Method                                            *
*                                                                             *
*  on UNIX                          Coded by S.Hitomi    10/05/1990           *
*                                                                             *
******************************************************************************/


DLLEXPORT void EulerIntgrl(INTEG_FUNC cf, double t, double h, int cell, 
                           int neqn, double **y, double *relerr, int *iflag){
  register int    i, k;

  (*cf) (t, y, ncs_yp);

  /* for initialize */
  if (*iflag == 1) {
    *iflag = 2;
    return;
  }

  for (k = 0; k < neqn; k++) {
    for (i = 0; i < cell; i++) {
      y[i][k] += h * ncs_yp[i][k];
    }
  }

  return;
}


/******************************************************************************
*                                                                             *
*         Runge-Kutta-Gill  Integration Method                                *
*                                                                             *
*                                                                             *
*  on UNIX                          Coded by S.Hitomi    10/05/1990           *
******************************************************************************/

DLLEXPORT void RKG4Intgrl(INTEG_FUNC cf, double t, double h, int cell, 
                          int neqn, double **y, double *relerr, int *iflag){
  int           i, j, s;
  static double a[4], b[4], c[4], **q;
  double        k, r;

  /* for initialize */
  if (*iflag == 1) {

  /*----------  initialize ------------*/
  /*  1 time                           */
  /*-----------------------------------*/
    (*cf) (t, y, ncs_yp);
  /*-----------------------------------*/
  /*  more 3 times                     */
  /*-----------------------------------*/
    (*cf) (t, y, ncs_yp);
    (*cf) (t, y, ncs_yp);
    (*cf) (t, y, ncs_yp);
  /*-----------------------------------*/
  /*  4 times (for delay length )      */
  /*-----------------------------------*/

    /* constant set */
    a[0] = c[0] = 0.5;
    a[1] = c[1] = 1. - sqrt(0.5);
    a[2] = c[2] = 1. + sqrt(0.5);
    a[3] = 1. / 6.;
    c[3] = 0.5; 

    b[0] = 2.;
    b[1] = 1.;
    b[2] = 1.;
    b[3] = 2.;

    /* Memory Allocation & Zero Clear */
    q = (double **) alloc2d(cell, neqn, sizeof(double));

    /* initialize off */ 
    *iflag = 2;

    return;

  }  /* if */

  /* RKG caluculation step (1,2,3,4) : s(0,1,2,3)*/
  for( s = 0; s < 4; s++ ) {
    switch( s ) {
    case 0: 
      (*cf) (t, y, ncs_yp);
      break;
    case 1:
    case 2: 
      (*cf) (t+h/2., y, ncs_yp);
      break;
    case 3:
      (*cf) (t+h, y, ncs_yp);
      break;
    }
    for (j = 0; j < neqn; j++) {
      for (i = 0; i < cell; i++) {
        k = h * ncs_yp[i][j];
        r = a[s] * (k - b[s] * q[i][j]);
        y[i][j] += r;
        q[i][j] += 3. * r - c[s] * k;

      }
    }
  }
}


/* local functions */

static void impcall(){
  puts("\n\t*** improper Call (in AdapIntgrl) ****\n");
}


static void toreset( int flg, double *relerr, double *abserr, double *dumre, 
                     int *iflag ){
  switch (flg) {
  case 1:
    *dumre = *relerr;
    break;
  case 2:
    *abserr = 1.0e-10;
    break;
  case 3:
    *relerr *= 10.0;
    *iflag = 2;
    break;
  }
  puts("  ** Toleraces Reset **");
  printf("  relerr = %10.3e   abserr = %10.3e  iflag = %2d\n",
         *relerr, *abserr, *iflag);
}


static void rkf45( INTEG_FUNC cf, int cell, int neqn, double **yd, double *t,
                   double *tout, double *relerr, double *abserr, int *iflag ){
  static short    init, hfaild, output;
  static double   ae, dt, ee, eeoet, esttol, maxesttol, et, hmin, rer;
  static double   s, scale, u26, epsp1, eps = 1., h, savre, savae;
  static double **f1, **f2, **f3, **f4, **f5;
  static int      mflag, jflag, kflag;
  register int    i, k;

  /** Check Input Parameters **/
  mflag = *iflag;
  if((neqn < 1) || (*relerr < 0.) || (*abserr < 0.) ||
     (mflag == 0) || (mflag > 8) ||
     ((mflag != 1) && (*t == *tout) && (kflag != 3))) {
    /** Invalid Input **/
    *iflag = 8;
    return;
  }
  if (mflag == 1) {
    /** Compute Machine Epsilon **/
    do {
      eps /= 2.0;
      epsp1 = eps + 1.0;
    } while (epsp1 > 1.0);
    u26 = 26.0 * eps;
    MachineEps = u26;
    
    /** Memory Allocation for Array of Two Dimension. **/
    f1 = (double **) alloc2d(cell, neqn, sizeof(double));
    f2 = (double **) alloc2d(cell, neqn, sizeof(double));
    f3 = (double **) alloc2d(cell, neqn, sizeof(double));
    f4 = (double **) alloc2d(cell, neqn, sizeof(double));
    f5 = (double **) alloc2d(cell, neqn, sizeof(double));
    
  } else if (mflag != 2) {
    switch (*iflag) {
    case 5:
      if (*abserr <= 0.0) {
        printf("input error\n");
        return;
      }
    case 4:
    case 3:
      *iflag = jflag;
      if (kflag == 3)
        mflag = abs(*iflag);
      break;
    default:
      printf("input error\n");
      return;
    }
  } else {
    if (((kflag == 5) && (*abserr == 0.0)) ||
        ((kflag == 6) && (*relerr <= savre) && (*abserr <= savae))) {
      printf("input error\n");
      return;
    }
    if ((kflag == 3) || (init == NCS_TRUE)) {
      *iflag = jflag;
      if (kflag == 3)
        mflag = abs(*iflag);
    }
  }
  
  /**  save input 'iflag' and continuation flag value
   **  for subsequent input checking
   **/
  jflag = *iflag;
  kflag = 0;
  
  /**
   **  save 'relerr', 'abserr' for
   **  checking input on subsequent calls
   **/
  savre = *relerr;
  savae = *abserr;
  
  /**  restrict relative error tolerance to be at least as large
   **  as '2*eps+remin' to avoid limitting presicion difficulties
   **  arising from impossible accuracy requests
   **/
  rer = 2.0 * eps + remin;
  if (*relerr <= rer) {
    /** relerr too small*/
    *relerr = rer;
    *iflag = 3;
    kflag = 3;
    return;
  }
  dt = *tout - *t;
  if (mflag == 1) {
    /** Initialization **/
    init = NCS_TRUE;
    (*cf)(*t, yd, ncs_yp);
    if (*t == *tout) {
      *iflag = 2;
      return;
    }
  }
  init = NCS_FALSE;
  h = dt;
  jflag = 2;
  
  /** if too close to output point, extrapolate and return **/
  if (dt <= u26 * (*t)) {
    for (k = 0; k < neqn; k++)
      for (i = 0; i < cell; i++)
        yd[i][k] += dt * ncs_yp[i][k];
    (*cf) (*tout, yd, ncs_yp);
    *t = *tout;
    *iflag = 2;
    return;
  }
  /** initialize output point indicator **/
  output = NCS_FALSE;
  scale = 2.0 / (*relerr);
  ae = scale * (*abserr);

  /** step by step integration **/
  do {
    /** set smallest allowance stepsize **/
    hmin = u26 * (*t);
    
    /** adjust stepsize if necessary to hit the output point **/
    dt = *tout - (*t);
    
    if (h >= dt) {
      /* complete -> output point */
      output = NCS_TRUE;
      h = dt;
    } else if ((2 * h > dt) && (h < dt)) {
      h = 0.5 * dt;
    }
    
    do {
      /* Finding Next StepSize */
      fehl(cf, cell, neqn, yd, t, &h, ncs_yp, f1, f2, f3, f4, f5, f1);
      
      hfaild = NCS_FALSE;
      maxesttol = eeoet = 0.0;
      for (k = 0; k < neqn; k++) {
        for (i = 0; i < cell; i++) {
          et = fabs(yd[i][k]) + fabs(f1[i][k]) + ae;
          if (et <= 0.0) {
            *iflag = 5;
            return;
          }
          ee = fabs((-2090.0 * ncs_yp[i][k] +
                     (21970.0 * f3[i][k] - 15048.0 * f4[i][k])) +
                    (22528.0 * f2[i][k] - 27360.0 * f5[i][k]));
          eeoet = dmax1(eeoet, ee / et);
        }
        esttol = h * eeoet * scale / 752400.0;
        maxesttol = dmax1(maxesttol, esttol);
        if (esttol > 1.0)
          hfaild = NCS_TRUE;
      }
      
      if (hfaild == NCS_TRUE) {
        /* Unsuccessful Step */
        output = NCS_FALSE;
        s = 0.1;
        if (maxesttol < 59049.0){
          if( (finite(pow(maxesttol, 0.2))!=0 ) ||
              ( errno == EDOM )){
#ifdef NCS_DEBUG
            fprintf( stderr, "maxesttol pow domain error");
#endif
            s = 0.9;
          }else{
            s = 0.9 / pow(maxesttol, 0.2);
          }
        }
        h *= s;
        
        if (h <= hmin) {
          puts("--StepSize too small--");
          printf("( t=%18.10f, h=%18.10f )\n", *t, h);
          *iflag = kflag = 6;
          return;
        }
      }
      
    } while (hfaild == NCS_TRUE);

    /*** Core Integrator for Taking a Single Step ***/
    /*
     *if (stepsaved == NCS_TRUE) save_step(dpt_++, *t, h);
     */
    
    (*t) += h;
    for (k = 0; k < neqn; k++)
      for (i = 0; i < cell; i++)
        yd[i][k] = f1[i][k];
    (*cf) (*t, yd, ncs_yp);
    
    /** Predict Next StepSize **/
    s = 5.0;
    if (maxesttol > 1.889568e-4){
      if( (finite(pow(maxesttol, 0.2))!=0 ) || ( errno == EDOM )) {
#ifdef NCS_DEBUG
        fprintf( stderr, "maxesttol pow domain error");
#endif
        s = 0.9;
      }else{
        s = 0.9 / pow(maxesttol, 0.2);
      }
    }
    if (hfaild)
      s = dmin1(s, 1.0);
    h = dmax1(s * h, hmin);

  } while (output != NCS_TRUE);
  /*** End of Core Integral ***/
  
  *t = *tout;
  *iflag = 2;
  
  return;

}

/*
 * -------------------------------------------------------------- Fehlbelg
 * Fourth-Fifth Order Runge-Kutta Method
 * --------------------------------------------------------------
 */

static void fehl(INTEG_FUNC cf, int cell, int neqn, double **yd, double *t, 
                 double *h, double **yp, double **f1, double **f2, 
                 double **f3, double **f4, double **f5, double **s){
  double       ch;
  register int i, k;

  ch = (*h) / 4.0;
  for (k = 0; k < neqn; k++)
    for (i = 0; i < cell; i++)
      f5[i][k] = yd[i][k] + ch * yp[i][k];
  (*cf) (*t + ch, f5, f1);
  
  ch = 3.0 * (*h) / 32.0;

  for (k = 0; k < neqn; k++)
    for (i = 0; i < cell; i++)
      f5[i][k] = yd[i][k] + ch * (yp[i][k] + 3.0 * f1[i][k]);
  (*cf) (*t + 3.0 * (*h) / 8.0, f5, f2);
  
  ch = (*h) / 2197.0;

  for (k = 0; k < neqn; k++)
    for (i = 0; i < cell; i++)
      f5[i][k] = yd[i][k] + ch *
        (1932.0 * yp[i][k] + (7296.0 * f2[i][k] - 7200.0 * f1[i][k]));
  (*cf) (*t + 12.0 * (*h) / 13.0, f5, f3);

  ch = (*h) / 4104.0;

  for (k = 0; k < neqn; k++)
    for (i = 0; i < cell; i++)
      f5[i][k] = yd[i][k] + ch * ((8341.0 * yp[i][k] - 845.0 * f3[i][k]) +
         (29440.0 * f2[i][k] - 32832.0 * f1[i][k]));
  (*cf) ((*t) + (*h), f5, f4);

  ch = (*h) / 20520.0;

  for (k = 0; k < neqn; k++)
    for (i = 0; i < cell; i++)
      f1[i][k] = yd[i][k] + ch *
        ((-6080.0 * yp[i][k] + (9295.0 * f3[i][k] - 5643.0 * f4[i][k])) +
         (41040.0 * f1[i][k] - 28352.0 * f2[i][k]));
  (*cf) ((*t) + (*h) / 2.0, f1, f5);

  ch = (*h) / 7618050.0;

  for (k = 0; k < neqn; k++)
    for (i = 0; i < cell; i++)
      s[i][k] = yd[i][k] + ch *
        ((902880.0 * yp[i][k] +
          (3855735.0 * f3[i][k] - 1371249.0 * f4[i][k])) +
            (3953664.0 * f2[i][k] + 277020.0 * f5[i][k]));
  return;
}

#ifdef __cplusplus
}
#endif
