/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************

             Functions to input/output to SATELLITE buffers

    $Id: ncssbuff.cpp,v 1.3 2005/02/23 04:40:23 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "libsatellite.h"

#define MOD_NCS_LIBNCSS_EXPORTS
#include "libncsc.h"
#include "libncss.h"
#include "ncssdata.h"

#ifdef __cplusplus
extern "C" {
#endif

static int interp_flag;

/* private functions */
static void      debug(int j, int ncount, double *xdata, double *ydata);
static double    LagInterp(double t0, double *xdata, double *ydata);
static void      decidep(int iypt, double *ybuf, double *xdata, double *ydata);
static int       rebuff(Exbuff *exbuff_info, int buff_index);
static int       read_time_buff(Exbuff *exbuff_info, int buff_index);
static double    get_buffer_data_at_time(double t, Exbuff *exbuff_info);
static Exbuff   *get_exbuf_info_st(int buffnum, char *datafile, 
                                  int timeid, char *timefile);
static void      remove_exbuf_info_st(Exbuff *buf);
static PLPointer remove_exbuf_info(PList buffinfo, PLPointer point);


#ifndef HAVE_STRDUP
static char *strdup(const char *s){
  char *ret;
  ret = (char*)malloc(strlen(s)+1);
  if(ret==NULL) return NULL;
  strcpy(ret,s);
  return ret;
}
#endif


/* global functions */

/*
 *  Function to Add Exinput Buffer Infomation to Structure
 *  input: PList buffinfo --- exinput buffer infomation(EIBI) list
 *         PLPointer buffinfo_p --- pointer of EIBI list 
 *         buffnum --- a number of exinput buffer
 *         datafile --- name of exinput buffer or exinput file
 *         timeid --- buffer id number of time information buffer
 *         timefile --- name of time infomation buffer or file
 *  output: pointer of getting buffer infomation structure
 */
DLLEXPORT PLPointer add_exbuf_info(PList buffinfo, PLPointer buffinfo_p, 
                                   int buffnum, char *datafile, int timeid,
                                   char *timefile){
  Exbuff *buf;

  buf = get_exbuf_info_st( buffnum, datafile, timeid, timefile );

  buffinfo_p = plist_insert( buffinfo, buffinfo_p, (PLMember)buf );
  if(buffinfo_p == NULL)
    exit(213); /* memory allocation error */

  buffinfo_p = plist_advance( buffinfo, buffinfo_p );
  return( buffinfo_p );
}  

/*
 *  Function to Remove all Exinput Buffer Infomation to Structure
 *  input: PList buffinfo --- exinput buffer infomation(EIBI) list
 *          PLPointer point --- pointer of EIBI list 
 *     output: void
 */
DLLEXPORT void remove_all_exbuf_info(PList buffinfo){
  Exbuff *buf;
  PLPointer point1;
  
  point1 = plist_go_end( buffinfo );
  while( point1 != buffinfo ){
    buf = (struct exbuff *)plist_read_data( buffinfo, point1 );
    remove_exbuf_info_st( buf );
    point1 = plist_delete( buffinfo, point1 );
  }
}


/*********************************************************************
*                                                                    *
*         Interporation Main Routine                                 *
*                              Coded by Sigeru Hitomi  [09/21/1990]  *
*                              Last Modified           [17,May,'96]  *
*                                                                    *
**********************************************************************/

DLLEXPORT double ncsl_interp(double pos, double buf_dim, 
                             double dummy0, double dummy1, double dummy2){
  double  xdata[4], ydata[4], y0;
  Exbuff *exbuff_info;
  double  buff_timestep;

  /* read buffer information */
  ncsg_buff_info_p = plist_go_nth( ncsg_buff_info, (int)pos );
  exbuff_info = (Exbuff *)plist_read_data( ncsg_buff_info, ncsg_buff_info_p );

  if (exbuff_info->ic == 0) {  /* read buffer & file */
    exbuff_info->ic = 1;

    if (exbuff_info->ncsg_inp_buffnum <= 0) {
      if(exbuff_info->ncsg_datfile == NULL || 
         exbuff_info->ncsg_datfile[0] == '\0'){
        fprintf(stdout,"exbuff_info->ncsg_inp_buffnum <= 0 error.\n");
        exit(216);
      }
    }
    exbuff_info->iypt = rebuff( exbuff_info, (int)buf_dim );

    if( exbuff_info->timeid != 0 ){ /* read time infomation */
      if( exbuff_info->iypt != read_time_buff( exbuff_info, 0 ) ){
        fprintf( stdout, 
           "Wanning: Mismatch Number of Data Buffer and Time Buffer\n" );
      }
    }
    /* Now every data buffer have a time information buffer 
       respectively.             */

    /* use or not Lagulange Interpolation? */
    if( exbuff_info->timeid != 0 ){
      buff_timestep = ncsg_endtime/(double)exbuff_info->iypt;
      interp_flag = NCS_FALSE;
      switch( ncsg_intg_method ){
       case 'F':
         interp_flag = NCS_TRUE;
         break;
       case 'E':
         if( buff_timestep > ncsg_calstep ){
           interp_flag = NCS_TRUE;
         }
         break;
       case 'R':
        if( buff_timestep > ncsg_calstep/2.0 ){
          interp_flag = NCS_TRUE;
        }
        break;
       default:
        interp_flag = NCS_TRUE;
        break;
      }
    }else{
      interp_flag = NCS_TRUE;
    }

    if( interp_flag != NCS_TRUE ){
      fprintf( stdout, "\n*** Infomation : Stim %d don't Use Interpolation ***\n", (int)pos );
    }
  }

  if( interp_flag == NCS_TRUE ){
    /* if no time data then data padding .. */
    decidep(exbuff_info->iypt, exbuff_info->ybuf, xdata, ydata);
    y0 = LagInterp(NCS_TIME, xdata, ydata);
  }else{
    y0 = get_buffer_data_at_time( NCS_TIME, exbuff_info );
  }

  return (y0);
}


/* local functions */

/*********************************************************************
*                                                                    *
*         4th Order Lagrange Interpolation                           *
*                                                                    *
*         FUNCTION :  NCSL_INTERP                                    *
*                                                                    *
*  on Unix                Coded by Shigeru Hitomi     09/23/1990     *
*--------------------------------------------------------------------*
*                                                                    *
*    double                                                          *
*       ncsl_interp( dummy0, dummy1, dummy2, dummy3, dummy4 )        *
*                                                                    *
*       double  dummy1 - 4    :  dummy parameter                     *
*       double                                                       *
*        (Interpolated value) :  return value                        *
*                                                                    *
*    GLOVAL VALIABLES                                                * 
*       double  NCS_TIME         :  Current Time                     *
*       double  ncsg_endtime     :  Calculation End Time             *
*       double  ncsg_calstep     :  Calculation Step                 *
*       int     ncsg_cdpoint     :  Current Data point ( dpoint < t )*
*       int     ncsg_inp_buffnum :  buffer or block number           *
*       char   *ncsg_datfile     :  Data filename                    *
*                                                                    *
*********************************************************************/

static double LagInterp(double t0, double *xdata, double *ydata){
  register int    j, k;
  double          sumx, sx;

  sumx = 0.0;
  for( j = 0; j < 4; j++ ){
    sx = 1.0;
    for( k = 0; k < 4; k++ ){
      if (j != k){
        sx *= (t0 - xdata[k]) / (xdata[j] - xdata[k]);
      }
    }
    sumx += sx * ydata[j];
  }

  /*
   * printf( "t0 = %f, sumx = %f\n", t0, sumx ); 
   * debug( j, ncount, xdata, ydata );
   */
  return( sumx );
}


/* local functions */

/*********************************************************************
*       Debugging routine                                            *
*                                                                    *
*                              Coded by Sigeru Hitomi                *
*                                                       09/21/1990   *
*********************************************************************/

static void debug(int j, int ncount, double xdata[], double ydata[]){

  printf("%10s %10s %10s %10s %10s\n",
         "J", "XDATA[0]", "XDATA[1]", "XDATA[2]", "XDATA[3]");
  printf("%10d %10.3g %10.3g %10.3g %10.3g\n",
         j, xdata[0], xdata[1], xdata[2], xdata[3]);

  printf("%10s %10s %10s %10s %10s\n",
         "NCOUNT", "YDATA[0]", "YDATA[1]", "YDATA[2]", "YDATA[3]");
  printf("%10d %10.3g %10.3g %10.3g %10.3g\n",
         ncount, ydata[0], ydata[1], ydata[2], ydata[3]);

}


/*********************************************************************
*                                                                    *
*         Decide position for Interpolation                          *
*                                                                    *
*                              Coded by Sigeru Hitomi                *
*                                                       09/21/1990   *
*                                                                    *
**********************************************************************/

static void decidep(int iypt, double *ybuf, double *xdata, double *ydata){
  register int  i, j;
  int           ncount;

  if( ncsg_cdpoint < 1 ){
    ncount = 1;
  }else{
    if ( ncsg_cdpoint >= (iypt - 1)){
      ncount = iypt - 2;
    } else {
      ncount = ncsg_cdpoint;
    }
  }

  j = ncount - 1;
  for( i = 0; i < 4; i++ ){
    xdata[i] = j * ncsg_calstep;
    ydata[i] = ybuf[j];
    j++;
  }
  /*
   * debug( j, ncount, xdata, ydata ); 
   * puts( "In decidep .. [Press CR]" ); getchar();
   */
}


/*********************************************************************
*                                                                    *
*         Read from Data Buffer  /  File                             *
*                              Coded by Sigeru Hitomi  [09/21/1990]  *
*                              Modified                [28,Aug,'94]  *
*                                                                    *
**********************************************************************/

static int rebuff(Exbuff *exbuff_info, int buff_index){
  const char *tmpdir;
  int     iypt, nt;
  double *buf;
  int     dim;
  int     index[MAX_INDEX];
  Header  head;
  int     buffnum;
  char   *datfile;
  int     sub_index[MAX_INDEX];

  buffnum = exbuff_info->ncsg_inp_buffnum;
  datfile = exbuff_info->ncsg_datfile;

  tmpdir = get_tmpdir();
  
  if( strlen(datfile) != 0 ){
    /* input by 'F' */
    char fname[NCS_FNAME_WD];
    if( strlen(datfile) + strlen(tmpdir) + 1 > NCS_FNAME_WD)
      exit(226);
#ifdef HAVE_SNPRINTF
    snprintf( fname, NCS_FNAME_WD, "%s/%s", tmpdir, datfile);
#else
    sprintf( fname, "%s/%s", tmpdir, datfile);
#endif
    buf = (double *)_ReadFile( fname, &head );
    if( buf == NULL ) exit(226);
    if(head.data_size != 8) exit( 227 ); /* TODO: adequate error message */
    dim = head.dim;
    if( dim > 2 ) exit( 227 );
    CopyIndex(index,head.index,dim);
  }else{
    /* input by 'B' */
    exit(211);
    /* buf = ReadBuffer( buffnum, &dim, index ); */
    /* { */
      /* sprintf("%s/%s", tmpdir, datfile); */
      /* buf = (double *)_ReadFile( buffnum, &head ); */
    /* } */
    /* here is always error TODO recheck ! */
    /* buf = NULL; */
    /* if(buf == NULL) exit(211); */
    /* dim = head.dim; */
    /* CopyIndex(index,head.index,dim); */
  }
  iypt = index[0];
  exbuff_info->ybuf = (double*)malloc( sizeof(double)*iypt );
  if( exbuff_info->ybuf == NULL ) exit(213);

  {
    int i;
    sub_index[1] = buff_index;
    for(i=0; i<iypt; i++){
      sub_index[0] = i;
      exbuff_info->ybuf[i] = buf[_Index(sub_index, dim, index )];
    }
  }

  if( iypt <= 0 ) {
    if( strlen(datfile) == 0 ){
      fprintf(stdout," >> Data Buffer[%3d] not ready <<%d\n", buffnum, iypt);
      exit(214);
    }else{
      fprintf(stdout," >> Data File[%s] not ready <<%d\n", datfile, iypt);
      exit(214);
    }
  }
  nt = (int) (ncsg_endtime / ncsg_calstep + 1.5);
  if( iypt < nt ){
    fprintf(stdout,"\n >> Data Buffer less than Calc. points <<\n");
    fprintf(stdout," >>Buffer dpoint = %d\n", iypt );
    fprintf(stdout," >>Calc.   point = %d\n", nt );
    exit(215);
  }

  FreeData(buf);
  return (iypt);
}


/*********************************************************************
*                                                                    *
*         Read from Time Infomation Buffer  ( /  File )              *
*                                 Coded                [17,May,'96]  *
*                                                                    *
**********************************************************************/

static int read_time_buff(Exbuff *exbuff_info, int buff_index){
  /* TODO CHECK !! here is comment outed by OZAKI!!! */
  int     buffid;
  char   *filename;
  int     dim;
  int     index[MAX_INDEX], sub_index[MAX_INDEX];
  Header  head;
  double *buf = NULL;
  int     i;

  buffid   = exbuff_info->timeid;
  filename = exbuff_info->timefile;
  
  if( strlen(filename) != 0 ){/* time information buffer set or not? */
    if( buffid == 0 ){     /* filename = file name */
      /* 'F' */
      buf = (double *)_ReadFile( filename, &head );
      if( buf == NULL ) exit(228);
      if(head.data_size != 8) exit( 227 );
      dim = head.dim;
      if( dim > 2 ) exit( 229 );
      CopyIndex(index,head.index,dim);
      
    }else{  /* finename = buffer name (ignore) */
      /* 'B' */
      exit(211);
      /* buf = ReadBuffer( buffid, &dim, index ); */ 
      /* buf = (double *)_ReadFile( bufname, &head ); */
      /* if( buf == NULL ) exit(211); */
      /* dim = head.dim; */
      /* if (dim <= 0) exit(214); */
      /* CopyIndex(index,head.index,dim); */
    }
    
    exbuff_info->itpt = index[0];
    exbuff_info->tbuf=(double*)malloc( sizeof(double)*index[0] );
    if( exbuff_info->tbuf == NULL ) exit(213);
    sub_index[1] = buff_index;
    for( i = 0; i < index[0]; i++ ){
      sub_index[0] = i;
      exbuff_info->tbuf[i] = buf[_Index(sub_index, dim, index )];
    }
  }else{ /* time information buffer isn't used */
    exbuff_info->tbuf =(double*)malloc( sizeof(double)*exbuff_info->iypt );
    if( exbuff_info->tbuf == NULL ) exit(213);
    exbuff_info->itpt = exbuff_info->iypt;
    for( i = 0; i < exbuff_info->iypt; i++ ){
      exbuff_info->tbuf[i] = ncsg_endtime / exbuff_info->iypt * i;
    }
  }

  if( buf != NULL ){
    FreeData(buf);
  }

  return (exbuff_info->itpt);
}


/*********************************************************************
         Get buffer data at time                   
                                                  [17,May,'96]  
**********************************************************************/
static double get_buffer_data_at_time(double t, Exbuff *exbuff_info){
  int    cnt;
  int    flag;
  double diff, y0=0.0;
  double thresh;

  flag = NCS_FALSE;
  if( t != 0.0 ){
    thresh = floor(log10(t)) - 6.0; 
  }else{
    thresh = -6.0;
  }
  for( cnt = 0; cnt < exbuff_info->itpt; cnt++ ){
    diff = fabs((double)(*(exbuff_info->tbuf+cnt))-t);
    if( diff != 0.0 ){
      diff = log10(diff);
      if( (diff != 0.0) && (diff < thresh) ){
        diff = 0.0;
      }else{
        diff = 1.0;
      }
    }
    if( diff == 0.0 ){
      y0 = (double)*(exbuff_info->ybuf+cnt);
      flag = NCS_TRUE;
      break;
    }
  }

  if( flag != NCS_TRUE ){
    exit(230);
  }

  return(y0);
}


/*************************************************************************
         Functions To Control Exinput Information Structure
                    Coded       [ 9,Aug,'94]
             Modified    [17,May,'96]
**************************************************************************/

/*
 *  Function to Get Exinput Buffer Infomation Structure(  )
 *  input: int buffnum --- a number of exinput buffer
 *             char *datafile --- name of exinput buffer or exinput file
 *             timeid --- buffer id number of time information buffer
 *             timefile --- name of time infomation buffer or file
 *     output: address of getting buffer infomation structure
 */
static Exbuff *get_exbuf_info_st(int buffnum, char *datafile, int timeid, 
                                 char *timefile){
  Exbuff *buf;
  char s[10];

  if( ( buf = (Exbuff *)malloc( sizeof( Exbuff ) ) ) == NULL ){
    exit( 213 );
  }

  buf->ncsg_inp_buffnum = buffnum;
  if( datafile != NULL ){
    if( strcmp(datafile,"BUFFER") != 0 ){
      buf->ncsg_datfile = strdup( datafile );
    }else{
      s[0] = NCS_EOS;
      buf->ncsg_datfile = strdup( s );
    }
  }else{
    s[0] = NCS_EOS;
    buf->ncsg_datfile = strdup( s );
  }
  buf->timeid = timeid;
  buf->timefile = strdup( timefile );
  buf->ic = 0;
  
  return( buf );
}


/*
 *  Function to Remove Exinput Buffer Infomation Structure(  )
 *  input: PLMember buf --- exinput buffer information structure
 *  output: none
 */
static void remove_exbuf_info_st(Exbuff *buf){
  free( buf->ncsg_datfile );
  free( buf->timefile );
  free( buf->ybuf );

  free( buf );
}

/*
 *  Function to Remove Exinput Buffer Infomation to Structure
 *  input: PList buffinfo --- exinput buffer infomation(EIBI) list
 *         PLPointer point --- pointer of EIBI list 
 *  output: pointer of getting buffer infomation structure
 */
static PLPointer remove_exbuf_info(PList buffinfo, PLPointer point){
  Exbuff *buf;

  buf = (struct exbuff *)plist_read_data( buffinfo, point );
  remove_exbuf_info_st( buf );
  point = plist_delete( buffinfo, point );
  return( point );
}  

#ifdef __cplusplus
}
#endif
