/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************

   	Function Library to Control NCS Simulation Condition File 
   		for information of EXINPUT variables(XIN)

    $Id: scfxin.c,v 1.2 2004/04/15 16:57:53 orrisroot Exp $

*****************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif /* HAVE_UNISTD_H */

#define MOD_NCS_LIBNCSC_EXPORTS
#include "libncsc.h"

#include "scf.h"

#ifdef __cplusplus
extern "C" {
#endif

/* File pointer for simulaiton condition file of exinput information */
static FILE *xin;

static int   XinOpenSCF();
static int   XinCreateSCF();
static int   XinCloseSCF();
static int   XinWriteSCF( XinFile *src );
static int   XinReadSCF( XinFile *src );
static void  XinTopSCF();
static void  XinEndSCF();
static void  XinNextSCF();
static void  XinNthNextSCF( int n );
static void  XinPreviousSCF();
static void  XinNthPreviousSCF( int n );
static void  XinOutputEntrySCF( XinFile *src, FILE *disp, int n);

/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make exinput information file
 *      ( ->SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XinMakeSCF(){
  XinFile   src;
  memset(&src, 0, sizeof(XinFile));
  if( XinCreateSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
  if( XinWriteSCF( &src ) == NCS_FALSE ){
    XinCloseSCF();
    return( NCS_FALSE );
  }
  XinCloseSCF();
  return( NCS_TRUE );
}


/*
 *  Add one entry to exinput information file
 *   XinFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XinAddInfoSCF( XinFile *add ){
  XinFile src;
  memset(&src, 0, sizeof(XinFile));
  if( XinOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  XinEndSCF();
  if( XinWriteSCF( add ) == SCF_ERROR ){
    XinCloseSCF();
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
    if( XinWriteSCF( &src ) == SCF_ERROR ){
    XinCloseSCF();
    return( NCS_FALSE );
  }
  XinCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Write one entry to n position of exinput information file 
 *   XinFile *add: add informaiton
 *   int n       : position
 *   Return      : TRUE  : success.
 *                 FALSE : failed.
 */
int XinPutInfoSCF( XinFile *add, int n ){
  if( XinOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  XinTopSCF();
  XinNthNextSCF( n );
  if( XinWriteSCF( add ) == SCF_ERROR ){
    XinCloseSCF();
    return( NCS_FALSE );
  }
  XinCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Set one entry to exinput information structure.
 *  XinFile *src    : storing at the location 
 *  char *mdl_name  : module name 
 *  int  comp       : component number
 *  int  xfp_num    : index of external input function
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int XinSetInfoSCF( XinFile *src, const char *mdl_name, int comp ){
  int  xfp_num;
  src->type = SCF_XIN_TYPE_NORMAL;
  if( mdl_name != NULL ){
    if( strlen( mdl_name ) >= ( SCF_NAME_LEN-1 ) ){
      strncpy( src->xin.rec3.mdl_name, mdl_name, SCF_NAME_LEN-1 );
      src->xin.rec3.mdl_name[SCF_NAME_LEN-1] = '\0';
      fprintf( stderr, "Warnning: module name is too long.\n" );
    }else{
      strcpy( src->xin.rec3.mdl_name, mdl_name );
    }
  } else {
    src->xin.rec3.mdl_name[0] = '\0';
  }
  if( comp >= 0 ){
    src->xin.rec3.comp_number = comp;
  } else {
    src->xin.rec3.comp_number = 0;
  }
  xfp_num = XinSearchInfoSCF( src );
  if( xfp_num >= 0 ){
    src->xin.rec3.xfd_number = xfp_num;
  } else {
    src->xin.rec3.xfd_number = 0;
  }
  return( NCS_TRUE );
}

/*
 *  Search same entry to exinput information file
 *   XinFile *cmp: compare entry
 *      Return      : n     : potion of the same entry
 *                    n < 0 : no match
 */
int XinSearchInfoSCF( XinFile *cmp ){
  XinFile src;
  int     n = 0;
  memset(&src, 0, sizeof(XinFile));
  if( XinOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  while( XinReadSCF( &src ) != SCF_ERROR ){
    if( src.type == SCF_EOF ){
      XinCloseSCF();
      return( -(n+1) );
      break;
    }
    if( (strcmp( cmp->xin.rec3.mdl_name, src.xin.rec3.mdl_name ) == 0) 
        && (cmp->xin.rec3.comp_number == src.xin.rec3.comp_number) ){
      break;
    }
    n++;
  }
  XinCloseSCF();
  return( n );
}


/*
 *  Write one entry to exinput information file
 *   XinFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
/*
int XinWriteInfoSCF( XinFile *src ){
  int             pos;
  pos = XinSearchInfoSCF( src );
  if( pos < 0 ){
    return( XinAddInfoSCF( src ) );
  } else {
    return( XinPutInfoSCF( src, pos ) );
  }
}
*/

/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of exinput information file
 *   XinFile *src: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XinGetInfoSCF( XinFile *src, int n ){
  if( XinOpenSCF() == NCS_FALSE ){
    return( 0 );
  }
  XinTopSCF();
  XinNthNextSCF( n );
    if( XinReadSCF( src ) == NCS_FALSE ){
    XinCloseSCF();
    return( NCS_FALSE );
  }else{
    XinCloseSCF();
    return( NCS_TRUE );
  }
}

/**************************************************************************
       Functions to delete entry of simulation condition file
 **************************************************************************/

/*
 *  Delete one entry from n potition of output information file
 *      int  n      : position   
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XinDeleteInfoSCF( int n ){
  XinFile  tmp;
  memset(&tmp, 0, sizeof(XinFile));
  if( n < 0 ){
    return( NCS_TRUE );
  }
  if( XinOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  XinTopSCF();
  XinNthNextSCF( n );
  do{
    XinNextSCF();
    if( XinReadSCF( &tmp ) == NCS_FALSE ){
      XinCloseSCF();
      return( NCS_FALSE );
    }
    XinNthPreviousSCF( 2 );
    tmp.xin.rec3.xfd_number--;
    if( XinWriteSCF( &tmp ) == NCS_FALSE ){
      XinCloseSCF();
      return( NCS_FALSE );
    }
  }while( tmp.type != SCF_EOF );
  
  XinCloseSCF();
  return( NCS_TRUE );
}


/*
 *  Delete all entry of output information file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XinDeleteAllInfoSCF(){
  XinFile src;
  int     tmp;
  memset(&src, 0, sizeof(XinFile));
  if( XinOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  XinTopSCF();
  src.type = SCF_EOF;
  if( XinWriteSCF( &src ) == NCS_FALSE ){
    return( NCS_FALSE );
  }
  while( (tmp=XinReadSCF( &src )) != NCS_FALSE ){
    if( src.type == SCF_EOF ){
      break;
    }
    src.type = SCF_ERASED;
    XinPreviousSCF();
    if( (tmp=XinWriteSCF( &src )) == NCS_FALSE ){
      XinCloseSCF();
      return( NCS_FALSE );
    }
  }
  XinCloseSCF();
  return( tmp );
}

/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/

/*
 *  Print list of all entry to exinput information file
 *   FILE *disp  : output file
 */
int XinListupSCF( const char *tmpfile ){
  FILE *disp;
  XinFile src;
  int  n = 0, row = 1;

  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;

  memset(&src, 0, sizeof(XinFile));
  if( XinOpenSCF() == NCS_FALSE ){
    fprintf( stderr, "Wanning: Failed to open the SCF\n" );
    fclose(disp);
    return( 0 );
  }
  fprintf( disp, "  No.      Module     Func.No.\n");
  while( XinReadSCF( &src ) != SCF_ERROR ){
    if( src.type == SCF_EOF ){
      if( n == 0 ){
        fprintf( disp, "No data.\n" );
        row++;
      } else {
        fprintf( disp, "\n" );
        row++;
      }
      break;
    }
    if( src.type != SCF_ERASED ){
      if( n == 0 ){
        fprintf( disp, "  No.      Module     Func.No.\n");
      }
      XinOutputEntrySCF( &src, disp, n  );
    }else{
      fprintf( disp, " ERASED DATA\n" );
    }
    row++;
    n++;
  }
  XinCloseSCF();
  fclose(disp);
  return( row );
}

/*
 *  Open exinput information file
 *      Return      : TRUE  : success
 *                    FALSE : failed
 */
static int XinOpenSCF(){
  const char *path;
  char mname[SCF_SCFN_PATH_SIZE];
  char fname[SCF_SCFN_PATH_SIZE];
  
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_XIN_EXTN );
#else
    sprintf( fname, "%s/%s.%s", 
             path, ModelNameSCFN( mname ), SCF_XIN_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }

  if( (xin = _OpenSCF( fname )) == NULL ){
    return( NCS_FALSE );
  }
  return( NCS_TRUE );
}

/*
 *  Open exinput information file
 *      Return      : TRUE  : sucess
 *                    FALSE : failed
 */
static int XinCreateSCF(){
  const char *path;
  char mname[SCF_SCFN_PATH_SIZE];
  char fname[SCF_SCFN_PATH_SIZE];
  
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_XIN_EXTN );
#else
    sprintf( fname, "%s/%s.%s", 
             path, ModelNameSCFN( mname ), SCF_XIN_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }

  if( (xin = _CreateSCF( fname )) == NULL ){
    return( NCS_FALSE );
  }
  return( NCS_TRUE );
}

/*
 *  Close exinput information file
 */
static int XinCloseSCF(){
    return( _CloseSCF( xin ) );
}

/*
 *  Write one entry to exinput information file
 *   XinFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int XinWriteSCF( XinFile *src ){
  if( _WriteSCF( xin, (void *)src, sizeof(XinFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Read one entry from exinput information file
 *   ObsFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int XinReadSCF( XinFile *src ){
  if( _ReadSCF( xin, (void *)src, sizeof(XinFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Set file pointer top of parameter information file
 */
static void XinTopSCF(){
  fseek( xin, 0L, SEEK_SET);
}

/*
 *  Set file pointer end of file
 *   FILE    *fp : file pointer of file
 */
static void XinEndSCF(){
  fseek( xin, (long)sizeof(XinFile)*(-1), SEEK_END );
}

/*
 *  advance file pointer
 */
static void XinNextSCF(){
  fseek( xin, (long)sizeof(XinFile), SEEK_CUR );
}

/*
 *  advance nth file pointer
 */
static void XinNthNextSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    XinNextSCF();
  }
}

/*
 *  retreat file pointer
 */
static void XinPreviousSCF(){
  fseek( xin, (long)sizeof(XinFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer
 */
static void XinNthPreviousSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    XinPreviousSCF();
  }
}

/*
 *  Display one entry to exinput information file
 *     FILE *disp  : output file
 */
static void XinOutputEntrySCF( XinFile *src, FILE *disp, int n  ){
  fprintf( disp, " %3d %9s(%3d)     %2d\n", n+1,
           src->xin.rec3.mdl_name, src->xin.rec3.comp_number,
           src->xin.rec3.xfd_number );
}

#ifdef __cplusplus
}
#endif
