/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************

         Function Library to Control NCS Simulation Condition File 
         for information of EXTERNAL INPUT FUNCTION parameters(XFP)

    $Id: scfxfp.c,v 1.5 2004/07/28 06:08:52 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <string.h>

#define MOD_NCS_LIBNCSC_EXPORTS
#include "libncsc.h"

#include "scf.h"

#ifdef __cplusplus
extern "C" {
#endif

/* File pointer for simulaiton condition file of XFP */
static FILE *xfp;

static int   XfpOpenSCF();
static int   XfpCreateSCF();
static int   XfpCloseSCF();
static int   XfpWriteSCF( XfpFile *src );
static int   XfpReadSCF( XfpFile *src );
static void  XfpTopSCF();
static void  XfpEndSCF();
static void  XfpNextSCF();
static void  XfpNthNextSCF( int n );
static void  XfpPreviousSCF();
static void  XfpNthPreviousSCF( int n );
static int   XfpGetNpXfdata( const char *type );
static void  FixSizeStrcpy( char *to, const char *from, size_t size);

/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make XFP file
 *      ( ->SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XfpMakeSCF(){
  XfpFile   src;
  memset(&src, 0, sizeof(XfpFile));
  if( XfpCreateSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
  if( XfpWriteSCF( &src ) == NCS_FALSE ){
    XfpCloseSCF();
    return( NCS_FALSE );
  }
  XfpCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Add one entry to XFP file
 *   XfpFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XfpAddInfoSCF( XfpFile *add ){
  XfpFile src;
  memset(&src, 0, sizeof(XfpFile));
  if( XfpOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  XfpEndSCF();
    if( XfpWriteSCF( add ) == SCF_ERROR ){
    XfpCloseSCF();
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
    if( XfpWriteSCF( &src ) == SCF_ERROR ){
    XfpCloseSCF();
    return( NCS_FALSE );
  }
  XfpCloseSCF();
  return( NCS_TRUE );
}


/*
 *  Write one entry to n position of XFP file 
 *   XfpFile *add: add informaiton
 *   int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XfpPutInfoSCF( XfpFile *add, int n ){
  if( XfpOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  XfpTopSCF();
  XfpNthNextSCF( n );
  if( XfpWriteSCF( add ) == SCF_ERROR ){
    XfpCloseSCF();
    return( NCS_FALSE );
  }
  XfpCloseSCF();
  return( NCS_TRUE );
}

/*************************************************************************
   functions to setting data
 *************************************************************************/
/*
 *  Set one entry to XFP structure.
 *   XfpFile *src    : storing at the location 
 *  char type       : type of external function
 *  int nval        : number of parameters
 *  double *val     : value of parameters
 *  char *name      : buffer name or file name 
 *  char *time      : buffer name for time information.
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int XfpSetInfoSCF( XfpFile *src, const char *type, double *val, 
                             const char *fname, const char *bname, 
                             const char *time ){
  char xff_num = 0;
  int  i;
  int  nval = 0;
  nval = XfpGetNpXfdata( type );
  switch( *type ){
  case 'R':    /* Ramp   */
    xff_num = SCF_XFP_FCN_RAMP;
    break;
  case 'P':    /* Pulse  */
    xff_num = SCF_XFP_FCN_PULSE;
    break;
  case 'F':    /* File   */
    xff_num = SCF_XFP_FCN_FILE;
    break;
  case 'B':    /* Buffer */
    xff_num = SCF_XFP_FCN_BUFFER;
    break;
#ifdef DSPTEC_EXTENSION
  case 'A':    /* AD3163 */
    xff_num = SCF_XFP_FCN_AD3163;
    break;
  case 'S':    /* Sin */
    xff_num = SCF_XFP_FCN_SIN;
    break;
  case 'C':    /* Sweep */
    xff_num = SCF_XFP_FCN_SWEEP;
    break;
#endif /* DSPTEC_EXTENSION */
  default:
    xff_num = atoi(type);
    break;
  }
  src->func_number = xff_num;

  switch( xff_num ){
  case 1:
  case 2:    /* Pulse & Ramp */
#ifdef DSPTEC_EXTENSION
  case 6:    /* Sin */
  case 7:    /* Sweep */
#endif /* DSPTEC_EXTENSION */
    src->type = SCF_XFP_TYPE_PULSE;
    for( i = 0; i < nval; i++ ){
      src->xfp.rec1.func_para[i] = val[i];
    }
    break;

  case 3:    /* File */
    src->type = SCF_XFP_TYPE_FILE;
    if( fname != NULL ){
      if( strlen( fname ) >= ( SCF_NAME_LEN-1 ) ){
        strncpy( src->xfp.rec2.datf_name, fname, SCF_NAME_LEN-1 );
        src->xfp.rec2.datf_name[SCF_NAME_LEN-1] = '\0';
        fprintf( stderr, "Warnning: file name is too long.\n" );
      }else{
        strcpy( src->xfp.rec2.datf_name, fname );
      }
    } else {
      src->xfp.rec2.datf_name[0] = '\0';
    }
    src->xfp.rec2.buffnum = (int)val[0];
    break;

  case 4:    /* Buffer */
    src->type = SCF_XFP_TYPE_BUFFER;
    src->xfp.rec2.buffnum = (int)val[0];
    src->xfp.rec2.buffdim = (int)val[1];
    /* fileobject name */
    /* >>> */
    /* strcpy( src->xfp.rec2.datf_name, "BUFFER" ); */
    if( fname != NULL ){
      if( strlen( fname ) >= ( SCF_NAME_LEN-1 ) ){
        strncpy( src->xfp.rec2.datf_name, fname, SCF_NAME_LEN-1 );
        src->xfp.rec2.datf_name[SCF_NAME_LEN-1] = '\0';
        fprintf( stderr, "Warnning: file name is too long.\n" );
      }else{
        strcpy( src->xfp.rec2.datf_name, fname );
      }
    } else {
      src->xfp.rec2.datf_name[0] = '\0';
    }
    /* <<< */
    if( bname != NULL ){
      if( strlen( bname ) >= ( SCF_NAME_LEN-1 ) ){
        strncpy( src->xfp.rec2.buffname, bname, SCF_NAME_LEN-1 );
        src->xfp.rec2.buffname[SCF_NAME_LEN-1] = '\0';
        fprintf( stderr, "Warnning: buffer name is too long.\n" );
      }else{
        strcpy( src->xfp.rec2.buffname, bname );
      }
    } else {
      src->xfp.rec2.buffname[0] = '\0';
    }
    src->xfp.rec2.timebuff = (int)val[2];
    if( time != NULL ){
      if( strlen( time ) >= ( SCF_NAME_LEN-1 ) ){
    strncpy( src->xfp.rec2.timename, time, SCF_NAME_LEN-1 );
    src->xfp.rec2.timename[SCF_NAME_LEN-1] = '\0';
    fprintf( stderr, 
             "Warnning: buffer name of timer info. is too long.\n" );
      }else{
        strcpy( src->xfp.rec2.timename, time );
      }
    } else {
      src->xfp.rec2.timename[0] = '\0';
    }
    break;
#ifdef DSPTEC_EXTENSION
  case 5:    /* AD3163 */
    /* TODO : AD3163 extension.. but not implement yet */
    src->type = SCF_XFP_TYPE_AD3163;
    for( i = 0; i < nval; i++ ){
      src->xfp.rec1.func_para[i] = 0.0;
    }
    break;
#endif /* DSPTEC_EXTENSION */
  default:
    src->type = SCF_XFP_TYPE_DEFAULT;
    for( i = 0; i < nval; i++ ){
      src->xfp.rec1.func_para[i] = val[i];
    }
    break;
  }      /* switch */
  return( NCS_TRUE );
}

/*
 *  Set message function parameters
 *   Xfpdata *data   : stucture to store messages
 *      Return          : number of functions - 1
 *                        
 */
int XfpSetDefaultsXfdata( Xfpdata *data ){
  int             sys_func = 4;
  int             nxf, count;
  static int      sys_fpnum[4] = {3, 5, 3, 3};
  static char    *sys_fname[4] = {"RAMP","PULSE","FILE","BUFFER"};
  static char    *sys_fpara[4][5] = {
    {"start_tm", "init_out", "gradient", NCS_EOS, NCS_EOS},
    {"start_tm", "init_out", "height", "width", "period"},
    {"filename", "buff No.", "Index", NCS_EOS, NCS_EOS},
    {"Variable", "buff No.","Time", NCS_EOS, NCS_EOS}
  };
  for( nxf = 0; nxf < sys_func; nxf++ ){
    strcpy(data[nxf].xf_name, sys_fname[nxf]);
    data[nxf].xf_npara = sys_fpnum[nxf];
    for( count = 0; count < data[nxf].xf_npara; count++ ){
      strcpy( data[nxf].xf_para[count], sys_fpara[nxf][count] );
    }
  }
  return( sys_func - 1 );
}

/*************************************************************************
   basic functions 
 *************************************************************************/
/*
 *  Write one entry to XfP file
 *   XfpFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XfpWriteInfoSCF( XinFile *comp, XfpFile *src ){
  int  pos;
  pos = XinSearchInfoSCF( comp );
  if( pos < 0 ){
    return( XfpAddInfoSCF( src ) );
  } else {
    return( XfpPutInfoSCF( src, pos ) );
  }
}

int XfpSetStimSCF( XinFile *sc_xin, XfpFile *sc_xfp ){
  int  pos;
  pos = XinSearchInfoSCF( sc_xin );
  if( pos < 0 ){
    sc_xin->xin.rec3.xfd_number = -pos;
    return( XinAddInfoSCF( sc_xin ) && XfpAddInfoSCF( sc_xfp ) );
  } else {
    sc_xin->xin.rec3.xfd_number = pos+1;
    return( XinPutInfoSCF(sc_xin,pos) && XfpPutInfoSCF(sc_xfp,pos) );
  }
}

/**************************************************************************
       Functions to read simulation condition file
**************************************************************************/

/*
 *  Read one entry from n potition from XFP file
 *   XfpFile *src: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XfpGetInfoSCF( XfpFile *src, int n ){
  if( XfpOpenSCF() == NCS_FALSE ){
    return( 0 );
  }
  XfpTopSCF();
  XfpNthNextSCF( n );
  if( XfpReadSCF( src ) == NCS_FALSE ){
    XfpCloseSCF();
    return( NCS_FALSE );
  }else{
    XfpCloseSCF();
    return( NCS_TRUE );
  }
}

/**************************************************************************
       Functions to delete entry of simulation condition file
**************************************************************************/

/*
 *  Delete one entry from n potition of output information file
 *      int  n      : position   
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XfpDeleteInfoSCF( int n ){
  XfpFile  tmp;
  memset(&tmp, 0, sizeof(XfpFile));
  if( n < 0 ){
    return( NCS_TRUE );
  }
  if( XfpOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  XfpTopSCF();
  XfpNthNextSCF( n );
  do{
    XfpNextSCF();
    if( XfpReadSCF( &tmp ) == NCS_FALSE ){
      XfpCloseSCF();
      return( NCS_FALSE );
    }
    XfpNthPreviousSCF( 2 );
    if( XfpWriteSCF( &tmp ) == NCS_FALSE ){
      XfpCloseSCF();
      return( NCS_FALSE );
    }
  }while( tmp.type != SCF_EOF );
  XfpCloseSCF();
  return( NCS_TRUE );
}


/*
 *  Delete all entry of output information file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int XfpDeleteAllInfoSCF(){
  int     tmp;
  XfpFile src;
  memset(&src, 0, sizeof(XfpFile));
  if( XfpOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  XfpTopSCF();    
  src.type = SCF_EOF;
  if( XfpWriteSCF( &src ) == NCS_FALSE ){
    return( NCS_FALSE );
  }
  while( (tmp=XfpReadSCF( &src )) != NCS_FALSE ){
    if( src.type == SCF_EOF ){
      break;
    }
    src.type = SCF_ERASED;
    XfpPreviousSCF();
    if( XfpWriteSCF( &src ) == NCS_FALSE ){
      XfpCloseSCF();
      return( NCS_FALSE );
    }
  }
  XfpCloseSCF();
  return( tmp );
}

int XfpRemoveInfoSCF( XinFile *src, int aflag ){
  int  dnum=0;
  /* one or more Buffer */
  if( aflag != SCF_ALL_MATCH ){
    dnum = src->xin.rec3.comp_number-1;
    return( XinDeleteInfoSCF(dnum) && XfpDeleteInfoSCF(dnum) );
  } else {
    /* all Buffer */
    return( XinDeleteAllInfoSCF() && XfpDeleteAllInfoSCF() );
  }
}

/**************************************************************************
       Functions to display content of simulation condition file
**************************************************************************/

/*
 * copy strings. if strings which is copied is smaller than size, ...
 *   char *to   : copy to
 *   char *from : copy from
 *   size       : size to copy 
 */
static void FixSizeStrcpy( char *to, const char *from, size_t size ){
  size_t  cnt;
  strcpy(to, from);
  for( cnt = strlen(to); cnt < size; cnt++ ){
    to[cnt] = NCS_SPACE;
  }
  to[cnt] = NCS_EOS;
}

/*
 *  Print list of all entry from XFP file
 *   FILE *disp  : output file
 */
int XfpListupSCF( const char *tmpfile ){
  FILE *disp;
  XfpFile src;
  int  row = 1;
  int  xfd_num = 0, tmp_fnum, tmp_pnum, tmp_npara=0;
  Xfpdata xf_data[10];
  char    tmp_xfpara[SCF_NAME_LEN];

  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;

  memset(&src, 0, sizeof(XfpFile));
  if( XfpSetDefaultsXfdata( xf_data ) < 0 ){
    fprintf( stderr, "Error: Failed to set the SCF data\n" );
    fclose(disp);
    return( 0 );
  };
  
  if( XfpOpenSCF() == NCS_FALSE ){
    fprintf( stderr, "Wanning: Failed to open the SCF\n" );
    fclose(disp);
    return( 0 );
  }
  
  while( XfpReadSCF( &src ) != SCF_ERROR ){
    if( src.type == SCF_EOF ){
      break;
    }
    
    fprintf( disp, "No.%3d ", ++xfd_num);
    tmp_fnum = src.func_number - 1;
    fprintf( disp, " Function  ");
    tmp_npara = xf_data[tmp_fnum].xf_npara;
    for (tmp_pnum = 0; tmp_pnum < tmp_npara; tmp_pnum++) {
      if( (src.func_number != SCF_XFP_FCN_BUFFER) || (tmp_pnum != 1) ){
        FixSizeStrcpy( tmp_xfpara, xf_data[tmp_fnum].xf_para[tmp_pnum], 10 );
        fprintf( disp, "%10s ", tmp_xfpara );
      }
    }
    fprintf( disp, "\n       ");
    row++;
    fprintf( disp, " %8s ", xf_data[tmp_fnum].xf_name);
    switch( src.type ){
    case SCF_XFP_TYPE_PULSE:  /* Pulse & Ramp */
      /* & Sin & Sweep (DSPTEC_EXTENSION) */
      for( tmp_pnum = 0; tmp_pnum < tmp_npara; tmp_pnum++ ){
        fprintf( disp, "  %-8g ", src.xfp.rec1.func_para[tmp_pnum]);
      }
      fprintf( disp, "\n");
      row++;
      break;
    case SCF_XFP_TYPE_BUFFER:  /* Buffer & File */
      if( src.func_number == 3 ){
        fprintf( disp, "   %8s ", src.xfp.rec2.datf_name );
        fprintf( disp, "   %8d ", src.xfp.rec2.buffnum );
        fprintf( disp, "   %8d \n", src.xfp.rec2.buffdim );
      }else{
        fprintf( disp, " %7s[%2d]", src.xfp.rec2.buffname, 
                 src.xfp.rec2.buffdim );
        if( strlen(src.xfp.rec2.timename) == 0 ){
          fprintf( disp, "(%8s)\n", "none" );
        }else{
          fprintf( disp, " %8s \n", src.xfp.rec2.timename );
        }
        }
      row++;
      break;
    case SCF_ERASED:  /* Erased data */
      fprintf( disp, "   ERASED DATA\n" );
      row++;
      break;
#ifdef DSPTEC_EXTENSION
    case SCF_XFP_TYPE_AD3163:
      /* ad3163 (DSPTEC_EXTENSION) */
      for( tmp_pnum = 0; tmp_pnum < tmp_npara; tmp_pnum++ ){
        fprintf( disp, "  %-8g ", src.xfp.rec1.func_para[tmp_pnum]);
      }
      fprintf( disp, "\n");
      row++;
      break;
#endif /* DSPTEC_EXTENSION */
    default:
      break;
    }
  }
  fprintf( disp, "\n" );
  XfpCloseSCF();
  row++;
  fclose(disp);
  return( row );
}


/*
 *  Print list of all entry to exinput information file
 *   FILE *disp  : output file
 */
int XfpListupAllSCF( const char *tmpfile ){
  int row = 2;
  FILE *disp;
  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;
  fprintf(disp, "\nEXTERNAL INPUTS\n");
  fclose(disp);
  row += XinListupSCF( tmpfile );
  row += XfpListupSCF( tmpfile );
  return( row );
}



/* private functions */

/*
 *  Open function of XFP file
 *      Return      : TRUE  : success
 *                    FALSE : failed
 */
static int XfpOpenSCF(){
  const char *path;
  char        mname[SCF_SCFN_PATH_SIZE];
  char        fname[SCF_SCFN_PATH_SIZE];
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_XFP_EXTN );
#else
    sprintf( fname, "%s/%s.%s", path, ModelNameSCFN( mname ), SCF_XFP_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  if( (xfp = _OpenSCF( fname )) == NULL ){
    return( NCS_FALSE );
  }
  return( NCS_TRUE );
}

/*
 *  Open function of XFP file
 *      Return      : TRUE  : sucess
 *                    FALSE : failed
 */
static int XfpCreateSCF(){
  const char *path;
  char        mname[SCF_SCFN_PATH_SIZE];
  char        fname[SCF_SCFN_PATH_SIZE];
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_XFP_EXTN );
#else
    sprintf( fname, "%s/%s.%s", path, ModelNameSCFN( mname ), SCF_XFP_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  if( (xfp = _CreateSCF( fname )) == NULL ){
    return( NCS_FALSE );
  }
  return( NCS_TRUE );
}

/*
 *  Close functions of XFP file
 */
static int XfpCloseSCF(){
  return( _CloseSCF( xfp ) );
}

/*
 *  Write one entry to functions of exinput information file
 *   XfpFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int XfpWriteSCF( XfpFile *src ){
  if( _WriteSCF( xfp, (void *)src, sizeof(XfpFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Read one entry from functions of XFP file
 *   ObsFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int XfpReadSCF( XfpFile *src ){
  if( _ReadSCF( xfp, (void *)src, sizeof(XfpFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Set file pointer top of file
 */
static void XfpTopSCF(){
  fseek( xfp, 0L, SEEK_SET );
}

/*
 *  Set file pointer end of file
 *   FILE    *fp : file pointer of file
 */
static void XfpEndSCF(){
  XfpFile tmp;
  XfpTopSCF();
  do{
    XfpReadSCF( &tmp );
  }while( tmp.type != SCF_EOF );
  XfpPreviousSCF();
}

/*
 *  advance file pointer
 */
static void XfpNextSCF(){
  fseek( xfp, (long)sizeof(XfpFile), SEEK_CUR );
}

/*
 *  advance nth file pointer
 */
static void XfpNthNextSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    XfpNextSCF();
  }
}

/*
 *  retreat file pointer
 */
static void XfpPreviousSCF(){
  fseek( xfp, (long)sizeof(XfpFile)*(-1), SEEK_CUR );
}

/*
 *  Set message function parameters
 *   Xfpdata *data   : stucture to store messages
 *      Return          : number of functions - 1
 *                        
 */
static int XfpGetNpXfdata( const char *type ){
  int num;
  switch( *type ){
  case 'R':    /* Ramp   */
    num = 3;
    break;
  case 'P':    /* Pulse  */
    num = 5;
    break;
  case 'F':    /* File   */
    num = 3;
    break;
  case 'B':    /* Buffer */
    num = 3;
    break;
#ifdef DSPTEC_EXTENSION
  case 'A':    /* AD3163 */
    num = 1;
    break;
  case 'S':    /* Sin */
    num = 3;
    break;
  case 'C':    /* Sweep */
    num = 4;
    break;
#endif /* DSPTEC_EXTENSION */
  default:
    num = 5;
    break;
  }
  return( num );
}

/*
 *  retreat nth file pointer
 */
static void XfpNthPreviousSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    XfpPreviousSCF();
  }
}

#ifdef __cplusplus
}
#endif
