/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************
  
       Function Library to Control NCS Simulation Condition File 

           for information of OUTPUT & OBSERVABLE variables

    $Id: scfout.c,v 1.3 2004/07/28 08:21:36 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <fcntl.h>

#define MOD_NCS_LIBNCSC_EXPORTS
#include "libncsc.h"

#include "scf.h"

#ifdef __cplusplus
extern "C" {
#endif

/* File pointer for simulaiton condition file of output information */
static FILE *out;

static int   OutOpenSCF();
static int   OutCreateSCF();
static int   OutCloseSCF();
static int   OutWriteSCF( OutFile *src );
static int   OutReadSCF( OutFile *src );
static void  OutTopSCF();
static void  OutEndSCF();
static void  OutNextSCF();
static void  OutNthNextSCF( int );
static void  OutPreviousSCF();
static void  OutNthPreviousSCF( int );
static int   OutListupVarSCF( FILE *disp );
static int   OutListupMonitorSCF( FILE *disp, OutFile *src );
static int   OutListupEnvSCF( FILE *disp );


/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make output information file
 *      ( SCF_OUT_NO->SCF_OUT_BUFF_MAX->SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int OutMakeSCF(){
  OutFile   src;
  memset(&src,0,sizeof(OutFile));
  if( OutCreateSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  src.type = SCF_OUT_NO;
  if( OutWriteSCF( &src ) == NCS_FALSE ){
    OutCloseSCF();
    return( NCS_FALSE );
  }
  src.type = SCF_OUT_BUFF_SIZE;
  src.out.rec4.max_number = SCF_OUT_BUFF_MAX;
  /*OutTopSCF( fp );*/
  if( OutWriteSCF( &src ) == NCS_FALSE ){
    OutCloseSCF();
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
  if( OutWriteSCF( &src ) == NCS_FALSE ){
    OutCloseSCF();
    return( NCS_FALSE );
  }
  
  OutCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Add one entry to end of output information file
 *   OutFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int OutAddInfoSCF( OutFile *add ){
  OutFile src;
  memset(&src, 0, sizeof(OutFile));
  if( OutOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  OutEndSCF();
    if( OutWriteSCF( add ) == SCF_ERROR ){
    OutCloseSCF();
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
    if( OutWriteSCF( &src ) == SCF_ERROR ){
    OutCloseSCF();
    return( NCS_FALSE );
  }
  OutCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Write one entry to n position of output information file 
 *   OutFile *add: add informaiton
 *   int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int OutPutInfoSCF( OutFile *add, int n ){
  if( OutOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  OutTopSCF();
  OutNthNextSCF( n );
    if( OutWriteSCF( add ) == SCF_ERROR ){
    OutCloseSCF();
    return( NCS_FALSE );
  }
  OutCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Set one entry to output information structure.
 *   OutFile *src    : storing at the location 
 *   int buff_id     : buffer ID to set information.
 *   char *buff_name : buffer name to set information.
 *   int buff_dim    : index of buffer to set information.
 *   char *mdl_name  : module name to set information.
 *   int comp_no     : component number to set information.
 *   int type        : type of output variable. 
 *                    ( output, input, observable )
 *   char *obs_name  : name of variable in model file.
 *   Return          : TRUE  : success.
 *                     FALSE : failed.
 */
int OutSetInfoSCF( OutFile *src, int buf_id, const char *buf_name, 
                             int buf_dim, const char *mdl_name, int comp_no,
                             int type, const char *obs_name ){
  src->out.rec3.buff_num = buf_id;
  if( buf_name != NULL ){
    if(strlen(buf_name) >= (NCS_BUF_NAME_LEN-1)){
      strncpy(src->out.rec3.buf_name, buf_name, NCS_BUF_NAME_LEN-1);
      src->out.rec3.buf_name[NCS_BUF_NAME_LEN-1] = '\0';
      fprintf(stderr, "Warnning: buffer name is too long.\n");
    }else{
      strcpy(src->out.rec3.buf_name, buf_name);
    }
  } else {
    src->out.rec3.buf_name[0] = '\0';
  }
  src->out.rec3.buff_dim = buf_dim;
  if( mdl_name != NULL ){
    if( strlen( mdl_name ) >= (SCF_NAME_LEN-1) ){
      strncpy( src->out.rec3.mdl_name, mdl_name, SCF_NAME_LEN-1 );
      src->out.rec3.mdl_name[SCF_NAME_LEN-1] = '\0';
      fprintf( stderr, "Warnning: module name is too long.\n" );
    }else{
      strcpy( src->out.rec3.mdl_name, mdl_name );
    }
  } else {
    src->out.rec3.mdl_name[0] = '\0';
  }
   
  src->out.rec3.comp_number = comp_no;
  src->type = (char)('0'+type);
  if( src->type == SCF_OUT_BUFF_OBS ){
    if( obs_name != NULL ){
      if( strlen( obs_name ) >= (SCF_NAME_LEN)-1 ){
        strncpy( src->out.rec3.obs_name, obs_name, SCF_NAME_LEN );
        src->out.rec3.obs_name[SCF_NAME_LEN-1] = '\0';
        fprintf( stderr, "Warnning: variable name is too long.\n" );
      }else{
        strcpy( src->out.rec3.obs_name, obs_name );
      }
    } else {
      src->out.rec3.obs_name[0] = '\0';
    }
  }
  return( NCS_TRUE );
}

/*
 *  Write one entry to output information file
 *   OutFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int OutWriteInfoSCF( OutFile *src ){
  int pos;
  pos = OutSearchInfoSCF( src );
  if( pos < 0 ){
    return( OutAddInfoSCF( src ) );
  } else {
    return( OutPutInfoSCF( src, pos ) );
  }
}



/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of output information file
 *   OutFile *src: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int OutGetInfoSCF( OutFile *src, int n ){
  if( OutOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  OutTopSCF();
  OutNthNextSCF( n );
    if( OutReadSCF( src ) == NCS_FALSE ){
    OutCloseSCF();
    return( NCS_FALSE );
  }else{
    OutCloseSCF();
    return( NCS_TRUE );
  }
}

/*
 *  Search same entry to observable information file
 *   ObsFile *cmp: compare entry
 *      Return      :  n > 0  : potion of the same entry
 *                     n < 0  : no match. n is end of entry.
 *                     0      : ERROR
 */
int OutSearchInfoSCF( OutFile *cmp ){
  OutFile src;
  int     n = 0;
  memset(&src, 0, sizeof(OutFile));
  if( OutOpenSCF() == NCS_FALSE ){
    return( 0 );
  }
  OutTopSCF();
  while( OutReadSCF( &src ) != SCF_ERROR ){
    if( src.type == SCF_EOF ){
      OutCloseSCF();  
      return( -(n+1) );
      break;
    }
    if( strcmp(cmp->out.rec3.buf_name,src.out.rec3.buf_name) == 0 ){
      if( cmp->out.rec3.buff_dim == SCF_ALL_MATCH ){
        break;
      } else {
        if( cmp->out.rec3.buff_dim == src.out.rec3.buff_dim ){
          break;
        }
      }
    }
    n++;
  }
  OutCloseSCF();  
  return( n );
}

/**************************************************************************
       Functions to delete entry of simulation condition file
 **************************************************************************/

/*
 *  Delete one entry from n potition of output information file
 *      int  n      : position   
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int OutDeleteInfoSCF( int n ){
  OutFile  tmp;
  memset(&tmp, 0, sizeof(OutFile));
  if( OutOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  OutTopSCF();
  OutNthNextSCF( n );
  do{
    OutNextSCF();
    if( OutReadSCF( &tmp ) == NCS_FALSE ){
      OutCloseSCF();
      return( NCS_FALSE );
    }
    OutNthPreviousSCF( 2 );
    if( OutWriteSCF( &tmp ) == NCS_FALSE ){
      OutCloseSCF();
      return( NCS_FALSE );
    }
  }while( tmp.type != SCF_EOF );
  OutCloseSCF();
  return( NCS_TRUE );
}


/*
 *  Delete all entry of output information file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int OutDeleteAllInfoSCF(){
  OutFile src;
  int     tmp;
  memset(&src, 0, sizeof(OutFile));
  if( OutOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  OutTopSCF();
  OutNextSCF();   /* skip information of BUFF MAX */
  src.type = SCF_EOF;
  if( OutWriteSCF( &src ) == NCS_FALSE ){
    return( NCS_FALSE );
  }
  while( (tmp=OutReadSCF( &src )) != NCS_FALSE ){
    if( src.type == SCF_EOF ){
      break;
    }
    OutPreviousSCF();
    src.type = SCF_ERASED;
    if( OutWriteSCF( &src ) == NCS_FALSE ){
      OutCloseSCF();
      return( NCS_FALSE );
    }
  }
  OutCloseSCF();
  return( tmp );
}

int OutRemoveInfoSCF( OutFile *src, int flag ){
  int  dnum;
  int  dflag = 0;
  /* one or more Buffer */
  if( flag != SCF_ALL_MATCH ){
    while( (dnum = OutSearchInfoSCF( src )) > 0 ){
      dflag++;
      if( OutDeleteInfoSCF( dnum ) == NCS_FALSE ){
        return( NCS_FALSE );
      }
    }
    if( dflag == 0 ){
      return( NCS_FALSE );
    } else {
      return( NCS_TRUE );
    }
  }  
  /* all Buffer */
  return( OutDeleteAllInfoSCF() );
}


/*
 *  Print list of all entry of output information file
 *   FILE *disp  : output file
 *      Return      : success : number of lines of lists
 *                    0       : failed
 */
int OutListupSCF( const char *tmpfile ){
  FILE *disp;
  int  row = 1;

  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;

  fprintf( disp, "\nOUTPUT\n");
  if( OutOpenSCF() == NCS_FALSE ){
    fclose(disp);
    return( 0 );
  }
  row += OutListupVarSCF( disp );
  row += OutListupEnvSCF( disp );
  OutCloseSCF();
  fclose(disp);
  return( row );
}


/* TODO : i want to delete this function */
int OutGetVarIndexSCF( const char *var_name ){
  int     num = -1;
  OutFile src;
  int             flag = 0;
  memset(&src, 0, sizeof(OutFile));
  
  if( OutOpenSCF() == NCS_FALSE ){
    return( num );
  }
  
  OutTopSCF();
  OutNextSCF();   /* skip information of BUFF MAX */
  
  while( OutReadSCF( &src ) != NCS_FALSE && src.type != SCF_EOF ){
    if( src.type != SCF_ERASED && src.type != SCF_OUT_NO ){
      num++;
      if(strcmp(var_name, src.out.rec3.buf_name) == 0) {
        flag = 1;
        break;
      }           
    }
  }       /* while */
  OutCloseSCF();
  
  if(flag)
    return( num );
  else
    return( -1 );
}

/* TODO : i want to delete this function */
int OutGetVarSizeSCF( const char *var_name ){
  int     num = -1;
  OutFile src;
  int             flag = 0;
  memset(&src, 0, sizeof(OutFile));
  
  if( OutOpenSCF() == NCS_FALSE ){
    return( num );
  }
  OutTopSCF();
  OutNextSCF();   /* skip information of BUFF MAX */
  
  while( OutReadSCF( &src ) != NCS_FALSE && src.type != SCF_EOF ){
    if( src.type != SCF_ERASED && src.type != SCF_OUT_NO ){
      num++;
      if(strcmp(var_name, src.out.rec3.buf_name) == 0) {
        flag += 1;
      }           
    }
  }       /* while */
  OutCloseSCF();
  
  if(flag)
    return( flag );
  else
    return( -1 );
}

/**************************************************************************
    Basic functions to control simulation condition file
 **************************************************************************/

/*
 *  Open output information file to read and write
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int OutOpenSCF(){
  const char *path;
  char     mname[SCF_SCFN_PATH_SIZE];
  char     fname[SCF_SCFN_PATH_SIZE];
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_OUT_EXTN );
#else
    sprintf( fname, "%s/%s.%s", 
             path, ModelNameSCFN( mname ), SCF_OUT_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }

  if( (out = _OpenSCF( fname )) == NULL ){
    return( NCS_FALSE );
  } else {
    return( NCS_TRUE );
  }
}

/*
 *  Create output information file
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int OutCreateSCF(){
  const char *path;
  char     mname[SCF_SCFN_PATH_SIZE];
  char     fname[SCF_SCFN_PATH_SIZE];
  
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_OUT_EXTN );
#else
    sprintf( fname, "%s/%s.%s", 
             path, ModelNameSCFN( mname ), SCF_OUT_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }

  if( (out = _CreateSCF( fname )) == NULL ){
    return( NCS_FALSE );
  } else {
    return( NCS_TRUE );
  }
}

/*
 *  Close output information file
 */
static int OutCloseSCF(){
  return( _CloseSCF( out ) );
}

/*
 *  Write one entry to output information file
 *   OutFile *src: reading at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE  : success.
 *                    FALSE : failed.
 */
static int OutWriteSCF( OutFile *src ){
  if( _WriteSCF( out, (void *)src, sizeof(OutFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Read one entry from output information file
 *   OutFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int OutReadSCF( OutFile *src ){
  if( _ReadSCF( out, (void *)src, sizeof(OutFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Set file pointer top of file
 */
static void OutTopSCF(){
  fseek( out, 0L, SEEK_SET );
  OutNextSCF();
}

/*
 *  Set file pointer end of file
 */
static void OutEndSCF(){
  OutFile  tmp;
  memset(&tmp,0,sizeof(OutFile));
  OutTopSCF();
  while( OutReadSCF( &tmp ) != NCS_FALSE && tmp.type != SCF_EOF );
  fseek( out, (long)sizeof(OutFile)*(-1), SEEK_CUR );
  /*   fseek( out, (long)sizeof(OutFile)*(-1), SEEK_END );*/
}

/*
 *  advance file pointer 
 */
static void OutNextSCF(){
  fseek( out, (long)sizeof(OutFile), SEEK_CUR );
}

/*
 *  advance nth file pointer 
 */
static void OutNthNextSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    OutNextSCF();
  }
}

/*
 *  retreat file pointer 
 */
static void OutPreviousSCF(){
  long offset;
  offset = sizeof(OutFile);
  fseek( out, -offset, SEEK_CUR );
}

/*
 *  retreat nth file pointer 
 */
static void OutNthPreviousSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    OutPreviousSCF();
  }
}

/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/

/*
 *  Print list of outupt variables 
 *   FILE *disp  : output file
 */
static int OutListupVarSCF( FILE *disp ){
  int     num = 0;
  int  row = 0;
  int   flag;
  OutFile  src;
  memset(&src, 0, sizeof(OutFile));

  OutTopSCF();
  OutNextSCF();  /* skip information of BUFF MAX */
  
  while( OutReadSCF( &src ) != NCS_FALSE && src.type != SCF_EOF ){
    flag = 0;
    if( num == 0 ){
      fprintf( disp, "  Variable       Model Variable\n");
      row++;
    }
    num++;
    if( src.type != SCF_ERASED && src.type != SCF_OUT_NO ){
      fprintf( disp, "%10s[%3d]   ",
               src.out.rec3.buf_name, src.out.rec3.buff_dim );
      switch( src.type ){
      case SCF_OUT_BUFF_OUT:
        fprintf( disp, "OUTPUT   of ");
        break;
      case SCF_OUT_BUFF_EXIN:
        fprintf( disp, "EX.INPUT of ");
        break;
      case SCF_OUT_BUFF_OBS:
        fprintf( disp, "%-8s of ", src.out.rec3.obs_name);
        break;
      default:
        flag = 1;
        break;
      }  
      if( flag == 0 ){
        fprintf( disp, "%s[%d]\n", src.out.rec3.mdl_name, 
                 src.out.rec3.comp_number);
        row++;
      }
    }
  }
  if( num == 0 ){
    fprintf( disp, "  No information\n" );
    row++;
  }
  return( row );
}

/*
 *  Print of variables of monitor mode 
 *   FILE *disp  : output file
 */
static int OutListupMonitorSCF( FILE *disp, OutFile *src ){
  int  count;
  char    moni_mode;
  int   row = 1;
  fprintf( disp, "\n MONITER MODE : " );
  switch( src->out.rec1.monitor_mode ){
  case SCF_OUT_MONITOR_NO_CRT:
    fprintf( disp, "NO CRT OUTPUT\n" );
    break;
  case SCF_OUT_MONITOR_CRT:
    fprintf( disp, "VALUES ONLY\n" );
    break;
  default:
    fprintf( disp, "NO SET\n" );
    break;
  }
  row++;
  moni_mode = src->out.rec1.monitor_mode;
  if( moni_mode == SCF_OUT_MONITOR_CRT ){
    for( count = 0; count < SCF_OUT_MONITOR_MAX; count++ ){
      if(src->out.rec1.vmoni_buffnum[count] != 0){
        fprintf( disp, " VALUE%d : %d(%d)\n", (count + 1), 
                 src->out.rec1.vmoni_buffnum[count],
                 src->out.rec1.vmoni_buffdim[count]);
      }else{
        fprintf( disp, " VALUE%d :\n", (count + 1));
      }
      row++;
    }    
  }    
  return( row );
}

/*
 *  Print list of environment variables 
 *   FILE *disp  : output file
 */
static int OutListupEnvSCF( FILE *disp ){
  OutFile  src;
  int  row = 0;
  memset(&src, 0, sizeof(OutFile));
  OutTopSCF();
  while( ( OutReadSCF( &src ) != NCS_FALSE )
         && ( src.type >= OUT_MONITOR_TYPE )
         && ( src.type != SCF_EOF ) ) {
    switch( src.type ){
    case SCF_OUT_MONITOR:
      row += OutListupMonitorSCF( disp, &src );
      break;
    case SCF_OUT_FILE:
      if( src.out.rec2.strf_flag == '1' ){
        fprintf( disp, " DATA FILE : %s\n", src.out.rec2.strf_name);
        row++;
      }
      break;
    case SCF_OUT_BUFF_SIZE:
      fprintf( disp, "Max of Number of Output = %d\n", 
               src.out.rec4.max_number);
      row++;
      break;
    default:
      break;  
    }  
  }
  fprintf( disp, "\n" );
  row++;
  return( row );
}

#ifdef __cplusplus
}
#endif
