/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************
  
       Function Library to Control NCS Simulation Condition File 
  	           for name of model source file etc.

        $Id: scfn.c,v 1.2 2004/04/15 16:57:53 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef   STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# ifdef  HAVE_STDLIB_H
#  include <stdlib.h>
# endif
# ifdef  HAVE_MALLOC_H
#  include <malloc.h>
# endif
# ifdef  HAVE_STRING_H
#  include <string.h>
# endif
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#define MOD_NCS_LIBNCSC_EXPORTS
#include "libncsc.h"

#ifdef WIN32
# define  FOPEN_R "rb"
# define  FOPEN_W "wb"
#else
# define  FOPEN_R "r"
# define  FOPEN_W "w"
#endif

#ifdef HAVE_IO_H
# include <io.h>
# define access(x,y) _access(x,y)
# define F_OK 00
# define W_OK 02
# define R_OK 04
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifndef HAVE_STRDUP
static char *strdup(const char* str){
  char *ret;
  size_t len = strlen(str);
  ret = (char*)malloc(len+1);
  if(ret) strcpy(ret,str);
  return ret;
}
#endif

static void  MakeSCFN();
static char *ReadSCFN( char *mfile, char *hfile, char *lfile );
static int   WriteSCFN(const char *mfile, const char *header,
                       const char *lib );
static int   CheckAccessSCF( const char *fname );

/*
 * Read file name of simulation condition file from SCFN file.
 *	char 	*path  : directory path of SCFN file
 *	char 	*mfile : name of model source file
 *	char 	*hfile : name of header file for library defined by user.
 *	char 	*lfile : name of library file defined by user.
 */
char *GetSCFN( char *mfile, char *hfile, char *lfile ){
  return( ReadSCFN( mfile, hfile, lfile ) );
}

/*
 * Read model source name of simulation condition file from SCFN file.
 *	char 	*model : name of model source file
 */
char *ModelNameSCFN( char *model ){
  char    *pointer;
  char	mfile[SCF_SCFN_PATH_SIZE];
  char	buff1[SCF_SCFN_PATH_SIZE], buff2[SCF_SCFN_PATH_SIZE];
  
  if( ReadSCFN( mfile, buff1, buff2 ) != NULL ){
    if( ( pointer = strrchr( mfile, '/' ) ) != NULL ){
      strcpy( model, pointer+1 );
    }else{
      strcpy( model, mfile );
    }
  } else {
    return( NULL );
  }
  
  return( model );
}


/*
 * Write modele source file name of simulation condition file to SCFN file.
 *	char 	*path  : directory path of SCFN file
 *	char 	*mfile : name of model source file
 */
int SetSCFN( const char *mfile, const char *header, 
                       const char *lib ){
  return( WriteSCFN( mfile, header, lib ) );
}


/*
 * get and set modele source file name to SCFN file.
 *	char 	*path  : directory path of SCFN file
 *	char 	*mfile : name of model source file
 */
char *GetAllFilesSCFN( char *marg, char *harg, char *larg ){
  char 	mfile[SCF_SCFN_PATH_SIZE];
  char 	hfile[SCF_SCFN_PATH_SIZE];
  char 	lfile[SCF_SCFN_PATH_SIZE];
  
  /* there is no argument */
  if( GetSCFN( mfile, hfile, lfile ) == NULL ){
    if( mfile == '\0' ){ 
      return( NULL );
    } else {
#ifdef HAVE_SNPRINTF
      snprintf( mfile, SCF_SCFN_PATH_SIZE, "%s.%s", marg, SCF_MODEL_EXTN );
#else
      sprintf( mfile, "%s.%s", marg, SCF_MODEL_EXTN );
#endif
    }
  }
  if( hfile == '\0' ){
    if( harg[0] != '\0' ){
      strcpy( hfile, harg );
      if( larg[0] != '\0' ){
        strcpy( lfile, larg );
      }
    }
  }

  /* check access of model file */
  if( access( mfile, F_OK ) == -1 ){
    fprintf( stderr, "Model file name = \"%s\" \n", mfile );
    return( NULL );
  }
	
  if( ( CheckAccessSCF( mfile ) == NCS_FALSE ) ||
      ( CheckAccessSCF( hfile ) == NCS_FALSE ) ||
      ( CheckAccessSCF( hfile ) == NCS_FALSE ) ){ 
    return( NULL );
  }
  
  /* set to SCF */
  strcpy( mfile, marg );
  strcpy( hfile, harg );
  strcpy( lfile, larg );
  if( SetSCFN( marg, harg, larg ) == NCS_FALSE ){
    return( NULL );
  }
  
  return( marg );
}

/*
 * Write modele source file name of simulation condition file to SCFN file.
 * for nasign command.
 *     char    *mfile : name of model source file
 *     char    *file1 : file name
 *     char    *file2 : file name
 */
int SignSCFN( const char *mfile, const char *file1, const char *file2 ){
  char    *pointer;
  char    *model;
  char    *header;
  char    *lib;
  int     flag;
  
  /*** model file name ***/
  model = strdup( mfile );
  pointer = strrchr( model, '.' );
  if( (pointer != NULL) && (strcmp( pointer, SCF_MODEL_EXTN ) == 0) ){
    *pointer = '\0';
  }
  
  /*** file name of header and library ***/
  if( (file1 != NULL) && (strlen(file1)!=0)){
    /* argument 2 was seted */
    if( (file2 == NULL) || (strlen(file2)==0)){
      /* argument 3 was seted */
      pointer = (char *)strrchr( file1, '.' );
      if( strcmp( pointer+1, SCF_HDF_EXTN ) != 0 ){
        /* argument1 was a name of header file */
        lib = strdup( file1 );
        header = strdup( "\0" );
        /*header_name[0] = NULL;*/
      }else{
        /* argument1 was a name of library */
        header = strdup( file1 );
        lib = strdup( "\0" );
        /*strcpy( header_name, file_name1 );
          lib_name[0] = NULL;*/
      }
    }else{
      header = strdup( file1 );
      lib = strdup( file2 );
    }
  }else{
    header = strdup( "\0" );
    lib = strdup( "\0" );
    /*header_name[0] = NULL;
      lib_name[0] = NULL;*/
  }
  
  flag = WriteSCFN( model, header, lib );
  free( model );
  free( header );
  free( lib );
  return( flag );
}

/*
 *  Print list of all entry file names of simulation condition file.
 *     FILE *disp  : output file
 *      Return      : success : number of lines of lists
 *                    0       : failed
 */
int ListupSCFN( const char *tmpfile ){
  FILE *disp;

  int  row = 0;
  char mfile[SCF_SCFN_PATH_SIZE];
  char hfile[SCF_SCFN_PATH_SIZE];
  char lfile[SCF_SCFN_PATH_SIZE];
  
  if( ReadSCFN( mfile, hfile, lfile ) == NULL ){
    return( 0 );
  }

  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;

  if( strlen( mfile ) == 0 ){
    fprintf( disp, "Current Simulation Condition File Name: none\n" );
  }else{
    fprintf( disp, "Current Simulation Condition File Name: %s\n", mfile );
  }
  row++;
  
  if( strlen( hfile ) > 1 ){
    fprintf( disp, "User's Library Header File Name : %s\n", hfile );
    row++;
  }
  
  if( strlen( lfile ) > 1 ){
    fprintf( disp, "User's Library File Name : %s\n", lfile );
    row++;
  }

  fclose(disp);
  return( row );
}

static void MakeSCFN(){
  FILE       *fp;
  const char *path;
  char        buff[SCF_SCFN_PATH_SIZE];
  
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf(buff, SCF_SCFN_PATH_SIZE, "%s/%s", path, SCF_SCFN_NAME);
#else
    sprintf(buff, "%s/%s", path, SCF_SCFN_NAME);
#endif
  } else {
    return;
  }
  if( (fp=fopen(buff, FOPEN_W)) == NULL ){
    return;
  }
  
  buff[0] = '\n';
  buff[1] = '\0';
  fputs( buff, fp );
  fputs( buff, fp );
  fputs( buff, fp );
  putc( NCS_EOS, fp );
  
  fclose( fp );
}

/*
 * Read file name of simulation condition file from SCFN file.
 *	char 	*path  : directory path of SCFN file
 *	char 	*mfile : name of model source file
 *	char 	*hfile : name of header file for library defined by user.
 *	char 	*lfile : name of library file defined by user.
 */
static char *ReadSCFN( char *mfile, char *hfile, char *lfile ){
  FILE	*fp;
  const char *path;
  char	buff[SCF_SCFN_PATH_SIZE];
  int	i, j;
  
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf(buff, SCF_SCFN_PATH_SIZE, "%s/%s", path, SCF_SCFN_NAME);
#else
    sprintf(buf, "%s/%s", path, SCF_SCFN_NAME);
#endif
  } else {
    return( NULL );
  }
  if( access(buff, R_OK ) == -1 ){
    MakeSCFN();
    mfile[0] = '\0';
    hfile[0] = '\0';
    lfile[0] = '\0';
    return( mfile );
  }
  if( (fp=fopen(buff, FOPEN_R)) == NULL ){
    return( NULL );
  }else{
    i = 0; 
    j = 0;
    mfile[0] = '\0';
    hfile[0] = '\0';
    lfile[0] = '\0';
    while( (buff[i++]=getc(fp)) != EOF )
      if( buff[i-1] == '\n' ){
        buff[i-1] = '\0';
        switch( j ){
        case 0:
          strcpy( mfile, buff ); j++; i=0; break;
        case 1:
          strcpy( hfile, buff ); j++; i=0; break;
        case 2:
          strcpy( lfile, buff ); j++; i=0; break;
        default:
          break;
        }
      }
    fclose( fp );
  }
  return( mfile );
}

/*
 * Write modele source file name of simulation condition file to SCFN file.
 *	char 	*path  : directory path of SCFN file
 *	char 	*mfile : name of model source file
 */
static int WriteSCFN( const char *mfile, const char *header, const char *lib ){
  FILE       *fp;
  const char *path;
  char	      buff[SCF_SCFN_PATH_SIZE];
  char	     *model_name=NULL, *header_name=NULL, *lib_name=NULL;
  
  if( mfile != NULL ){
    strcpy( buff, mfile );
    strcat( buff, "\n\0" );
    model_name = strdup( buff );
  } else {
    return( NCS_FALSE );
  }
  if( header != NULL ){
    strcpy( buff, header );
    strcat( buff, "\n\0" );
    header_name = strdup( buff );
  } else {
    buff[0] = '\n';
    buff[1] = '\0';
    header_name = strdup( buff );
  }
  if( lib != NULL ){
    strcpy( buff, lib );
    strcat( buff, "\n\0" );
    lib_name = strdup( buff );
  } else {
    buff[0] = '\n';
    buff[1] = '\0';
    lib_name = strdup( buff );
  }

  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf(buff, SCF_SCFN_PATH_SIZE, "%s/%s", path, SCF_SCFN_NAME);
#else
    sprintf(buf, "%s/%s", path, SCF_SCFN_NAME);
#endif
  } else {
    return( NCS_FALSE );
  }
  if( (fp=fopen(buff, FOPEN_W)) == NULL ){
    return( NCS_FALSE );
  }
  fputs( model_name, fp );
  fputs( header_name, fp );
  fputs( lib_name, fp );
  putc( NCS_EOS, fp );
  
  fclose( fp );

  free( model_name );
  free( header_name );
  free( lib_name );
  return( NCS_TRUE );
}

/*
 *  accessable check files
 *	char *fname : file (path) name
 *	RETURN      : TRUE  : OK
 *		    : FALSE : failed
 */
static int CheckAccessSCF( const char *fname ){
  if( fname[0] != '\0' ){
    if( access( fname, R_OK ) == -1 ){
      fprintf( stderr, "File name = \"%s\" \n", fname );
      return( NCS_FALSE );
    }
  }
  return( NCS_TRUE );
}

#ifdef __cplusplus
}
#endif
