/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: libncsc.h,v 1.4 2004/09/28 04:18:31 orrisroot Exp $ */
 /*****************************************************************************

         Header file of common functions for NCS Module to SATELLITE

*****************************************************************************/

#ifndef _LIBNCSC_H_
#define _LIBNCSC_H_

#ifdef DLLEXPORT
# undef DLLEXPORT
#endif
#ifdef DLLIMPORT
# undef DLLIMPORT
#endif
#ifdef DECLSPEC
# undef DECLSPEC
#endif
#ifdef WIN32
#  define DLLEXPORT __declspec(dllexport)
#  define DLLIMPORT __declspec(dllimport)
# ifdef MOD_NCS_LIBNCSC_EXPORTS
#  define DECLSPEC DLLEXPORT
# else
#  define DECLSPEC DLLIMPORT
# endif
#else
# define DLLEXPORT
# define DLLIMPORT
# define DECLSPEC
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef WIN32
# define NCS_MODEL_DYNAMIC_LOADING
#endif

/* common definition */
/* ncs simulator working directory */
#define NCS_TEMPDIR_ENV       "SL4_NCS_TEMPDIR"

/* FLAG */
#define NCS_SUCCESS      (0)
#define NCS_ERROR        (1)
#define NCS_ERROR0       (0)
#define NCS_FLAG_ON      ('1')
#define NCS_FLAG_OFF     ('0')

#define NCS_TRUE          1
#define NCS_FALSE         0

#define NCS_MODE0        (0)

#define NCS_EQUIVALENT   (0)  /* for strcmp() */

/* CHARACTER CODE */
#define NCS_EOS          0x00
#define NCS_SPACE        0x20

#ifndef PI
# ifndef M_PI
#  define PI (3.14159265358979323846)
# else
#  define PI M_PI
# endif
#endif

/* scfout.c */
#define NCS_MONI_NVAR     4  /* maximum number of monitering variable */
#define NCS_DATA_FNAME   22  /* length of data file name + EOS        */
#define NCS_BUF_NAME_LEN 32  /* length of buffer name */
#define OUT_MONITOR_TYPE '4'  /* moniter mode                */


/* size of file name */
#define NCS_FNAME_WD    512  /* size of file name with directory */


/* ************************* lib/scf.c *************************** */
/*     Function Library to Control NCS Simulation Condition File   */
/*                 for low layered file access                     */
/* *************************************************************** */

#define  SCF_ERROR         (0)
#define  SCF_SUCCESS       (1)
#define  SCF_ALL_MATCH    (-1)
#define  SCF_EOF         ('\0')
#define  SCF_ERASED       ('0')
#define  SCF_NAME_LEN      (33)
#define  SCF_NAME_WID       36 

int         SetWorkDirSCF( const char *path );
int         CleanWorkDirSCF();
const char *GetWorkDirSCF();

/* ************************* lib/scfn.c *************************** */
/*      Function Library to Control NCS Simulation Condition File   */
/*                 for name of model source file etc.               */
/* **************************************************************** */

/* simulation condition file name file     */
#define  SCF_SCFN_NAME           "_ncs_sfcn.dat"
#define  SCF_MODEL_EXTN          "mdl"
#define  SCF_HDF_EXTN            "h"
#define  SCF_SCFN_PATH_SIZE      (255)

char *GetSCFN( char *mfile, char *hfile, char *lfile );
char *ModelNameSCFN( char *model );
int   SetSCFN( const char *mfile, const char *header, 
                        const char *lib );
char *GetAllFilesSCFN( char *marg, char *harg, char *larg );
int   SignSCFN( const char *mfile, const char *file1,
                         const char *file2 );
int   ListupSCFN( const char *tmpfile );

/* ************************* lib/scfprm.c *************************** */
/*       Function Library to Control NCS Simulation Condition File    */
/*              for information of PARAMETER variables                */
/* ****************************************************************** */

/* file extension of simulation condition */
#define  SCF_PRM_EXTN            "prm"
/* type information: scf_mat.type */
#define  SCF_PRM_TYPE_NORMAL     ('1')  /* normal type       */
#define  SCF_PRM_TYPE_INT        ('2')  /* normal type       */
/* flag to display list of parameters */
#define  SCF_PRM_ALL_MDL         "ALL"
#define  SCF_PRM_ALL_MDL_OLD    "-ALL"

/* PARAMETER FILE */
typedef struct scf_prm {
  char   type;                              /*  1 byte  */
  char   mdl_name[SCF_NAME_LEN];            /* 33 bytes */
  char   prm_name[SCF_NAME_WID];            /* 36 bytes */
  union  prm_data {
    double  prm_real;                       /*  4 bytes */
    long    prm_intg;                       /*  4 bytes */
  } prm_val;                                /*  4 bytes */
} PrmFile;                                  /* 82 bytes */

int  PrmMakeSCF();
int  PrmAddInfoSCF( PrmFile *add );
int  PrmPutInforSCF( PrmFile *add, int n );
int  PrmSetInfoSCF( PrmFile *src, const char *type,
                             const char *mdl_name, const char *prm_name, 
                             double data);
int  PrmSearchInfoSCF( PrmFile *cmp );
int  PrmWriteInfoSCF( PrmFile *src );
int  PrmGetValueSCF( PrmFile *src );
int  PrmListupSCF( const char *tmpfile, const char *module );

/* ************************* lib/scfxin.c *************************** */
/*      Function Library to Control NCS Simulation Condition File     */
/*               for information of EXINPUT variables(XIN)            */
/* ****************************************************************** */

/* file extension of simulation condition */
#define  SCF_XIN_EXTN   "xin"
/* scf_xin.type */
#define  SCF_XIN_TYPE_NORMAL  '1'

/* EXTERNAL INPUT DATA FILE */
typedef struct scf_xin {
  char   type;                              /*  1 byte  */
  union  xin_data {
    struct xin_rec1 {
      int    usrxf_nprm;                    /*  4 bytes */
      char   usrxf_name[SCF_NAME_LEN];      /* 33 bytes */
      char   usrxf_pname[SCF_NAME_LEN];     /* 33 bytes */
    } rec1;                                 /* 70 bytes */
    struct xin_rec2 {
      char   usrxf_pname[2][SCF_NAME_LEN];  /* 66 bytes */
    } rec2;                                 /* 66 bytes */
    struct xin_rec3 {
      char   mdl_name[SCF_NAME_LEN];        /* 33 bytes */
      int    comp_number;                   /*  4 bytes */
      int    xfd_number;                    /*  4 bytes */
    } rec3;                                 /* 41 bytes */
  } xin;                                    /* 70 bytes */
} XinFile;                                  /* 71 bytes */

int XinMakeSCF();
int XinAddInfoSCF( XinFile *add );
int XinPutInfoSCF( XinFile *add, int n );
int XinSetInfoSCF( XinFile *src, const char *mdl_name, int comp );
int XinSearchInfoSCF( XinFile *cmp );
/* int XinWriteInfoSCF( XinFile *src ); */
int XinGetInfoSCF( XinFile *src, int n );
int XinDeleteInfoSCF( int n );
int XinDeleteAllInfoSCF();
int XinListupSCF( const char *disp );


/* ************************* lib/scfxfp.c *************************** */
/*    Function Library to Control NCS Simulation Condition File       */
/*     for information of EXTERNAL INPUT FUNCTION parameters(XFP)     */
/* ****************************************************************** */

/* file extension of simulation condition */
#define  SCF_XFP_EXTN            "xfp"
#define  SCF_XFP_MAX               5
/* scf_xfp.type */
#define  SCF_XFP_TYPE_PULSE      ('1')
#define  SCF_XFP_TYPE_RAMP       ('1')
#define  SCF_XFP_TYPE_BUFFER     ('2')
#define  SCF_XFP_TYPE_FILE       ('2')
#ifdef   DSPTEC_EXTENSION
# define  SCF_XFP_TYPE_AD3163    ('3')
# define  SCF_XFP_TYPE_SIN       ('1')
# define  SCF_XFP_TYPE_SWEEP     ('1')
#endif  /* DSPTEC_EXTENSION */
#define  SCF_XFP_TYPE_DEFAULT    ('1')

/* scf_xfp.func_number */
#define  SCF_XFP_FCN_PULSE        (2)
#define  SCF_XFP_FCN_RAMP         (1)
#define  SCF_XFP_FCN_BUFFER       (4)
#define  SCF_XFP_FCN_FILE         (3)
#ifdef   DSPTEC_EXTENSION
# define  SCF_XFP_FCN_AD3163      (5)
# define  SCF_XFP_FCN_SIN         (6)
# define  SCF_XFP_FCN_SWEEP       (7)
#endif /* DSPTEC_EXTENSION */

/* EXTERNAL INPUT FUNCTION PARAMETER FILE */
typedef struct scf_xfp {
  char    type;                              /*  1 byte  */
  int     func_number;                       /*  4 bytes */
  union   xfp_data { 
    struct xfp_rec1 {
      double  func_para[SCF_XFP_MAX];        /* 40 bytes */
    } rec1;                                  /* 40 bytes */
    struct xfp_rec2 {
      int     buffnum;                       /*  4 bytes */
      int     buffdim;                       /*  4 bytes */
      char    datf_name[SCF_NAME_LEN];       /* 22 bytes */
      char    buffname[SCF_NAME_LEN];        /* 32 bytes */
      int     timebuff;                      /*  4 bytes */
      char    timename[SCF_NAME_LEN];        /* 32 bytes */
    } rec2;                                  /* 98 bytes */
  } xfp;
} XfpFile;                                   /*103 bytes */

typedef struct rec_xfdata {
  char  xf_name[SCF_NAME_LEN];
  char  xf_para[5][SCF_NAME_LEN];
  int   xf_npara;
} Xfpdata;

int XfpMakeSCF();
int XfpAddInfoSCF( XfpFile *add );
int XfpPutInfoSCF( XfpFile *add, int n );
int XfpSetInfoSCF( XfpFile *src, const char *type, double *val,
                            const char *fname, const char *bname, 
                            const char *time );
int XfpSetDefaultsXfdata( Xfpdata *data );
int XfpWriteInfoSCF( XinFile *cmp, XfpFile *src );
int XfpSetStimSCF( XinFile *sc_xin, XfpFile *xc_xfp );
int XfpGetInfoSCF( XfpFile *src, int n );
int XfpDeleteInfoSCF( int n );
int XfpDeleteAllInfoSCF();
int XfpRemoveInfoSCF( XinFile *src, int aflag );
int XfpListupSCF( const char *tmpfile );
int XfpListupAllSCF( const char *tmpfile );

/* ************************* lib/scfobs.c *************************** */
/*    Function Library to Control NCS Simulation Condition File       */
/*           for information of OBSERVABLE variables                  */
/* ****************************************************************** */

/* file extension of simulation condition */
#define  SCF_OBS_EXTN      "obs"
#define  SCF_OBS_NORMAL    ('1')

/* OBSERVABLE INFORMATION */
typedef struct observable_info {
  char   type;                          
  struct obs_info {
    char module[SCF_NAME_LEN];
    int  max_comp;
    char var[SCF_NAME_LEN];
  } info;
} ObsFile;

int ObsMakeSCF();
int ObsAddInfoSCF( ObsFile *add );
int ObsGetInfoSCF( ObsFile *src );
int ObsSearchInfoSCF( ObsFile *cmp );
int ObsListupSCF( const char *tmpfile );
int ObsSetInfoSCF( ObsFile *src, const char *mdl_name, 
                            int comp, const char *var_name );

/* ************************* lib/scfout.c *************************** */
/*    Function Library to Control NCS Simulation Condition File       */
/*         for information of OUTPUT & OBSERVABLE variables           */
/* ****************************************************************** */

/* OUTPUT DATA FILE */
typedef struct scf_out {
  char   type;                              /*  1 byte  */
  union  out_data {
    struct out_rec1 {
      char   monitor_mode;                  /*  1 byte  */
      int    vmoni_buffnum[NCS_MONI_NVAR];  /* 16 bytes */
      int    vmoni_buffdim[NCS_MONI_NVAR];  /* 16 bytes */
    } rec1;                                 /* 33 bytes */
    struct out_rec2 {
      char   strf_flag;                     /*  1 byte  */
      char   strf_name[NCS_DATA_FNAME];     /* 22 bytes */
    } rec2;                                 /* 23 bytes */
    struct out_rec3 {
      int    buff_num;                      /*  4 bytes */
      char   buf_name[NCS_BUF_NAME_LEN];    /* 32 bytes */
      int    buff_dim;                      /*  4 bytes */ /* for nmout */
      char   mdl_name[SCF_NAME_LEN];        /* 33 bytes */
      int    comp_number;                   /*  4 bytes */
      char   obs_name[SCF_NAME_WID];        /* 36 bytes */
    } rec3;                                 /*113 bytes */
    struct out_rec4 {
      int    max_number;		    /*  4 bytes */
    } rec4;
  } out;                                    /*113 bytes */
} OutFile;                                  /*114 bytes */

#define  SCF_OUT_EXTN   "out"     /* file extension of simulation condition */

/*
#ifndef OFSET_HEAD
# define  OFFSET_HEAD 0L
#endif
#ifndef READ_ERR
# define  READ_ERR    -1
#endif
*/

#define  OUT_RECSIZE        (sizeof(struct scf_out))
#define  OUT_RECORD1        (0)
#define  OUT_RECORD2        (OUT_RECSIZE)
#define  OUT_RECORD3        (OUT_RECSIZE*2)
#define  OUT_RECORD4        (OUT_RECSIZE*3)

/* OUTPUT BUFFER INFOMATION: scf_out.type */
#define  SCF_OUT_BUFF_OUT    ('1')   /* from output statement       */
#define  SCF_OUT_BUFF_EXIN   ('2')   /* from exinput statement      */
#define  SCF_OUT_BUFF_OBS    ('3')   /* from observable statement   */
#define  SCF_OUT_MONITOR     ('4')   /* moniter mode                */
#define  SCF_OUT_FILE        ('5')   /* output data file            */
#define  SCF_OUT_BUFF_SIZE   ('6')   /* specified max output buffer */
#define  SCF_OUT_NO          ('7')   /* no assign                   */

/* for mode of monitor buffer */
#define  SCF_OUT_MONITOR_MAX     (4)   /* max number of output to monitor */
#define  SCF_OUT_MONITOR_NO_CRT ('0')  /* no CRT output                   */
#define  SCF_OUT_MONITOR_CRT    ('1')  /* values only                     */

#define  SCF_OUT_BUFF_MAX       (255)  /* maximum number of variable  */

int OutMakeSCF();
int OutAddInfoSCF( OutFile *add );
int OutPutInfoSCF( OutFile *add, int n );
int OutSetInfoSCF( OutFile *src, int buf_id, const char *buf_name, 
                            int buf_dim, const char *mdl_name, int comp_no, 
                            int type, const char *obs_name );
int OutWriteInfoSCF( OutFile *src );
int OutGetInfoSCF( OutFile *src, int n );
int OutSearchInfoSCF( OutFile *cmp );
int OutDeleteInfoSCF( int n );
int OutDeleteAllInfoSCF();
int OutRemoveInfoSCF( OutFile *src, int flag );
int OutListupSCF( const char *tmpfile );

int OutGetVarIndexSCF( const char *var_name ); /* TODO delete */
int OutGetVarSizeSCF( const char *var_name ); /* TODO delete */


/* ************************* lib/scfmat.c *************************** */
/*      Function Library to Control NCS Simulation Condition File     */
/*             for information of MATHEMATICAL variables              */
/* ****************************************************************** */

/* Integration method & Condition file */
struct integal_info {
  char     method;                          /*  1 byte  */
  int      maxcell;                         /*  4 bytes */
  double   relerr;                          /*  8 bytes */
};                                          /* 13 bytes */

/* Mode and timer data file */
typedef struct scf_mat {
  char   type;                              /*  1 byte  */
  char   mode;                              /*  1 byte  */
  union  time_data {
    double  sim_timer[4];                   /* 24 bytes ? */
    long    iteration[4];                   /* 12 bytes ? */
  } timer;                                  /* 24 bytes ? */
  struct integal_info integral;             /* 13 bytes */ 
} MatFile;                                  /* 39 bytes ? */

/* file extension of simulation condition */
#define  SCF_MAT_EXTN           "mat"
/* type information: scf_mat.type */
#define  SCF_MAT_TYPE_NORMAL    ('1')  /* normal type       */
/* method of integration: integral_info.method */
#define  SCF_MAT_INTEG_EULER    ('E')
#define  SCF_MAT_INTEG_RKG      ('R')
#define  SCF_MAT_INTEG_FERBERG  ('F')

int  MatMakeSCF();
int  MatAddInfoSCF( MatFile *add );
int  MatPutInforSCF( MatFile *add, int n );
int  MatSetInfoSCF( MatFile *src, double last_time, double calc_time,
                             double store_time, double write_time,
                             const char *integ_method, int integ_maxcell,
                             double integ_relerr );
int  MatSearchInfoSCF( MatFile *cmp );
int  MatWriteInfoSCF( MatFile *src );
int  MatGetInfoSCF( MatFile *str, int n );
int  MatTimeListupSCF( const char *tmpfile );
int  MatIntegListupSCF( const char *tmpfile );

/* ************************* lib/scfmat.c *************************** */
/*        Function Library to Control NCS Simulation Condition File   */
/*                   for information of DELAY variables               */
/* ****************************************************************** */

/* file extension of simulation condition */
#define  SCF_DLY_EXTN   "dly" 

/* DELAY INFORMATION */
struct delay_info {
  char     mdl_name[SCF_NAME_LEN];          /* 33 bytes */
  char     inp_name[SCF_NAME_LEN];          /* 33 bytes */
  double   time;                            /*  8 bytes */
  double   init_out;                        /*  8 bytes */
  char     init_flag;                       /*  1 bytes */
};                                          /* 83 bytes */ 

typedef struct scf_dly {
  char   type;                              /*  1 bytes */
  struct delay_info  delay;                 /* 83 bytes */
} DlyFile;                                  /* 84 bytes */

/* type information: scf_delay.type */
#define  SCF_DLY_TYPE_NORMAL  ('1')  /* normal type       */

/* type information: scf_mat.type */
#define  SCF_DLY_FLAG_AUTO    ('A')  /* normal type       */
#define  SCF_DLY_FLAG_FIX     ('F')  /* normal type       */

int  DlyMakeSCF();
int  DlyAddInfoSCF( DlyFile *add );
int  DlyPutInfoSCF( DlyFile *add, int n );
int  DlySetInfoSCF( DlyFile *src, const char *mdl_name, 
                             const char *inp_name, double time, 
                             double init_out, const char *flag );
int  DlyWriteInfoSCF( DlyFile *src );
int  DlyGetInfoSCF( DlyFile *src, int n );
int  DlySearchInfoSCF( DlyFile *cmp );
int  DlyDeleteInfoSCF( int n );
int  DlyDeleteAllInfoSCF();
int  DlyListupSCF( const char *tmpfile );

#ifdef __cplusplus
}
#endif


#endif /* _LIBNCSC_H_ */
