/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: fir.cpp,v 1.3 2005/01/16 14:50:59 orrisroot Exp $ */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include "SL_macro.h"
#include "SL_cmd.h"

/*****************************************************
*   Finite Inpulse Response Filter                   *
*      FILTERING SPAN = 301                          *
*        COEF ( -50 )  TO  COEF ( 50 )               *
*----------------------------------------------------*
*      X = FIR (PBUF,BUF1)                           *
*          PBUF : BUFFER OF FIR COEF.                *
*          BUF1 : INPUT BUFFER                       *
*----------------------------------------------------*
*                        1986 , 7 , 3                *
*   Fortran to C     at  1988 , 8 , 3 by M.Akima     *
*   BUFFER VERSION                                   *
*       At 17th Mar 1991 modified by K.Hagiwara      *
*   SATELLITE LANGUAGE VERSION                       *
*       At 29th Dec 1992 modified by S.Hitomi        *
*   MINOR CHANGE                                     *
*       At 11th Oct 1999 modified by M.Kitahara      *
*****************************************************/

#ifdef __cplusplus
extern "C" {
#endif

static int  fir1(Buffer *para,  int span, Buffer *input, int dpt,
                 Buffer *output );
static int  fir2(Buffer *para,  int *indexp, Buffer *input, int *indexi,
                 Buffer *output, Buffer optional_value);

DLLEXPORT int mod_ispp_fir(){
  Buffer         *work, *work2, *p_work;
  Buffer          optional_value;
  int             dpt, span;
  int             dimp, indexp[MAX_INDEX];
  int             dimx, indexx[MAX_INDEX];

  int             i, index_size;

  /* Get Arguments from SATELLITE LANGUAGE */
  p_work = (Buffer *) GetSeries(0, &dimp, indexp);
  work   = (Buffer *) GetSeries(1, &dimx, indexx);
  
  if (p_work == NULL || work == NULL) return (4);
  if (dimp !=  dimx ) return (16);
  if (dimp > 2 || dimp < 1 ) return (24);

  if( GetArgNum() == 2 ){
    optional_value = 0;
    index_size = IndexSize( dimx, indexx );
    for( i=0;i<index_size;i++ )
      optional_value += work[i];
    optional_value /= index_size;
  }
  else{
    if( strcmp(GetArgType(2),"scalar") != 0 ){
      return ( 24 );
    }
    else
      optional_value = GetScalar( 2 );
  }

  if (( work2 = (Buffer *) AllocBuffer(IndexSize(dimx,indexx) )) == NULL )
    return (8);

  if ( dimx == 1 ) {
    span = indexp[0];
    dpt = indexx[0];
    if (fir1( p_work, span, work, dpt, work2 ) == -1) return (2);
  } else if ( dimx == 2 )
    if (fir2( p_work, indexp, work, indexx, work2, optional_value ) == -1 )
       return (2);

  ReturnSeries(work2, dimx, indexx);
  return 0;
}


static int fir1(Buffer *para,int span,Buffer *input,int dpt,Buffer *output ){
  int             K1, K2;
  int             i, Ifpt;
  int             Imin, Imax;
  int             j1, k, Ifo;
  Buffer          sum;

  Ifo = span;
  if (Ifo % 2 == 0) {
    printf("Input Odd Numeric !!\n");
    return -1;
  }
  Ifpt = (Ifo - 1) / 2;
  Imin = Ifpt + 1;
  Imax = dpt - Ifpt;

  if(Imin > Imax){
    printf("Illegal Input Filter or Data !!\n");
    return -1;
  }

  for (i = Imin; i <= Imax; i++) {
    j1 = i;
    sum = 0.0;
    for (k = 1; k <= Ifo; k++) {
      K1 = (span / 2) - Ifpt + k;
      K2 = j1 - Ifpt + k - 1;
      sum += (para[K1-1] * input[K2-1]);
    }
    output[i-1] = sum;
  }
  for (i = 1; i <= Ifpt; i++) {
    j1 = dpt - i + 1;
    output[i - 1]  = output[Imin - 1];
    output[j1 - 1] = output[Imax - 1];
  }
  return 0;
}

static int fir2(Buffer *y,int *index2,Buffer *x,int *index1,Buffer *z,
                Buffer optional_value ){
  int    i, j, p, q, dpt1, dpt2, dpt3;
  int    m1, n1, m2, n2, cm2, cn2;

  m1 = index1[0]; /* data x */
  n1 = index1[1]; /* data x */
  m2 = index2[0]; /* para y */
  n2 = index2[1]; /* para y */
  cm2 = m2/2;
  cn2 = n2/2;

  if (m2 % 2 == 0) {
    printf("Input Odd Numeric !!\n");
    return -1;
  }
  if (n2 % 2 == 0) {
    printf("Input Odd Numeric !!\n");
    return -1;
  }

  for ( i = 0 ; i < m1; i++ ) {
    dpt3 = i*n1;
    for ( j = 0 ; j < n1; j++ ) {
      z[dpt3+j] = 0.0;
      for( p = 0; p < m2; p++ ) {
	dpt1 = i*n1+j+(p-cm2)*n1;
	dpt2 = p*n2;
	for( q = 0; q < n2; q++ ){
          if( ((i+p-cm2)>=0)&&((i+p-cm2)<m1)&&((j+q-cn2)>=0)&&((j+q-cn2)<n1) )
	     z[dpt3+j] += x[dpt1+q-cn2]*y[dpt2+q];
          else 
             z[dpt3+j] += optional_value*y[dpt2+q];
	}
      }
    }
  }
  return 0;
}

#ifdef __cplusplus
}
#endif
