/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: fftn.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $ */
#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"

#define SWAP(a,b) tempr=(a);(a)=(b);(b)=tempr

#ifdef __cplusplus
extern "C" {
#endif

static void  fourn (double *data, int *nn, int ndim, int isign);

DLLEXPORT int mod_ispp_fftn(){
  register int    i;
  int             dimr, dimi, indexr[MAX_INDEX], indexi[MAX_INDEX];
  Buffer         *real, *imag, *data;
  int             real_id, imag_id, siz, isign;
  char           *ialp;
  static int      nn[MAX_INDEX + 1];
  
  ialp    = GetString(0);
  real    = GetSeries(1, &dimr, indexr);
  imag    = GetSeries(2, &dimi, indexi);
  real_id = GetBufferID(3);
  imag_id = GetBufferID(4);
  
  if (real == NULL || imag == NULL)
    return (4);
  if (dimr != dimi)
    return (16);
  if (!EqualIndex(indexr, indexi, dimr))
    return (18);
  
  if ( real_id == 0 || imag_id == 0 )
    return (3);
  
  CopyIndex(nn + 1, indexr, dimr);
  siz = IndexSize(dimr, indexr);
  
  data = AllocBuffer(2 * siz + 1);

  isign = 1;
  if ( ialp[0] == 'I' || ialp[0] == 'i' )
    isign = -1;
  
  /* two-dimensional array (inverse transform) */

  /* real, imag ---> data */
  for (i = 0; i < siz; i++) {
    data[2 * i + 1] = real[i];
    data[2 * i + 2] = imag[i];
  }
  
  printf("%d  %d  %d   %d\n", dimr, nn[1], nn[2], isign);
  isign *= -1;
  fourn(data, nn, dimr, isign);
  isign *= -1;
  
  
  /* data ---> real, imag */
  for (i = 0; i < siz; i++) {
    real[i] = data[2 * i + 1];
    imag[i] = data[2 * i + 2];
  }
  /* inverse transform */
  if (isign == -1) {
    for (i = 0; i < siz; i++) {
      real[i] = real[i] / siz;
      imag[i] = imag[i] / siz;
    }
  }
  /* two-dimensional array (Fourier transform) */
  
  if (WriteBuffer(real_id, dimr, indexr, real) == -1 ||
      WriteBuffer(imag_id, dimi, indexi, imag) == -1 ) {
    printf("cannot write buffer\n");
    return (3);
  }

  return 0;
}


static void fourn(double *data,int *nn, int ndim,int isign){
     /*
      * Replaces "data" by its "ndim"-dimensional discrete Fourier transform,
      * if "isign" is input as 1. "nn[1..ndim]" is an integer array containing
      * the lengths of each dimension (number of complex values), which MUST
      * all be powers of 2. "data" is a real array of length twice the product
      * of these lengths, in which the data are stored as one proceeds along
      * "data". For a two-dimensional array, this os equivalent to storing the
      * array by rows. IF "isign" is input as -1, "data" is replaced by its
      * inverse transform times the product of the lengths of all dimensions.
      */
  int             i1, i2, i3, i2rev, i3rev, ip1, ip2, ip3, ifp1,
  ifp2;
  int             ibit, idim, k1, k2, n, nprev, nrem, ntot;
  double          tempi, tempr;
  /* Double precision for trigonometric recurrences. */
  double          theta, wi, wpi, wpr, wr, wtemp;
  
  ntot = 1;
  /* Compute total number of complex values. */
  for (idim = 1; idim <= ndim; idim++)
    ntot *= nn[idim];
  nprev = 1;
  /* Main loop over the dimensions. */
  for (idim = ndim; idim >= 1; idim--) {
    n = nn[idim];
    nrem = ntot / (n * nprev);
    ip1 = nprev << 1;
    ip2 = ip1 * n;
    ip3 = ip2 * nrem;
    i2rev = 1;
    /* This is the bit reversal section of the routine. */
    for (i2 = 1; i2 <= ip2; i2 += ip1) {
      if (i2 < i2rev) {
	for (i1 = i2; i1 <= i2 + ip1 - 2; i1 += 2) {
	  for (i3 = i1; i3 <= ip3; i3 += ip2) {
	    i3rev = i2rev + i3 - i2;
	    SWAP(data[i3], data[i3rev]);
	    SWAP(data[i3 + 1], data[i3rev + 1]);
	  }
	}
      }
      ibit = ip2 >> 1;
      while (ibit >= ip1 && i2rev > ibit) {
	i2rev -= ibit;
	ibit >>= 1;
      }
      i2rev += ibit;
    }
    /* Here begins the Danielson-Lanczos section of the routine. */
    ifp1 = ip1;
    while (ifp1 < ip2) {
      ifp2 = ifp1 << 1;
      /* Initialize for the trig. recurrence. */
      theta = isign * 6.28318530717959 / (ifp2 / ip1);
      wtemp = sin(0.5 * theta);
      wpr = -2.0 * wtemp * wtemp;
      wpi = sin(theta);
      wr = 1.0;
      wi = 0.0;
      for (i3 = 1; i3 <= ifp1; i3 += ip1) {
	for (i1 = i3; i1 <= i3 + ip1 - 2; i1 += 2) {
	  for (i2 = i1; i2 <= ip3; i2 += ifp2) {
	    /* Danielson-Lanczos formula. */
	    k1 = i2;
	    k2 = k1 + ifp1;
	    tempr = wr * data[k2] - wi * data[k2 + 1];
	    tempi = wr * data[k2 + 1] + wi * data[k2];
	    data[k2] = data[k1] - tempr;
	    data[k2 + 1] = data[k1 + 1] - tempi;
	    data[k1] += tempr;
	    data[k1 + 1] += tempi;
	  }
	}
	/* Trigonometric recurrence. */
	wr = (wtemp = wr) * wpr - wi * wpi + wr;
	wi = wi * wpr + wtemp * wpi + wi;
      }
      ifp1 = ifp2;
    }
    nprev *= n;
  }
}

#ifdef __cplusplus
}
#endif
