/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: eigen.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $ */

  /*****************************************************
   *      Calculate EIGEN VECTOR & EIGEN VALUE
   *
   *      EIGEN  X,VEC,VAL
   *            X : MATRIX
   *          VEC : OUTPUT EIGEN VECTOR
   *          VAL : OUTPUT EIGEN VALUE
   *
   *          Jul 24, 1997 by Dora(T.Hayasaka)
   *          Nov  8, 1999 by Dora(T.Hayasaka)
   *****************************************************/


#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"

#define DOUBLE_MINIMUM 1.0e-15

#ifdef __cplusplus
extern "C" {
#endif

double jacobi(double **a,int n,double *d,double **v);
double** malloc2D(int c1,int c2);
double* malloc1D(int c1);
void free_data(double **buf, int c);
void rotate(double **a,int i,int j,int k,int l,double s,double tau);

DLLEXPORT int mod_ispp_eigen(){
  Buffer  *a, *val, *vec, code, **dat, *w, **v;
  int     index[MAX_INDEX], index3[MAX_INDEX];
  int     dim, dim3, n, i, j, val_id, vec_id;
  int     ct, ct2;

  a = GetSeries( 0, &dim, index );
  if ( a == NULL )
    return (4);

  if ( dim != 2 || index[0] != index[1] )
    return (7);

  vec_id = GetBufferID(1);
  val_id = GetBufferID(2);
  if ( vec_id <= 0 || val_id <= 0 )  return (17);
  n = index[0] ; 
  val = AllocBuffer( n );
  vec = AllocBuffer( n * n );
  if ( val == NULL || vec == NULL )
    return (8);

  dim3 = 1;
  index3[0] = n;

  dat = malloc2D(n+1,n+1);
  v   = malloc2D(n+1,n+1);

  ct=0;
  for( i=1;i<=n;i++)
    for (j=1;j<=n;j++)
      dat[i][j]=a[ct++];

  w=(double *)emalloc((n+1)*sizeof(double));
  code = jacobi( dat, n, w, v );

  ct=0;
  ct2=0;
  for( i=1;i<=n;i++){
    val[ct2++] = w[i];
    for(j=1;j<=n;j++){
      vec[ct++] = v[i][j];
    }
  }

  if ( WriteBuffer( vec_id, 2, index, vec ) == -1 ) return (3);
  if ( WriteBuffer( val_id, 1, index3, val ) == -1 ) return (3);

  FreeBuffer( a );
  FreeBuffer( vec );
  FreeBuffer( val );
  free_data( dat, n+1 );
  free_data(  v , n+1 );
  efree( w );

  ReturnScalar( code );
  return 0;
}

double jacobi(double **a,int n,double *d,double **v){
  int j, iq, ip, i;
  double tresh, theta, tau, t, sm, s, h, g, c, *b, *z;

  b = AllocBuffer( n+1 );
  z = AllocBuffer( n+1 );

  for ( ip = 1; ip <= n; ip++ ){
    for ( iq = 1; iq <= n; iq++ ){ 
      v[ip][iq] = 0.0;
    }
    v[ip][ip] = 1.0;
  }

  for ( ip = 1; ip <= n; ip++ ){
    b[ip] = d[ip] = a[ip][ip];
    z[ip] = 0.0;
  }

  for ( i = 1; i <= 50; i++ ){
    sm = 0.0;
    for ( ip = 1; ip <= n - 1; ip++ ){
      for ( iq = ip + 1; iq <= n; iq++ ){
	sm += fabs(a[ip][iq]);
      }
    }

    if ( sm < DOUBLE_MINIMUM ){
      return 0.0;
    }

    if ( i < 4 ) {
      tresh = 0.2 * sm / (n * n);
    } else {
      tresh = 0.0;
    }

    for ( ip = 1; ip <= n - 1; ip++ ){
      for (iq = ip + 1; iq <= n; iq++ ){
	g = 100.0 * fabs(a[ip][iq]);
	if ( (i > 4) && (g < DOUBLE_MINIMUM) ){
	  a[ip][iq] = 0.0;
	} else if ( fabs(a[ip][iq]) > tresh ){
	  h = d[iq] - d[ip];
	  if ( g < DOUBLE_MINIMUM ){
	    t = a[ip][iq] / h;
	  } else {
	    theta = 0.5 * h / a[ip][iq];
	    t = 1.0 / (fabs(theta) + sqrt(1.0 + theta * theta));
	    if ( theta < 0.0 ) {
	      t = -t;
	    }
	  }

	  c = 1.0 / sqrt(1 + t * t);
	  s = t * c;
	  tau = s / (1.0 + c);
	  h = t * a[ip][iq];
	  z[ip] -= h;
	  z[iq] += h;
	  d[ip] -= h;
	  d[iq] += h;
	  a[ip][iq] = 0.0;

	  for ( j = 1; j <= ip - 1; j++ ){ 
	    rotate(a,j,ip,j,iq,s,tau); 
	  } 
	  for ( j = ip + 1; j <= iq - 1; j++ ){ 
	    rotate(a,ip,j,j,iq,s,tau); 
	  }
	  for ( j = iq + 1; j <= n; j++ ){ 
	    rotate(a,ip,j,iq,j,s,tau); 
	  }
	  for ( j = 1; j <= n; j++ ){ 
	    rotate(v,j,ip,j,iq,s,tau); 
	  } 
	}
      }
    }

    for ( ip = 1; ip <= n; ip++ ){
      b[ip] += z[ip];
      d[ip] = b[ip];
      z[ip] = 0.0;
    }
  }
  return -1.0;
}



double** malloc2D(int c1,int c2){
  double **buf, *tmp;
  int i;

  buf = (double **)emalloc(sizeof(double *) * c1);
  for (i = 0; i < c1; i++){
    tmp = malloc1D(c2);
    buf[i] = tmp;
  }

  return buf;
}

double* malloc1D(int c1){
  double *buf;
  buf = (double *)emalloc(sizeof(double) * c1);

  return buf;
}

void free_data(double **buf, int c){
  int i;

  for (i = 0; i < c; i++){
    efree(buf[i]);
  }

  return;
}

void rotate(double **a,int i,int j,int k,int l,double s,double tau) {  
  double g, h;

  g = a[i][j]; 
  h = a[k][l];
  a[i][j] = g - s * (h + g * tau); 
  a[k][l] = h + s * (g - h * tau);

  return; 
}

#ifdef __cplusplus
}
#endif
