/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: burg.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $ */
#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"
#include "SLfloat.h"

/*****************************************************
 *   BURG ALGORITHM
 *
 *  BURG IB1 , IB2 , IB3 , [A , X] , IODR , IDPT , TOLA , IB4
 *       IB1 : INPUT BUFFER NUMBER
 *       IB2 : OUTPUT ( SPECTRUM )
 *       IB3 : OUTPUT ( AIC CURVE )
 *       A , X : AIC MIN. OR FIXED ORDER
 *       IODR : MAXIMUM ORDER OF AR
 *       IDPT : POINTS OF POWER SPECTRUM
 *       TOLA : TOLERANCE OF AIC
 *       IB4  : OUTPUT ( PREDICTION ERROR )
 *
 *   CODED BY  I. YAMADA    9,22,83 , 11,25,83
 *  	     Translate F to C at 2th Aug 1988
 *                          T.Kobayashi (7/6/89)
 *
 *****************************************************
 *		PC to EWS by aka	12,12,92
 *****************************************************
 *		Satelite to Satellite   14,1,93
 *****************************************************/
#define	PAI	M_PI

#ifdef __cplusplus
extern "C" {
#endif

static void  fblst (double *f, double *b, int n, int max, double *a, 
		    double *aic, double *p,double *ak, int ic, int b4);

/*
static void  mlp _ANSI_ARGS_((double *al, double *a, int odr));
*/

DLLEXPORT int mod_ispp_burg(){
  int             i, j, l, kl;
  int             b1, b2, b3, b4, odr, ndpt;
  double          tola;
  char           *ialpha;
  char			  ibuf[16];

  int             con, nn, dim1;
  int             iaic, naic;
  int             odra = 0, odr1, dpt;
  int             index1[MAX_INDEX], index2[MAX_INDEX], index3[MAX_INDEX];
  static double   a[5050];
  double          aic[100], ak[100], awork[100];
  double          fdpt, s1, s2, s, aicmin, adif;
  double          p[101];

  Buffer         *rdata0, *rdata1, *rdata2;

  /*-----< Check Fatel Errors >----------*/
  if ( get_sampling() <= 0.0) {
    return (9);
  }
  /*----<  Set Order >-----*/

  /* GET PARAMETERS */
  b1 	 = GetBufferID(0);
  b2 	 = GetBufferID(1);
  b3 	 = GetBufferID(2);
  ialpha = GetString(3);
  if(!ialpha) {
	  ibuf[0] = 0x30+(int)GetScalar(3);
  }
  else
	  strcpy(ibuf, ialpha);
  odr 	 = (int) GetScalar(4);
  ndpt 	 = (int) GetScalar(5);
  tola 	 = (double) GetScalar(6);
  b4 	 = GetBufferID(7);

  switch (ibuf[0]/*ialpha[0]*/) {
  case 'A': con = 2; break;
  case 'E': con = 3; break;
  case 'K': con = 4; break;
  case 'F': con = 5; break;
  case 'B': con = 6; break;
  case 'C': con = 7; break;
  /* case 'L': con = 8; break; */
  default:  con = 1;
  }

  /* Check Buffer ID*/
  if (b1 == b2 || b2 == b3 || b1 == b3) {
    return (3);
  }

  /* Check Set Order */
  printf("odr= %d\n", odr); 	
  if (odr < 1 || odr > 60) {
    return (22);
  }

  /* Check SET POINTS OF POWER SPECTRUM */
  printf("ndpt= %d\n", ndpt);	
  if (ndpt < 1) {
    return (10);
  }

  /* Check SET TOLERANCE OF AIC */
  printf("tola= %f\n", tola);	
  if (tola < 0.0 || tola > 3.0) {
    return (2);
  }

  /* LOAD DATA */
  rdata0 = ReadBuffer(b1, &dim1, index1);
  if (rdata0 == NULL)
    return (4);
  if (dim1 != 1)
    return (24);
  nn = index1[0];

  rdata1 = AllocBuffer(nn);
  rdata2 = AllocBuffer(ndpt);
  if (rdata1 == NULL || rdata2 == NULL) return (8);

  /* BURG ALGORITHM */
  fblst(rdata0, rdata1, nn, odr, a, aic,
	p, ak, con, b4);

  dpt = ndpt;

  l = 1;
  for (i = 1; i < odr + 1; i++)
    l += i - 1;
  l--;
  for (i = 0; i < odr; i++) {
    kl = i + l;
    awork[i] = a[kl];
  }

  switch (con) {
  case 1:
  case 2:
    for (i = 0; i < odr; i++)
      rdata1[i] = aic[i];
    odra = odr;
    if (con == 1)
      break;

    aicmin = aic[0];
    iaic = 0;
    for (i = 1; i < odr; i++) {
      if (aic[i] < aicmin) {
	aicmin = aic[i];
	iaic = i;
      }
    }
    naic = iaic - 1;
    for (i = 0; i < naic; i++) {
      j = naic - i;
      adif = aic[j] - aicmin;
      if (adif < 0.0)
	adif *= -1.0;
      if (adif > tola)
	break;
      iaic = j;
      aicmin = aic[j];
    }

    printf("ORDER = %3d   AIC = %15.8f\n", iaic, aicmin);

    odr = iaic;
    break;

  case 3:
    odr1 = odr + 1;
    for (i = 0; i < odr1; i++)
      rdata1[i] = p[i];
    odra = odr1;
    break;

  case 4:
  case 5:
  case 6:
    for (i = 0; i < odr; i++)
      rdata1[i] = ak[i];
    odra = odr;
    break;
  case 7:
    for (i = 0; i < odr; i++)
      rdata1[i] = awork[i];
    odra = odr;
    break;
/*
  case 8:
    mlp(rdata1, awork, odr);
    odra = odr + 1;
    break;
*/
  }

  for (i = 0; i < odr; i++) {
    j = l + i;
    printf(" AR COEFFICIENT(%3d)=%15.7f\n", i + 1, a[j]);
  }

  l = 1;
  for (i = 1; i < odr + 1; i++)
    l += i - 1;
  l--;
  for (i = 0; i < odr; i++) {
    kl = i + l;
    awork[i] = a[kl];
  }
  fdpt = ndpt - 1;
  for (i = 0; i < ndpt; i++) {
    s1 = 1.0;
    s2 = 0.0;
    for (j = 1; j < odr + 1; j++) {
      kl = j + l;
      s1 += a[kl - 1] * cos(PAI * (double) (i * j) / fdpt);
      s2 += a[kl - 1] * sin(PAI * (double) (i * j) / fdpt);
    }
    s = get_sampling() * (s1 * s1 + s2 * s2);
    rdata2[i] = 2.0 * (double) p[odr] / s;
  }

  index2[0] = dpt;
  index3[0] = odra;
  WriteBuffer(b2, 1, index2, rdata2);
  WriteBuffer(b3, 1, index3, rdata1);

  if (con != 5 && con != 6) {
    for (i = 0; i < odr; i++)
      rdata1[i] = awork[i];
    index3[0] = odr;
    WriteBuffer(b4, 1, index3, rdata1);
  }

  FreeBuffer(rdata1);
  FreeBuffer(rdata2);
  return 0;
}


static void fblst(double *f, double *b,int n,int max,double *a,double *aic,
		  double *p,double *ak, int ic, int b4){
  double          num, den;
  int             i, j, k, l, ll, m, m1, lold, lmax, nm;
  double          e0, q;
  int             index4[MAX_INDEX];

  /* INITIALIZATION */
  m = 0;
  l = 1;
  e0 = 0.0;
  for (i = 0; i < n; i++) {
    e0 += (f[i] * f[i]);
    b[i] = f[i];
  }

  den = 2.0 * e0;
  p[0] = e0 / (double) n;
  q = 1.0;

  /* COMPUTE REFLECTION COEFFICIENTS */
  while (m != max) {
    lold = l;
    l = m + l;
    lmax = l + m;
    m++;
    num = 0.0;
    nm = n - m;
    for (i = 0; i < nm; i++)
      num += (b[i] * f[i + 1]);

    den = den * q - f[0] * f[0] - b[nm] * b[nm];
    a[lmax - 1] = -2.0 * num / den;
    q = 1.0 - a[lmax - 1] * a[lmax - 1];
    p[m] = p[m - 1] * q;
    aic[m - 1] = (double) n *(double) log(p[m]) + 2.0 * (double) m;
    ak[m - 1] = a[lmax - 1];

    /* LEVINSON RECURSION */
    if (m != 1) {
      m1 = m - 1;
      for (i = 0; i < m1; i++) {
	j = l + i;
	k = lold + i;
	ll = l - i - 1;
	a[j - 1] = a[k - 1] + a[lmax - 1] * a[ll - 1];
      }
    }
    /* PREDICTION ERROR UPDATE */
    for (i = 0; i < nm; i++) {
      f[i] = f[i + 1] + a[lmax - 1] * b[i];
      b[i] = b[i] + a[lmax - 1] * f[i + 1];
    }

    index4[0] = n;
    if (ic == 5)
      WriteBuffer(b4, 1, index4, f);
    if (ic == 6)
      WriteBuffer(b4, 1, index4, b);

  }
  printf("p[ %d ] = %f \n ", m, p[m]);
}

/*
static void
mlp(al, a, odr)
     double         *al, *a;
     int             odr;
{
  int             i, j, k, jk, mj;

  al[0] = 1.0;
  for (i = 0; i < odr; i++)
    a[0] = a[0] + a[i] * a[i];
  for (i = 1; i < odr; i++) {
    j = i - 1;
    al[i] = 2.0 * a[j];
    mj = odr - j;
    for (k = 0; k < mj; k++) {
      jk = k + j;
      al[i] = al[i] + 2.0 * a[k] * a[jk];
    }
  }
  al[odr] = 2.0 * a[odr - 1];
}
*/

#ifdef __cplusplus
}
#endif
