/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/lib/window.c,v 1.8 2004/08/12 07:13:53 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifdef HAVE_SYS_TIME_H
# include <sys/time.h>
#endif
#ifdef HAVE_SYS_RESOURCE_H
# include <sys/resource.h>
#endif

#include "libgpm.h"
#include "libgpmpriv.h"

#ifdef __cplusplus
extern "C" {
#endif

static int gpm_childcntl_copy_to_context(gpm_child_controller_t *childcntl);
static int gpm_childcntl_destroy(gpm_child_controller_t *childcntl);

static int window_proc_run(gpm_child_controller_t *childcntl);
static int window_proc_wait(int pid, int *status);
static int libgpm_window_resize(int wnum, char p, char o, double w, double h);

static int is_window_available();

int libgpm_window_create(int wnum, int paper, int orient, 
                         double width, double height){
  int  num, code;
  gpm_child_controller_t childcntl;
  num = wnum - 1;

  /* if already window exists then resize window */
  if(libgpm_window_resize(num, (char)paper, (char)orient, width, height) == 0)
    return 0;

  /* initialize child process controller */
  if(libgpm_process_control_init(&childcntl) != 0){
    libgpm_process_control_clean(&childcntl);
    return -1;
  }

  /* create window child process */
  code = window_proc_run(&childcntl);
  if(code != 0){
    libgpm_process_control_clean(&childcntl);
    return code;
  }

  gpm_childcntl_copy_to_context(&childcntl);
  gpm_childcntl_destroy(&childcntl);

  /* request window number and opening window size */
  libgpm_comm_send_int(wnum);
  libgpm_comm_send_int(paper);
  libgpm_comm_send_int(orient);
  libgpm_comm_send_float(height);
  libgpm_comm_send_float(width);

  /* get reply message from window process */
  if(libgpm_comm_recive_int(&code) != 0){
    /* process already quited */
    window_proc_wait(gpm_window_context.pid, NULL);
    libgpm_process_control_clean(&childcntl);
    return -1;
  }
  /* save window status */
  libgpm_context_write_win(num);

  /* clean up */
  libgpm_process_control_clean(&childcntl);
  return code;
}

static int libgpm_window_resize(int num, char p, char o, double w, double h){
  int old_x, old_y, new_x, new_y;
  GpmContext tmp;
  tmp = GpmCont;
  if(libgpm_context_read_win(num) != 0)
    return -1;
  if(regpm2(num) != 0)
    return -1;
  libgpm_mm2pixel(GpmCont.paper, &new_x, &new_y, w, h);
  switch(GpmCont.device){
  case GPM_DEVICE_DISPLAY:
  case GPM_DEVICE_DISPFILE:
    if(libgpm_proto_get_window_attributes(&old_x, &old_y) != 0){
      GpmCont = tmp;
      return -1;
    }
    if(old_x != new_x || old_y != new_y || 
       p != GpmCont.paper || o != GpmCont.orientation)
      if(libgpm_proto_resize_window(p, o, w, h) != 0){
        GpmCont = tmp;
        return -1;
      }
    libgpm_proto_clear_window();
    break;
  case GPM_DEVICE_NONE:
    GpmCont = tmp;
    return -1;
  }
  GpmCont.paperWidth  = (float)w;
  GpmCont.paperHeight = (float)h;
  wrgpm2(num);
  GpmCont = tmp;
  if(GpmCont.winNum == num){
    GpmCont.paperWidth  = (float)w;
    GpmCont.paperHeight = (float)h;
  }
  return 0;
}

int libgpm_window_close(int wnum){
  int num, code;
  GpmContext tmp;
  num = wnum - 1;
  tmp = GpmCont;
  code = 1; /* does not exist window */
  if(libgpm_context_read_win(num) != 0)
    return -1;
  if(regpm2(num) != 0)
    return -1;
  switch(GpmCont.device){
  case GPM_DEVICE_DISPLAY:
  case GPM_DEVICE_DISPFILE:
    libgpm_proto_close_window();
    window_proc_wait(gpm_window_context.pid, NULL);
    libgpm_context_close_win(num);
    libgpm_context_write_win(num);
    code = 0; /* success */
  }
  GpmCont.device = GPM_DEVICE_NONE;
  wrgpm2(num);
  GpmCont = tmp;
  if(GpmCont.winNum == num)
    GpmCont.device = GPM_DEVICE_NONE;
  return code;
}

static int window_proc_run(gpm_child_controller_t *childcntl){
#ifdef WIN32
  char bin[_MAX_PATH];
  static const char GPMWIN[] = "\\modules\\gpm\\gpmwin.exe";
  size_t i, len;
  if(GetModuleFileName(NULL, bin, _MAX_PATH) == 0) return -1;
  len = strlen(bin);
  for(i=len-1; i>0; i--){ if(bin[i] == '\\'){ bin[i]='\0'; break; } }
  if(bin[i] == '\\') bin[i]='\0';
  len = strlen(bin) + strlen(GPMWIN) + 1;
  if(len > _MAX_PATH) return -1;
  strcat(bin, GPMWIN);
#else
  static const char *bin=GPMWINDOW_COMMAND;
#endif
  return libgpm_process_create(bin, childcntl);
}

static int gpm_childcntl_copy_to_context(gpm_child_controller_t *childcntl){
  /* set current status to gpm window context */
  gpm_window_context.pid   = childcntl->pid;
  gpm_window_context.fd[0] = childcntl->pifd[0];
  gpm_window_context.fd[1] = childcntl->pofd[1];
  return 0;
}

static int gpm_childcntl_destroy(gpm_child_controller_t *childcntl){
  /* unset child controller for gpm_childcntl_clean() */
  childcntl->pid     = -1;
#ifdef WIN32
  childcntl->pifd[0] = INVALID_HANDLE_VALUE;
  childcntl->pofd[1] = INVALID_HANDLE_VALUE;
#else
  childcntl->pifd[0] = -1;
  childcntl->pofd[1] = -1;
#endif
  return 0;
}

static int window_proc_wait(int pid, int *status){
  return libgpm_process_wait(pid, status);
}

static int is_window_available(){
  int ret = 1;
  if(gpm_window_context.pid == -1)
    ret = 0;
  return ret;
}

#ifdef __cplusplus
}
#endif
