/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/lib/psfunc.c,v 1.10 2004/09/02 17:43:05 orrisroot Exp $ */

/*********************************************************
 *                                                        *
 *        PostScript Printer Driver Routine               *
 *                                                        *
 *********************************************************/
#define LIBGPM_PSFUNC_C_SOURCE

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <math.h>

#include "libgpm.h"
#include "libgpmpriv.h"

#include "psinit.h" /* init header data of postscript */
#include "pstray.h" /* tray set data of postscript */

#ifdef __cplusplus
extern "C" {
#endif

#define MtoI         (72.0/25.4)  /* convert 1 m to 1/72 inchs */
#define CSSIZE          17        /* number of center symbols */
#define PS_STACK_MAX    100
#define LF              0x0a

/* type definition */
typedef struct _gpmps_dev_t {
  float   xorg;
  float   yorg;
  float   xpre;
  float   ypre;
  int     color;
  float   level;
} gpmps_dev_t;

typedef struct _gpmps_context_t {
  FILE        *ofp;
  int          init_done; /* initializing status */
  char         paper;
  char         orientation;
  int          is_colorps;
  int          is_compat2x;
  int          is_rainbow;
  gpmps_dev_t  dev;
  unsigned int times;
  unsigned int maxpath;
  int          size;
  int          type;
  int          mode;
  int          font;
  double       height;
  double       factor;
} gpmps_context_t;

/*    global variable     */
static gpmps_context_t gpmps_cont;

void ps_context_init(FILE *fp, char paper, char orientation, int is_cps){
  gpmps_cont.ofp         = fp;
  gpmps_cont.paper       = paper;
  gpmps_cont.orientation = orientation;
  gpmps_cont.is_colorps  = is_cps;
  
  gpmps_cont.is_compat2x = 0;
  gpmps_cont.is_rainbow  = 0;
  
  gpmps_cont.dev.xorg    = -1.0;
  gpmps_cont.dev.yorg    = -1.0;
  gpmps_cont.dev.xpre    = -1.0;
  gpmps_cont.dev.ypre    = -1.0;
  gpmps_cont.dev.color   = -1;
  gpmps_cont.dev.level   = -1.0;
  gpmps_cont.times       = 0;
  gpmps_cont.maxpath     = 0;
  gpmps_cont.size        = -1;
  gpmps_cont.type        = -1;
  gpmps_cont.mode        = -1;
  gpmps_cont.font        = -1;
  gpmps_cont.height      = 0.0;
  gpmps_cont.factor      = 1.0;
  gpmps_cont.init_done   = 0;
}

/***************************
 *        ps_ginit          *
 ***************************/
void ps_ginit(){
  if(gpmps_cont.mode == 1){
    fprintf(gpmps_cont.ofp, "ST%c", LF);
    gpmps_cont.times = 0; gpmps_cont.maxpath = 0; gpmps_cont.mode = -1;
  }
  gpmps_cont.mode = -1;
  gpmps_cont.dev.xorg  = -1.0;
  gpmps_cont.dev.yorg  = -1.0;
  gpmps_cont.dev.color = -1;
  gpmps_cont.dev.level = -1.0;
  if(gpmps_cont.init_done == 0){
    gpmps_cont.font = -1;
    gpmps_cont.height = 0.0;
    gpmps_cont.size = -1;
    gpmps_cont.type = -1;
    gpmps_cont.factor = -1.0;
    fprintf(gpmps_cont.ofp, "ginit%c", LF);
    if(gpmps_cont.orientation == GPM_ORIENT_LANDSCAPE){
      if(gpmps_cont.paper == GPM_PAPER_A4){
        fprintf(gpmps_cont.ofp, "589 10 translate%c", LF);
      }else{
        fprintf(gpmps_cont.ofp, "723 10 translate%c", LF);
      }
      fprintf(gpmps_cont.ofp, "90 rotate%c%c", LF, LF);
    }else{
      fprintf(gpmps_cont.ofp, "0 4 translate%c", LF);
    }
    gpmps_cont.init_done = 1;
  }
}

/***************************
 *        ps_open           *
 ***************************/
void ps_open(int version){
  /* initialize for global valiables  */
  gpmps_cont.mode    = -1;
  gpmps_cont.init_done = 0;
  gpmps_cont.is_compat2x = (version == 2);
  ps_ginit();
}

/***************************
 *        ps_close          *
 ***************************/
void ps_close(){
  fprintf(gpmps_cont.ofp, "%%%%EOF%c", LF);
}

/***************************
 *        ps_end            *
 ***************************/
void ps_end(){
}

/***************************
 *      ps_initAdobe        *
 ***************************/
void ps_initAdobe(const char *creator, const char *infilename){
  fprintf(gpmps_cont.ofp, "%%!PS-Adobe-2.0%c", LF);
  fprintf(gpmps_cont.ofp, "%%%%Title: %s%c", infilename, LF);
  fprintf(gpmps_cont.ofp, "%%%%Creator: %s%c", creator, LF);
  fprintf(gpmps_cont.ofp, "%%%%Pages: (atend)%c", LF);
  fprintf(gpmps_cont.ofp, "%%%%DocumentPaperSizes: ");
  if(gpmps_cont.paper == GPM_PAPER_B4)
    fprintf(gpmps_cont.ofp, "B4%c", LF);
  else
    fprintf(gpmps_cont.ofp, "A4%c", LF);
  fprintf(gpmps_cont.ofp, "%%%%Orientation: ");
  if(gpmps_cont.orientation == GPM_ORIENT_LANDSCAPE)
    fprintf(gpmps_cont.ofp, "Landscape%c", LF);
  else
    fprintf(gpmps_cont.ofp, "Portrait%c", LF);
  fprintf(gpmps_cont.ofp, "%%%%EndComments%c", LF);
} 

/***************************
 *        ps_init           *
 ***************************/
void ps_init(){
  const char **p;
  /*  printf("initgraphics%c", LF); */
  for(p=gpmps_init_data; *p!=NULL; p++){
    fprintf(gpmps_cont.ofp, "%s%c", *p, LF);
  }
  fprintf(gpmps_cont.ofp, "%c", LF);
}

/***************************
 *        ps_tray           *
 ***************************/
void ps_setTray(){
  const char **p;
  fprintf(gpmps_cont.ofp, "initgraphics%c", LF);
  for(p=gpmps_tray_data; *p != NULL; p++){
    fprintf(gpmps_cont.ofp, "%s%c", *p, LF);
  }
  if(gpmps_cont.paper == GPM_PAPER_A4)
    fprintf(gpmps_cont.ofp, "@b4%c", LF);
  else
    fprintf(gpmps_cont.ofp, "@a4%c", LF);
  if(gpmps_cont.orientation == GPM_ORIENT_LANDSCAPE)
    fprintf(gpmps_cont.ofp, "@landscape%c", LF);
  fprintf(gpmps_cont.ofp, "%c", LF);
}


/***************************
 *        ps_page           *
 ***************************/
void ps_page(int pageNumber){
  fprintf(gpmps_cont.ofp, "%%%%Page: %d %d%c", pageNumber, pageNumber, LF);
}

/***************************
 *        ps_pages          *
 ***************************/
void ps_pages(int pageNumber){
  fprintf(gpmps_cont.ofp, "%%%%Trailer%c%%%%Pages: %d%c", LF, pageNumber, LF);
}

/***************************
 *        ps_roll           *
 ***************************/
void ps_roll(){
  if(gpmps_cont.mode == 1){
    fprintf(gpmps_cont.ofp, "ST%c", LF);
    gpmps_cont.mode = -1;
  }
  fprintf(gpmps_cont.ofp, "showpage%c", LF);
  gpmps_cont.init_done = 0;
}

/***************************
 *     ps_rainbow_mode      *
 ***************************/
void ps_rainbow_mode(int mode){
  gpmps_cont.is_rainbow = mode;
}

/***************************
 *        ps_origin         *
 ***************************/
void ps_origin(double x, double y){
  gpmps_cont.dev.xorg = (float)x;
  gpmps_cont.dev.yorg = (float)y;
}

/***************************
 *        ps_color          *
 ***************************/
void ps_color(int color){
  /* color table for version 4 */
  static double gray[16]  = {0.0, 0.1, 0.2, 0.3, 0.5, 0.6, 0.8, 1.0,
                             0.8, 0.6, 0.5, 1.0, 1.0, 1.0, 1.0, 1.0};
  static double red[8]   = {0.0, 0.0, 1.0, 1.0, 0.0, 0.0, 1.0, 1.0};
  static double green[8] = {0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0};
  static double blue[8]  = {0.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 1.0};
  /* color table for version 2.9x compat */
  static double  gray2[16] = {1.0, 0.8, 0.6, 0.3, 0.0, 0.0, 0.0, 0.0,
                             0.3, 0.6, 0.8, 1.0, 1.0, 1.0, 1.0, 1.0};
  static double   red2[8]  = {1.0, 0.0, 1.0, 1.0, 0.0, 0.0, 1.0, 0.0};
  static double green2[8]  = {1.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 0.0};
  static double  blue2[8]  = {1.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 0.0};
  if(color != gpmps_cont.dev.color || gpmps_cont.dev.level != -1.0){
    if(gpmps_cont.mode == 1){
      fprintf(gpmps_cont.ofp, "ST%c", LF);
      gpmps_cont.times = 0; gpmps_cont.maxpath = 0; gpmps_cont.mode = -1;
    }
    
    if(color < 0) return;
    
    gpmps_cont.dev.color = color;
    gpmps_cont.dev.level = -1.0;
    if(gpmps_cont.is_compat2x){
      if(gpmps_cont.is_colorps && color < 8){
        fprintf(gpmps_cont.ofp, "%.1f %.1f %.1f setrgbcolor ",
                red2[color], green2[color], blue2[color]);
      }else{
        color = color % 16;
        fprintf(gpmps_cont.ofp, "%.1f setgray ", gray2[color]);
      }
    }else{
      if(gpmps_cont.is_colorps && color < 8){
        fprintf(gpmps_cont.ofp, "%.1f %.1f %.1f setrgbcolor ",
                red[color], green[color], blue[color]);
      }else{
        color %= 16;
        fprintf(gpmps_cont.ofp, "%.1f setgray ", gray[color]);
      }
    }
  }
}

static void ps_set_rainbow_color(double level, double *r, 
                                 double *g, double *b){
  if(level >= 0.0 && level < 0.2){
    *r = 0.0;
    *g = 0.0;
    *b = level*5.0;
  }else if(level < 0.4){
    *r = 0.0;
    *g = (level-0.2)*5.0; 
    *b = 1.0;
  }else if(level < 0.6){
    *r = 0.0;
    *g = 1.0;
    *b = (0.6-level)*5.0;
  }else if(level < 0.8){
    *r = (level-0.6)*5.0; 
    *g = 1.0;
    *b = 0.0;
  }else if(level <= 1.0){
    *r = 1.0;
    *g = (1.0-level)*5.0; 
    *b = 0.0;
  }
}

/***************************
 *        ps_rainbow       *
 ***************************/
void ps_rainbow(double level){
  if(fabs((double)gpmps_cont.dev.level - (double)level) > 0.001 
     || gpmps_cont.dev.color != -1){
    if(gpmps_cont.mode == 1){
      fprintf(gpmps_cont.ofp, "ST%c", LF);
      gpmps_cont.times = 0; gpmps_cont.maxpath = 0; gpmps_cont.mode = -1;
    }
    gpmps_cont.dev.level = (float)level;
    gpmps_cont.dev.color = -1;
    if(gpmps_cont.is_colorps != 0){
      double  r = 0.0, g = 0.0, b = 0.0;
      ps_set_rainbow_color(level, &r, &g, &b);
      fprintf(gpmps_cont.ofp, "%.3f %.3f %.3f setrgbcolor ", r, g, b);
    }else{
      if(!gpmps_cont.is_rainbow)
        level = 1.0 - level;
      fprintf(gpmps_cont.ofp, "%.3f setgray ", level);
    }
  }
}

/***************************
 *        ps_plot           *
 ***************************/
void ps_plot(double x, double y, int mode){
  static double x1, y1, x0=0.0, y0=0.0;
  x1 = x + gpmps_cont.dev.xorg;
  y1 = y + gpmps_cont.dev.yorg;
  if((gpmps_cont.mode == -1 && mode != 0) || mode < 0){
    gpmps_cont.times++; gpmps_cont.maxpath++; gpmps_cont.mode = 0;
    x0 = gpmps_cont.dev.xpre + gpmps_cont.dev.xorg;
    y0 = gpmps_cont.dev.ypre + gpmps_cont.dev.yorg;
    fprintf(gpmps_cont.ofp, "%.2f %.2f MV ", x0, y0);
  }
  if(mode >= 0){
    gpmps_cont.dev.xpre = (float)x;
    gpmps_cont.dev.ypre = (float)y;
  }
  switch(mode){
  case -1: case 1:
    if(gpmps_cont.maxpath < PS_STACK_MAX){
      if(fabs(x0 - x1) > 0.02  || fabs(y0 - y1) > 0.02){
        fprintf(gpmps_cont.ofp, "%.2f %.2f LN ", x1, y1);
        gpmps_cont.times++; gpmps_cont.maxpath++;
        gpmps_cont.mode = 1;
      }
      break; /* this line must be this position */
    }
  case -2: case 2:
    fprintf(gpmps_cont.ofp, "%.2f %.2f LN ST%c", x1, y1, LF);
    gpmps_cont.maxpath = 0; gpmps_cont.times = 0; gpmps_cont.mode = -1;
    break;
  case 3:
    fprintf(gpmps_cont.ofp, "%.2f %.2f LN FL%c", x1, y1, LF);
    gpmps_cont.maxpath = 0; gpmps_cont.times = 0; gpmps_cont.mode = -1;
    break;
  case 0:
  default:
    fprintf(gpmps_cont.ofp, "%.2f %.2f MV ", x1, y1);
    gpmps_cont.times++; gpmps_cont.maxpath++;
    gpmps_cont.mode = 0;
    break;
  }
  if((gpmps_cont.times+1) % 5 == 0){ gpmps_cont.times = 0; puts(""); }

  if(mode < 0){
    fprintf(gpmps_cont.ofp, "%.2f %.2f MV ", x0, y0);
    gpmps_cont.mode = 0; gpmps_cont.times++; gpmps_cont.maxpath++;
    if((gpmps_cont.times+1) % 5 == 0){ gpmps_cont.times = 0; puts(""); }
  }
}

  
/***************************
 *        ps_factor         *
 ***************************/
void ps_factor(double factor){
  double f;
  if(gpmps_cont.mode == 1){
    fprintf(gpmps_cont.ofp, "ST%c", LF);
    gpmps_cont.times = 0; gpmps_cont.maxpath = 0; gpmps_cont.mode = -1;
  }
  if(fabs(gpmps_cont.factor - factor) > 0.0001){
    gpmps_cont.factor = factor;
    f = MtoI * factor;
    fprintf(gpmps_cont.ofp, "%f %f scale%c", f, f, LF);
  }
}


/***************************
 *        ps_box           *
 ***************************/
void ps_box(double x1, double y1, double x2, double y2, int mode, 
            double level, int color){
  x1 = gpmps_cont.dev.xorg + x1;
  y1 = gpmps_cont.dev.yorg + y1;
  x2 = gpmps_cont.dev.xorg + x2;
  y2 = gpmps_cont.dev.yorg + y2;
  gpmps_cont.dev.xpre = (float)x2;
  gpmps_cont.dev.ypre = (float)y2;
  if(gpmps_cont.mode == 1){
    fprintf(gpmps_cont.ofp, "ST%c", LF);
    gpmps_cont.times = 0; gpmps_cont.maxpath = 0; gpmps_cont.mode = -1;
  }
  if(mode == 2){
    if     (level < 0.0) level = 0.0;
    else if(level > 1.0) level = 1.0;
    if(!gpmps_cont.is_colorps && gpmps_cont.is_rainbow)
      level = 1.0 - level;
    ps_rainbow(level);
  }else
    ps_color(color);
  fprintf(gpmps_cont.ofp, "%.2f %.2f %.2f %.2f box ", x2 - x1, y2 - y1, 
          x1, y1);
  switch (mode){
  case 2: case 3:
    fprintf(gpmps_cont.ofp, "FL%c", LF);
    break;
  case 1: default:
    fprintf(gpmps_cont.ofp, "ST%c", LF);
    break;
  }
}


/***************************
 *       ps_lwidth          *
 ***************************/
void ps_lwidth(int width){
  double linewidth;
  if(gpmps_cont.size != width){
    gpmps_cont.size = width;
    switch (width){
    case 1:  linewidth = 1.0; break;
    case 2:  linewidth = 2.0; break;
    case 3:  linewidth = 4.0; break;
    case 4:  linewidth = 6.0; break;
    case 0:
    default: linewidth = 0.5; break;
    }
    if(gpmps_cont.mode == 1){
      fprintf(gpmps_cont.ofp, "ST%c", LF);
      gpmps_cont.times = 0; gpmps_cont.maxpath = 0; gpmps_cont.mode = -1;
    }
    fprintf(gpmps_cont.ofp, "%.2f setlinewidth%c", linewidth / MtoI, LF);
  }
}

/***************************
 *       ps_ltype           *
 ***************************/
void ps_ltype(int type){
  if(gpmps_cont.type != type){
    if(gpmps_cont.mode == 1){
      fprintf(gpmps_cont.ofp, "ST%c", LF);
      gpmps_cont.times = 0; gpmps_cont.maxpath = 0; gpmps_cont.mode = -1;
    }
    gpmps_cont.type = type;
    switch (type){
    case 1:
      fprintf(gpmps_cont.ofp, " [2 2]");            break;
    case 2:
      fprintf(gpmps_cont.ofp, " [1 1 3 1]");        break;
    case 3:
      fprintf(gpmps_cont.ofp, " [1 1 1 1 3 1]");    break;
    case 4:
      fprintf(gpmps_cont.ofp, " [1 1]");            break;
    case 5:
      fprintf(gpmps_cont.ofp, " [0.5 1.5]");        break;
    case 6:
      fprintf(gpmps_cont.ofp, " [0.5 3.5]");        break;
    case 7:
      fprintf(gpmps_cont.ofp, " [0.5 0.5]");        break;
    case 0:
    default:
      fprintf(gpmps_cont.ofp, " []");               break;
    }
    fprintf(gpmps_cont.ofp, " 0 setdash%c", LF);
  }
}

#define FONT_MAX 12
/***************************
 *        ps_string         *
 ***************************/
void ps_string(double x, double y, const char *str, double height, 
               double theta, int centermode, char font){
  char  *str2;
  int    parenNumber;
  size_t i, j;
  char   rshowmode[10], nshowmode[10];
  
  /* Table of fonts */
  static char    *fontname[3] = {"Times", "Helvetica", "Courier"};
  static char    *fonttype[3][4] = {
    {"Roman", "Italic", "Bold", "BoldItalic"},
    {"", "Oblique", "Bold", "BoldOblique"},
    {"", "Oblique", "Bold", "BoldOblique"}};
  if(strlen(str) > 0 && height > 0.01){
    if(gpmps_cont.height != height || gpmps_cont.font != font){
      /* name of font, size of font */
      if(font >= FONT_MAX){
        font = 0;
      }
      if(strlen(fonttype[font / 4][font % 4]) == 0)
        fprintf(gpmps_cont.ofp, "/%s", fontname[font / 4]);
      else
        fprintf(gpmps_cont.ofp, "/%s-%s", fontname[font / 4],
                fonttype[font / 4][font % 4]);
      fprintf(gpmps_cont.ofp, " findfont%c", LF);
      fprintf(gpmps_cont.ofp, "%.2f scalefont setfont%c", height, LF);
      gpmps_cont.font   = font;
      gpmps_cont.height = height;
    }
    switch (centermode){
    case 0:      /* Left */
      strcpy(rshowmode, "rshow_l");
      strcpy(nshowmode, "nshow_l");
      break;
    case 1:      /* Right */
      strcpy(rshowmode, "rshow_r");
      strcpy(nshowmode, "nshow_r");
      break;
    case 2:      /* Centering */
      strcpy(rshowmode, "rshow_c");
      strcpy(nshowmode, "nshow_c");
      break;
    }
    parenNumber = 0;
    for (i = 0; i < strlen(str); i++){
      if(*(str + i) == '(' || *(str + i) == ')'){
        parenNumber++;
      }
    }
    j = 0;
    str2 = (char *)calloc(strlen(str) + parenNumber + 1, sizeof(char));
    for(i = 0; i < strlen(str); i++){
      if(*(str + i) == '(' || *(str + i) == ')'){
        *(str2 + j) = '\\';
        j++;
        *(str2 + j) = *(str + i);
        j++;
      }else{
        *(str2 + j) = *(str + i);
        j++;
      }
    }
    
    if(gpmps_cont.is_compat2x){
      if(!gpmps_cont.is_colorps)
        ps_color(7); /* BLACK color */
    }

    if((int)theta != 0){
      /* rotate */
      fprintf(gpmps_cont.ofp, "(%s) %d %.2f %.2f %s%c", str2, (int)theta,
              x + gpmps_cont.dev.xorg, y + gpmps_cont.dev.yorg, rshowmode, LF);
    }else{
      fprintf(gpmps_cont.ofp, "(%s) %.2f %.2f %s%c", str2,
              x + gpmps_cont.dev.xorg, y + gpmps_cont.dev.yorg, nshowmode, LF);
    }
    free(str2);
  }
}

/***************************
 *        ps_symbol         *
 ***************************/
void ps_csymbol(double x, double y, int cdata, double height){
  int i;
  if((i = cdata) >= CSSIZE){
    i = 0;
  }
  if(gpmps_cont.mode == 1){
    fprintf(gpmps_cont.ofp, "ST%c", LF);
    gpmps_cont.times = 0; gpmps_cont.maxpath = 0; gpmps_cont.mode = -1;
  }
  /* draw center symbols */
  fprintf(gpmps_cont.ofp, "%.2f %.2f %.2f csymbol%d%c",
          x + gpmps_cont.dev.xorg, y + gpmps_cont.dev.yorg,
          (height > 0.01) ? height : 0.01, i, LF);
}

#ifdef __cplusplus
}
#endif
