/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/lib/proc-unix.c,v 1.1 2004/06/15 08:54:28 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifdef HAVE_SYS_TIME_H
# include <sys/time.h>
#endif
#ifdef HAVE_SYS_RESOURCE_H
# include <sys/resource.h>
#endif

#include "libgpm.h"
#include "libgpmpriv.h"

#ifdef __cplusplus
extern "C" {
#endif

/* #define DEBUG_PROC_UNIX_C 1 */

#ifndef WIN32
int libgpm_process_control_init(gpm_child_controller_t *childcntl){
  childcntl->pifd[0] = -1;
  childcntl->pifd[1] = -1;
  childcntl->pofd[0] = -1;
  childcntl->pofd[1] = -1;
  childcntl->pid = -1;
  /* stop signal */
  childcntl->sig_int  = signal(SIGINT,  SIG_IGN);
  childcntl->sig_term = signal(SIGTERM, SIG_IGN);
  childcntl->sig_quit = signal(SIGQUIT, SIG_IGN);
  /* create comminulation pipes */
  if(pipe(childcntl->pifd) != 0){
    return -1;
  }
  if(pipe(childcntl->pofd) != 0){
    return -1;
  }
  return 0;
}

int libgpm_process_control_clean(gpm_child_controller_t *childcntl){
  /* close pipe */
  if(childcntl->pifd[0] != -1)
    close(childcntl->pifd[0]);
  if(childcntl->pifd[1] != -1)
    close(childcntl->pifd[1]);
  if(childcntl->pofd[0] != -1)
    close(childcntl->pofd[0]);
  if(childcntl->pofd[1] != -1)
    close(childcntl->pofd[1]);
  if(childcntl->pid != -1)
    childcntl->pid = -1;
  /* start signal */
  signal(SIGINT,  childcntl->sig_int);
  signal(SIGTERM, childcntl->sig_term);
  signal(SIGQUIT, childcntl->sig_quit);
  return 0;
}

int libgpm_process_create(const char *bin, gpm_child_controller_t *childcntl){
  childcntl->pid = fork();
  switch(childcntl->pid){
  case -1: /* fork fail */
    return -1;
  case 0: /* child */
    close(0); dup(childcntl->pofd[0]); /* set read size disc. to stdin   */
    close(childcntl->pofd[0]); /* close original write file desc. */
    close(1); dup(childcntl->pifd[1]); /* set write size disc. to stdout */
    close(childcntl->pifd[1]); /* close original read file desc. */
    /* close parent file desc. */
    close(childcntl->pifd[0]);
    close(childcntl->pofd[1]);
    /* close(2); */ /* comment out for debugging */
    /* initialize signal */
    signal(SIGINT,  SIG_DFL);
    signal(SIGQUIT, SIG_DFL);
    signal(SIGTERM, SIG_DFL);
    execlp(bin, bin, NULL);
    _exit(102); /* illegal exit */
    break;
  default: /* parent */
    /* close child file desc. */
    close(childcntl->pofd[0]);
    childcntl->pofd[0] = -1;
    close(childcntl->pifd[1]);
    childcntl->pifd[1] = -1;
    break;
  }
#ifdef DEBUG_PROC_UNIX_C
  fprintf(stderr, "window process running : pid %d\n", childcntl->pid);
#endif
  return 0;
}

int libgpm_process_wait(int pid, int *status){
  int wpid,st,ret,num;
  wpid = waitpid(pid, &st, 0);
  if(wpid != pid) return -1;
  ret = 4; num = 0;
  if(WIFEXITED(st)){
    num = WEXITSTATUS(st); /* normal exit */
    ret = 0;
#ifdef WCOREDUMP
  }else if(WCOREDUMP(st)){
    num = 0;  /* core dumped */
    ret = 3;
#endif
  }else if(WIFSIGNALED(st)){
    num = WTERMSIG(st);    /* caught signal */
    ret = 1;
  }else if(WIFSTOPPED(st)){
    num = WSTOPSIG(st);    /* stopped */
    ret = 2;
  }
  if(status)
    *status = num;
  return ret;
}

#endif

#ifdef __cplusplus
}
#endif
