/*
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/lib/gpmcmd.c,v 1.27 2005/01/09 20:56:50 orrisroot Exp $ */

#define GPMCMD_C_SOURCE

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <math.h>

#include "libgpm.h"
#include "libgpmpriv.h"

#include "font.h"

#ifdef __cplusplus
extern "C" {
#endif

#ifndef COLOR_YES
#define COLOR_YES  1
#endif
#ifndef COLOR_NO
#define COLOR_NO  0
#endif

#define FFX        3.544615
#define FFY        3.544615
#define FFF_A4     0.691156
#define FFF_B4     0.558242
#define FFF_FREE   1.0

#ifndef M_PI
# define M_PI 3.14159265358979323846
#endif

/* static variables */
typedef struct _gplot_static_variables_t {
  gpm_xpoint_t point_stack[GPM_ENV_MAX_PLOTSTACK];
  int    n;
  char   before_level;
  char   before_col;
} gplot_static_variables_t;

static gplot_static_variables_t gplot_vars;

/* sub-routines */
static double gpm_aint(double x);
static double gpm_autoscale(double a1, double a2);
static int    symbol(double x, double y, double rcos, double rsin);
static void   plot_symbol(double x, double y, double rw, double rh, 
                          double rcos, double rsin, int *SymBuf);
static void   xy_sub(double x, double y, int iix, int iiy, int *ix, int *iy,
                     double rw, double rh, double rcos, double rsin);
static void   gplot_static_variables_init();
static double trans_func(double x1, double y1, double x2, double y2, double x);


int checktodvi(){
  int  device;
  int  result;
  device = (int)GpmCont.device;
  if(device == GPM_DEVICE_NONE || device == GPM_DEVICE_DISPLAY)
    result = 0;
  else
    result = 1;
  gpmdev_cont.common.device = device;
  return result;
}

/*********************************************************
 *        gopen                                           *
 *********************************************************/
int gopen(char pa, char ori, char dev, double pw, double ph){
  int dmx, dmy;
  gpmdev_cont.common.paper       = pa;
  gpmdev_cont.common.orientation = ori;
  gpmdev_cont.common.device      = dev;
  if(dev == GPM_DEVICE_NONE){
    return -1;
  }

  gpmdev_cont.common.xleng = (float)pw;
  gpmdev_cont.common.yleng = (float)ph;
  libgpm_mm2pixel(pa, &dmx, &dmy, pw, ph);

  if(dev == GPM_DEVICE_DISPLAY || dev == GPM_DEVICE_DISPFILE){
    /* get window attributes */
    int ix, iy;
    if(libgpm_proto_get_window_attributes(&ix, &iy) != 0){
      libgpm_window_close(GpmCont.winNum+1);
      return -1;
    }
    gpmdev_cont.win.ixleng = (unsigned int)ix;
    gpmdev_cont.win.iyleng = (unsigned int)iy;
  }else{
    gpmdev_cont.win.ixleng = 0;
    gpmdev_cont.win.iyleng = 0;
  }
  
  if(dev == GPM_DEVICE_DISPFILE || dev == GPM_DEVICE_FILE){
    libgpm_dvi_open(&gpmdev_cont.dvi, GpmCont.dvifile, 0);
  }

  switch(pa){
  case GPM_PAPER_A4:
    gpmdev_cont.win.ffx = (float)FFF_A4;
    gpmdev_cont.win.ffy = (float)FFF_A4;
    break;
  case GPM_PAPER_B4:
    gpmdev_cont.win.ffx = (float)FFF_B4;
    gpmdev_cont.win.ffy = (float)FFF_B4;
    break;
  default:
    gpmdev_cont.win.ffx = (float)FFF_FREE;
    gpmdev_cont.win.ffy = (float)FFF_FREE;
    break;
  }
  gpmdev_cont.win.ffx *= (float)FFX;
  gpmdev_cont.win.ffy *= (float)FFY;

  gpmdev_cont.graph.xmin     = 0.0;
  gpmdev_cont.graph.ymin     = 0.0;
  gpmdev_cont.graph.cline    = 0;
  gpmdev_cont.graph.xmax     = gpmdev_cont.common.xleng;
  gpmdev_cont.graph.ymax     = gpmdev_cont.common.yleng;
  gpmdev_cont.graph.xorg     = 20.0;
  gpmdev_cont.graph.yorg     = 20.0;
  gpmdev_cont.graph.xaxis    = 100.0;
  gpmdev_cont.graph.yaxis    = 100.0;
  gpmdev_cont.graph.factor   = 1.0;
  gpmdev_cont.graph.xpre     = gpmdev_cont.graph.xorg;
  gpmdev_cont.graph.ypre     = gpmdev_cont.graph.yorg;

  gpmdev_cont.scale.xdat_min = 0.0;
  gpmdev_cont.scale.xdat_max = 0.0;
  gpmdev_cont.scale.ydat_min = 0.0;
  gpmdev_cont.scale.ydat_max = 0.0;

  gpmdev_cont.symbol.cdata   = 0x20;
  gpmdev_cont.symbol.height  = 2.5;
  gpmdev_cont.symbol.theta   = 0.0;
  gpmdev_cont.symbol.font    = 0;

  gnewpen(GPM_COLOR_BLACK);
  gpmdev_cont.pen.color   = GPM_COLOR_BLACK;
  gpmdev_cont.pen.rainbow = -1;
  gpmdev_cont.pen.mode    = 0;
  gpmdev_cont.pen.size    = 0;
  gpmdev_cont.pen.type    = 0;

  gplot_static_variables_init();
  return 0;
}

/*********************************************************
 *        gclose                                          *
 *********************************************************/
void gclose(){
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPFILE || 
     gpmdev_cont.common.device == GPM_DEVICE_FILE)
    libgpm_dvi_close(&gpmdev_cont.dvi);
}

/*********************************************************
 *        gflush                                          *
 *********************************************************/
void gflush(){
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_flush_window();
}

/*********************************************************
 *        gnewpen                                         *
 *********************************************************/
void gnewpen(int color){
  int c  = abs(color) % MAX_COLOR;
  gpmdev_cont.pen.color   = (char)c;
  gpmdev_cont.pen.rainbow = (char)-1;
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_set_color(c, -1);
}

/*********************************************************
 *        gcsize                                          *
 *********************************************************/
void gcsize(double size){
  if(size > 0.0)
    gpmdev_cont.symbol.height = (float)size;
}

/*********************************************************
 *        gfactor                                         *
 *********************************************************/
void gfactor(double factor){
  if(factor > 0.0)
    gpmdev_cont.graph.factor = (float)factor;
}

/*********************************************************
 *        gfsize                                          *
 *********************************************************/
void gfsize(double x, double y){
  if( x > 0.0 && y > 0.0 ) {
    gpmdev_cont.graph.xaxis = (float)x;
    gpmdev_cont.graph.yaxis = (float)y;
  }
}

/*********************************************************
 *        glimit                                          *
 *********************************************************/
void glimit(double xmin, double ymin, double xmax, double ymax){
  if(xmin == 0.0 && xmax == 0.0 && ymin == 0.0 && ymax == 0.0){
    gpmdev_cont.graph.xmin = 0.0;
    gpmdev_cont.graph.ymin = 0.0;
    gpmdev_cont.graph.xmax = gpmdev_cont.common.xleng;
    gpmdev_cont.graph.ymax = gpmdev_cont.common.yleng;
  }else{
    xmin += gpmdev_cont.graph.xorg;
    xmax += gpmdev_cont.graph.xorg;
    ymin += gpmdev_cont.graph.yorg;
    ymax += gpmdev_cont.graph.yorg;
    if(xmin >= 0.0 && xmin < xmax)
      gpmdev_cont.graph.xmin = (float)xmin;
    if(xmax >= 0.0 && xmin > xmax)
      gpmdev_cont.graph.xmin = (float)xmax;
    if(xmin <= gpmdev_cont.common.xleng && xmin > xmax)
      gpmdev_cont.graph.xmax = (float)xmin;
    if(xmax <= gpmdev_cont.common.xleng && xmin < xmax)
      gpmdev_cont.graph.xmax = (float)xmax;
    if(ymin >= 0.0 && ymin < ymax)
      gpmdev_cont.graph.ymin = (float)ymin;
    if(ymax >= 0.0 && ymin > ymax)
      gpmdev_cont.graph.ymin = (float)ymax;
    if(ymin <= gpmdev_cont.common.yleng && ymin > ymax)
      gpmdev_cont.graph.ymax = (float)ymin;
    if(ymax <= gpmdev_cont.common.yleng && ymin < ymax)
      gpmdev_cont.graph.ymax = (float)ymax;
  }
}


/*********************************************************
 *        gorigin                                         *
 *********************************************************/
void gorigin(double x, double y){
  if(x >= 0.0 && y >= 0.0 && 
     x < gpmdev_cont.common.xleng && y < gpmdev_cont.common.yleng){
    gpmdev_cont.graph.xorg = (float)x;
    gpmdev_cont.graph.yorg = (float)y;
    gpmdev_cont.graph.xpre = gpmdev_cont.graph.xorg;
    gpmdev_cont.graph.ypre = gpmdev_cont.graph.yorg;
  }
}


/*********************************************************
 *        gpen                                            *
 *********************************************************/
void gpen(int ip_size, int ip_type, int ip_mode){
  int imode, itype, isize;
  if(ip_size >= 0)
    gpmdev_cont.pen.size = ip_size;
  if(ip_type >= 0 && ip_type <= 7)
    gpmdev_cont.pen.type = ip_type;
  if(ip_mode >= 0 && ip_mode <= 2)
    gpmdev_cont.pen.mode = ip_mode;
  itype  = (int)gpmdev_cont.pen.type;
  isize  = (int)gpmdev_cont.pen.size;
  imode  = (int)gpmdev_cont.pen.mode;
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_set_dash_width_dmode(itype, isize, imode);
}


/*********************************************************
 *        gscale                                          *
 *********************************************************/
void gscale(double xmin, double ymin, double xmax, double ymax){
  if(xmin < xmax){
    gpmdev_cont.scale.xdat_min = (float)xmin;
    gpmdev_cont.scale.xdat_max = (float)xmax;
  }else{
    gpmdev_cont.scale.xdat_min = (float)xmax;
    gpmdev_cont.scale.xdat_max = (float)xmin;
  }
  if(ymin < ymax){
    gpmdev_cont.scale.ydat_min = (float)ymin;
    gpmdev_cont.scale.ydat_max = (float)ymax;
  }else{
    gpmdev_cont.scale.ydat_min = (float)ymax;
    gpmdev_cont.scale.ydat_max = (float)ymin;
  }
}


/*********************************************************
 *        grainbow                                        *
 *********************************************************/
void grainbow(double level){ /* 0.0 <= level <= 1.0 */
  int n;
  int div     = RAINBOW_DIVNUM;
  int basecol = RAINBOW_BASECOL;
  if((level >= 0.0) && (level <= 1.0)){
    n = (int)(level*(double)(div * basecol));
  }else{
    n = (level < 0.0) ? 0 : div*basecol;
  }
  gpmdev_cont.pen.color   = (char)-1;
  gpmdev_cont.pen.rainbow = (char)n;
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_set_color(-1, n);
}


/*********************************************************
 *        gplot                                           *
 *********************************************************/
void gplot(double x, double y, int ipen_mode){
  double x1, y1, x2, y2;
  int ix1, iy1, ix2, iy2;
  int flagx1, flagy1, flagx2, flagy2;

  if(ipen_mode > 3){
    ipen_mode = 3;
  }else if(ipen_mode < -3){
    ipen_mode = -3;
  }
  if(ipen_mode == 0){
    gpmdev_cont.graph.xpre = (float)x;
    gpmdev_cont.graph.ypre = (float)y;
    gpmdev_cont.graph.cline = 0;
  } else {
    flagx1 = 0;
    flagy1 = 0;
    flagx2 = 0;
    flagy2 = 0;
    x1 = gpmdev_cont.graph.xpre  + gpmdev_cont.graph.xorg;
    y1 = gpmdev_cont.graph.ypre  + gpmdev_cont.graph.yorg;
    x2 = x + gpmdev_cont.graph.xorg;
    y2 = y + gpmdev_cont.graph.yorg;
    if(x1 > gpmdev_cont.graph.xmax){
      y1 = trans_func(x1, y1, x2, y2, gpmdev_cont.graph.xmax);
      x1 = gpmdev_cont.graph.xmax;
      flagx1 = 1;
    }else if(x1 < gpmdev_cont.graph.xmin){
      y1 = trans_func(x1, y1, x2, y2, gpmdev_cont.graph.xmin);
      x1 = gpmdev_cont.graph.xmin;
      flagx1 = -1;
    }
    if(y1 > gpmdev_cont.graph.ymax){
      x1 = trans_func(y1, x1, y2, x2, gpmdev_cont.graph.ymax);
      y1 = gpmdev_cont.graph.ymax;
      flagy1 = 1;
   }else if(y1 < gpmdev_cont.graph.ymin){
      x1 = trans_func(y1, x1, y2, x2, gpmdev_cont.graph.ymin);
      y1 = gpmdev_cont.graph.ymin;
      flagy1 = -1;
    }
    if(x2 > gpmdev_cont.graph.xmax){
      y2 = trans_func(x1, y1, x2, y2, gpmdev_cont.graph.xmax);
      x2 = gpmdev_cont.graph.xmax;
      flagx2 = 1;
    }else if(x2 < gpmdev_cont.graph.xmin){
      y2 = trans_func(x1, y1, x2, y2, gpmdev_cont.graph.xmin);
      x2 = gpmdev_cont.graph.xmin;
      flagx2 = -1;
    }
    if(y2 > gpmdev_cont.graph.ymax){
      x2 = trans_func(y1, x1, y2, x2, gpmdev_cont.graph.ymax);
      y2 = gpmdev_cont.graph.ymax;
      flagy2 = 1;
    }else if(y2 < gpmdev_cont.graph.ymin){
      x2 = trans_func(y1, x1, y2, x2, gpmdev_cont.graph.ymin);
      y2 = gpmdev_cont.graph.ymin;
      flagy2 = -1;
    }
    if(flagx1 * flagx2 <= 0  && flagy1 * flagy2 <= 0){
      if(flagx2 != 0 || flagy2 != 0) ipen_mode = 2;
      if(flagx1 != 0 || flagy1 != 0) gpmdev_cont.graph.cline = 0;
      ix2 = (int)(x2 * gpmdev_cont.graph.factor * gpmdev_cont.win.ffx + 0.5);
      iy2 = gpmdev_cont.win.iyleng - 
        (int)(y2 * gpmdev_cont.graph.factor * gpmdev_cont.win.ffy + 0.5);
      if(!gpmdev_cont.graph.cline){
        ix1 = (int)(x1 * gpmdev_cont.graph.factor * gpmdev_cont.win.ffx + 0.5);
        iy1 = gpmdev_cont.win.iyleng - 
          (int)(y1 * gpmdev_cont.graph.factor * gpmdev_cont.win.ffy + 0.5);
        gplot_vars.point_stack[0].x = ix1;
        gplot_vars.point_stack[0].y = iy1;
        gplot_vars.n = 1;
      }
      x1 -= gpmdev_cont.graph.xorg;
      y1 -= gpmdev_cont.graph.yorg;
      x2 -= gpmdev_cont.graph.xorg;
      y2 -= gpmdev_cont.graph.yorg;
      if(!gpmdev_cont.graph.cline)
        libgpm_dvi_write_plot(&gpmdev_cont.dvi, x1, y1, 0, &gpmdev_cont.pen);
      if((fabs( x1 - x2 ) > 0.0001) || (fabs( y1 - y2 ) > 0.0001) ||
          (ipen_mode >= 2)){
        libgpm_dvi_write_plot(&gpmdev_cont.dvi, x2, y2, ipen_mode,
                              &gpmdev_cont.pen);
      }

      if(gplot_vars.n > 0 &&
         (gplot_vars.point_stack[gplot_vars.n-1].x != ix2 ||
          gplot_vars.point_stack[gplot_vars.n-1].y != iy2)){
        gplot_vars.point_stack[gplot_vars.n].x = ix2;
        gplot_vars.point_stack[gplot_vars.n].y = iy2;
        gplot_vars.n++;
      }
      gpmdev_cont.graph.cline = 1;

    }else{
      gpmdev_cont.graph.cline = 0;
    }
  }

  if(ipen_mode > 0){
    gpmdev_cont.graph.xpre = (float)x;
    gpmdev_cont.graph.ypre = (float)y;
  }

  if(gplot_vars.n >= GPM_ENV_MAX_PLOTSTACK ||
     (gplot_vars.before_col != gpmdev_cont.pen.color || 
      gplot_vars.before_level != gpmdev_cont.pen.rainbow ||
      gpmdev_cont.graph.cline == 0 || ipen_mode != 1)){
    if(gplot_vars.n == 1){
      if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY || 
         gpmdev_cont.common.device == GPM_DEVICE_DISPFILE){
        libgpm_proto_draw_points(gplot_vars.point_stack, gplot_vars.n);
      }
    }else{
      if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY || 
         gpmdev_cont.common.device == GPM_DEVICE_DISPFILE){
        if(ipen_mode != 3){
          libgpm_proto_draw_lines(gplot_vars.point_stack, gplot_vars.n);
        }else{
          libgpm_proto_fill_polygon(gplot_vars.point_stack, gplot_vars.n);
        }
      }
    }
    if(ipen_mode < 0 && gplot_vars.n > 1)
      gplot_vars.n--;
    gplot_vars.point_stack[0].x = gplot_vars.point_stack[gplot_vars.n-1].x;
    gplot_vars.point_stack[0].y = gplot_vars.point_stack[gplot_vars.n-1].y;
    gplot_vars.n = 1;
  }
  gplot_vars.before_col   = gpmdev_cont.pen.color;
  gplot_vars.before_level = gpmdev_cont.pen.rainbow;
}

/*********************************************************
 *        gbox                                            *
 *********************************************************/
void gbox(double x1, double y1, double x2, double y2, int mode, double level){
/*  gpm_xpoint_t  pts[2]; */
  int   ix1, iy1, ix2, iy2, tmp;
  x1 = gpmdev_cont.graph.xorg + x1;
  y1 = gpmdev_cont.graph.yorg + y1;
  x2 = gpmdev_cont.graph.xorg + x2;
  y2 = gpmdev_cont.graph.yorg + y2;
  gpmdev_cont.graph.xpre = (float)x2;
  gpmdev_cont.graph.ypre = (float)y2;
  /* Frame Over Check */
  if((x1 <= gpmdev_cont.graph.xmax || x2 <= gpmdev_cont.graph.xmax) &&
     (y1 <= gpmdev_cont.graph.ymax || y2 <= gpmdev_cont.graph.ymax) &&
     (x1 >= gpmdev_cont.graph.xmin || x2 >= gpmdev_cont.graph.xmin) &&
     (y1 >= gpmdev_cont.graph.ymin || y2 >= gpmdev_cont.graph.ymin)){
    if(x1 > gpmdev_cont.graph.xmax) x1 = gpmdev_cont.graph.xmax;
    if(y1 > gpmdev_cont.graph.ymax) y1 = gpmdev_cont.graph.ymax;
    if(x2 > gpmdev_cont.graph.xmax) x2 = gpmdev_cont.graph.xmax;
    if(y2 > gpmdev_cont.graph.ymax) y2 = gpmdev_cont.graph.ymax;
    if(x1 < gpmdev_cont.graph.xmin) x1 = gpmdev_cont.graph.xmin;
    if(y1 < gpmdev_cont.graph.ymin) y1 = gpmdev_cont.graph.ymin;
    if(x2 < gpmdev_cont.graph.xmin) x2 = gpmdev_cont.graph.xmin;
    if(y2 < gpmdev_cont.graph.ymin) y2 = gpmdev_cont.graph.ymin;

    libgpm_dvi_write_box(&gpmdev_cont.dvi, x1 - gpmdev_cont.graph.xorg, 
                         y1 - gpmdev_cont.graph.yorg,
                         x2 - gpmdev_cont.graph.xorg,
                         y2 - gpmdev_cont.graph.yorg,
                         mode, level, &gpmdev_cont.pen);

    /* following part was modified for MAP by M.Shirakawa */
    ix1 = (int)(x1 * gpmdev_cont.graph.factor * gpmdev_cont.win.ffx + 0.51);
    /* iy1 = gpmdev_cont.win.iyleng - 
       (int)(y1 * gpmdev_cont.graph.factor * gpmdev_cont.win.ffy + 0.49); */
    iy1 = gpmdev_cont.win.iyleng - 
      (int)(y1 * gpmdev_cont.graph.factor * gpmdev_cont.win.ffy + 0.51);
    ix2 = (int)(x2 * gpmdev_cont.graph.factor * gpmdev_cont.win.ffx + 0.51);
    iy2 = gpmdev_cont.win.iyleng - 
      (int)(y2 * gpmdev_cont.graph.factor * gpmdev_cont.win.ffy + 0.51);
    if(ix1 > ix2){ tmp = ix1; ix1 = ix2; ix2 = tmp; }
    if(iy1 < iy2){ tmp = iy1; iy1 = iy2; iy2 = tmp; }

    if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
       gpmdev_cont.common.device == GPM_DEVICE_DISPFILE){
      switch(mode){
      case 1: /* Box Draw */
        libgpm_proto_draw_rectangle(ix1, iy2, ix2 - ix1 + 1, iy1 - iy2 + 1);
        break;
      case 2:  /* Fill Box Draw */
      case 3:
      default:
        libgpm_proto_fill_rectangle(ix1, iy2, ix2 - ix1 + 1, iy1 - iy2 + 1);
        break;
      }
    }
  }
}

/*********************************************************
 *        gerase                                          *
 *********************************************************/
void gerase(){
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPFILE ||
     gpmdev_cont.common.device == GPM_DEVICE_FILE){
    libgpm_dvi_close(&gpmdev_cont.dvi);
    libgpm_dvi_create(GpmCont.dvifile, gpmdev_cont.common.paper,
                      gpmdev_cont.common.orientation, 
                      gpmdev_cont.common.device);
    libgpm_dvi_open(&gpmdev_cont.dvi, GpmCont.dvifile, 0);
  }
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_clear_window();
}

/*********************************************************
 *        gfont                                           *
 *********************************************************/
void gfont(int type){
  gpmdev_cont.symbol.font = type;
}

/*********************************************************
 *        glabel                                          *
 *********************************************************/
void glabel(double x, double y, const char *ichar, double height,
            double theta, int imode, const char *specialfont){
  int     i, ilen;
  double  off, rad, rcos, rsin;

  if(gpmdev_cont.common.device == GPM_DEVICE_DISPFILE ||
     gpmdev_cont.common.device == GPM_DEVICE_FILE){
    libgpm_dvi_write_label(&gpmdev_cont.dvi, x, y, ichar, height, theta, 
                           imode, &gpmdev_cont.pen, &gpmdev_cont.symbol);
  }
  if(gpmdev_cont.common.device == GPM_DEVICE_NONE ||
     gpmdev_cont.common.device == GPM_DEVICE_FILE)
    return;

  libgpm_proto_set_dash_width_dmode(0, 0, (int)gpmdev_cont.pen.mode);
  ilen = (int)strlen(ichar);
  gcsize(height);
  gpmdev_cont.symbol.theta  = (float)theta;
  switch(imode){
  case 1:  /* Right */
    off = getstrwidth(ichar, height, gpmdev_cont.symbol.font);
   break;
  case 2:  /* Centering */
    off = getstrwidth(ichar, height, gpmdev_cont.symbol.font)*0.5;
    break;
  case 0:  /* Left */
  default:
    off = 0.0;
    break;
  }

  /* Flexible Angle */
  rad  = theta * M_PI / 180.0;
  rcos = (double)cos((double)rad);
  rsin = (double)sin((double)rad);
  x += gpmdev_cont.graph.xorg - off * rcos;
  y += gpmdev_cont.graph.yorg - off * rsin;

  for(i = 0; i < ilen; i++){
    gpmdev_cont.symbol.cdata = ichar[i];
    symbol(x, y, rcos, rsin);
    x = gpmdev_cont.graph.xpre;
    y = gpmdev_cont.graph.ypre;
  }
}

/**********************************************************
 *                                                        *
 *        ggraph                                           *
 *                                                        *
 *********************************************************/
void ggraph(float *xdata, float *ydata, int idat_number, int inc, 
            int imode, int symbl, float symbl_size){
  int    iflag_g, iflag_s, iflag_f, iflag_b;
  double  x, y, y0, xdlt, ydlt, xdlt_inv, ydlt_inv;
  double  xpre, ypre;
  register int    i;

  iflag_g = iflag_s = iflag_f = iflag_b = 0;

  switch (imode) {
  case 2: /* Line & Center symbol */
    iflag_g = 1;
  case 1: /* Center Symbol */
    iflag_s = 1;
    break;
  case 4: /* Bar and Graph */
    iflag_g = 1;
  case 3: /* Bar */
    iflag_b = 1;
    break;
  case 5: /* Filled Graph */
    iflag_f = 1;
    break;
  case 0: /* Line */
  default:
    iflag_g = 1;
    break;
  }

  xdlt = (gpmdev_cont.scale.xdat_max - gpmdev_cont.scale.xdat_min) / 
    gpmdev_cont.graph.xaxis;
  ydlt = (gpmdev_cont.scale.ydat_max - gpmdev_cont.scale.ydat_min) /
    gpmdev_cont.graph.yaxis;

  xdlt_inv = (xdlt != 0.0) ? 1.0/xdlt : 1.0;
  ydlt_inv = (ydlt != 0.0) ? 1.0/ydlt : 1.0;

  if(inc == 0)
    inc = 1;

  x = (xdata[0] - gpmdev_cont.scale.xdat_min) * xdlt_inv;
  y = (ydata[0] - gpmdev_cont.scale.ydat_min) * ydlt_inv;
  if(!iflag_f){
    if(iflag_g){
      gplot(x, y, 0);
      for(i = inc; i < idat_number-1; i += inc){
        x = (xdata[i] - gpmdev_cont.scale.xdat_min) * xdlt_inv;
        y = (ydata[i] - gpmdev_cont.scale.ydat_min) * ydlt_inv;
        gplot(x, y, 1);
      }
      x = (xdata[idat_number-1] - gpmdev_cont.scale.xdat_min) * xdlt_inv;
      y = (ydata[idat_number-1] - gpmdev_cont.scale.ydat_min) * ydlt_inv;
      gplot(x, y, 2);
    }else if(iflag_b){
      y0 = - gpmdev_cont.scale.ydat_min * ydlt_inv;
      gplot(x, y0, 0);
      gplot(x, y,  1);
      for (i = inc; i < idat_number-1; i += inc) {
        x = (xdata[i] - gpmdev_cont.scale.xdat_min) * xdlt_inv;
        y = (ydata[i] - gpmdev_cont.scale.ydat_min) * ydlt_inv;
        gplot(x, y0, 0);
        gplot(x, y,  1);
      }
      x = (xdata[idat_number-1] - gpmdev_cont.scale.xdat_min) * xdlt_inv;
      y = (ydata[idat_number-1] - gpmdev_cont.scale.ydat_min) * ydlt_inv;
      gplot(x, y0, 0);
      gplot(x, y, 2);
    }

    if(iflag_s == 1){
      x = (xdata[0] - gpmdev_cont.scale.xdat_min) * xdlt_inv;
      y = (ydata[0] - gpmdev_cont.scale.ydat_min) * ydlt_inv;
      gcsize(symbl_size);
      gsymbol(x, y, symbl);
      for(i = inc; i < idat_number; i += inc) {
        x = (xdata[i] - gpmdev_cont.scale.xdat_min) * xdlt_inv;
        y = (ydata[i] - gpmdev_cont.scale.ydat_min) * ydlt_inv;
        gsymbol(x, y, symbl);
      }
    }
  } else {
    xpre = x;
    ypre = y;
    y0 = - gpmdev_cont.scale.ydat_min  * ydlt_inv;
    for(i = inc; i < idat_number; i+= inc){
      x = (xdata[i] - gpmdev_cont.scale.xdat_min) * xdlt_inv;
      y = (ydata[i] - gpmdev_cont.scale.ydat_min) * ydlt_inv;
      gplot(xpre, y0,   0);
      gplot(xpre, ypre, 1);
      gplot(x,    y,    1);
      gplot(x,    y0,   1);
      gplot(xpre, y0,   3);
      xpre = x;
      ypre = y;
    }
  }
}

/*********************************************************
 *        gframe                                          *
 *********************************************************/
void gframe(){
  gbox(0.0, 0.0, gpmdev_cont.graph.xaxis, gpmdev_cont.graph.yaxis, 1, -1.0);
}

/*********************************************************
 *        gaxis                                           *
 *********************************************************/
void gaxis(double x, double y, double aleng, double theta,
           double dmin, double dmax, double ainc, int iformat,
           int imode, char *ctitle, double size, int igrid, double gleng){
  double delt, work, count, xorg, yorg, xx, yy;
  double sleng, rcos, rsin, rad, vpos, tpos;
  double pval, ofst;
  int   inc, iflag;
  int   itype;
  long  val;
  char  cval[33], *buffer;
  
  xorg = x;
  yorg = y;
  
  libgpm_proto_set_dash_width_dmode(0, gpmdev_cont.pen.size, 
                                    (int)gpmdev_cont.pen.mode);
  libgpm_proto_set_color((int)gpmdev_cont.pen.color, -1);
  
  rad = theta;
  theta = theta * 180.0 / M_PI;
  rcos = cos(rad);
  rsin = sin(rad);
  
  /* 10^n */
  work = (fabs(dmax) > fabs(dmin)) ? dmax : dmin;
  if(work < 0.0)  work *= -1.0;
  if(work == 0.0) work  =  1.0; 
  pval = gpm_aint(log10(work));
  
  if(iformat < 0 &&
     (dmax > pow(2.0,32.0) || -dmin > pow(2.0,32.0))){
    printf("axis: warning: cannot represent without index value.\n");
    iformat *= -1;
  }
  
  if(fabs(dmax - dmin) / pow(10.0,pval) < 3.0 &&
     abs(iformat) == 2)
    pval = pval -1.0;
  
  if((pval > 3 || pval < 0) && (iformat > 0)){
    ofst = pow(10.0, pval);
  }else{
    ofst = 1.0;
    pval = 0.0;
  }

  if(iformat < 0)
    iformat *= -1;

  /* Symbol Position */
  if(imode == 0){  /* On The Axis */
    vpos = 2.0;
    tpos = vpos + size + 2.0;
    sleng = 1.5;
  }else{          /* Under The Axis */
    vpos = -2.0 - size;
    tpos = vpos - 2.0 - size;
    sleng = -1.5;
  }
  
  if(ainc == 0.0){
    ainc = gpm_autoscale(dmin, dmax);
    if(ainc >= 1.0){
      if(iformat == 2)  /* integer scale */
        ainc = floor(ainc);
      ainc /= ofst;
    }else{
      pval = gpm_aint(log10(ainc));
      pval -= ( iformat == 2 ) ? 1.0 : ( ainc < 0.1 );
      ofst = pow(10.0, pval);
      ainc /= ofst;
      if(iformat != 2 && ainc == 0.25) ainc = 0.2;
      if(iformat == 2) /* integer scale */
        ainc = floor(ainc);
    }
    if(ainc == 0.0 ) ainc = 1.0;
  } else
    ainc /= ofst;

  /* Axis Draw */
  dmin /= ofst;
  dmax /= ofst;
  delt = (dmax - dmin) / aleng;
  
  work = dmin / ainc;
  inc = (int)work - 1;
  
  if(igrid != 0){
    sleng = gleng * (-(imode == 0) + (imode == 1));

    itype = igrid - 1;
    libgpm_proto_set_dash_width_dmode((int)itype, (int)gpmdev_cont.pen.size,
                                      (int)gpmdev_cont.pen.mode);
  }

  x = aleng * rcos;
  y = aleng * rsin;
  gplot(xorg, yorg, 0);
  gplot(xorg + x, yorg + y, 2);

  iflag = 0;
  for(count = (double) inc * ainc; count <= dmax + 0.000001;
       count += (ainc / 2.0)){

    iflag ^= 1;
    if(dmin-0.000001 > count)
      continue;
    
    x = (count - dmin) / delt * rcos;
    y = (count - dmin) / delt * rsin;
    
    gplot(xorg + x, yorg + y, 0);
    xx = -1.0 * sleng * rsin;
    yy = sleng * rcos;
    gplot(xorg + x + xx, yorg + y + yy, 2);
    
    if(iflag == 0)
      continue;
    
    /* Scale Value */
    xx = -1.0 * vpos * rsin;
    yy = vpos * rcos;
    
    if(iformat == 2){
      val = (long)count;
      sprintf(cval, "%ld", val);
    }else if(iformat == 1){
      if(-0.00001 < count && count < 0.00001){
        strcpy(cval, "0.0");
      }else{
        sprintf(cval, "%.4f0", count);
        for(buffer=cval; *buffer != '\0'; buffer++);
        if(*buffer == '\0') buffer--;
        for(;buffer != cval && *buffer == '0'; buffer--);
        if(buffer != cval && *buffer == '.'){
          *(buffer+2) = '\0';
        }else{
          if(buffer != cval && *buffer != '0')
            *(buffer+1) = '\0';
        }
      }
    }
    if(iformat == 1 || iformat == 2)
      glabel(xorg + x + xx, yorg + y + yy, cval, size, theta, 2, NULL);
  }
  
  /* Title */
  x = aleng / 2.0 * rcos - tpos * rsin;
  y = aleng / 2.0 * rsin + tpos * rcos;
  
  glabel(xorg + x, yorg + y, ctitle, size, theta, 2, NULL);
  
  /* 10 ^ pval  */
  if(pval != 0.0){
    x = aleng * rcos - tpos * rsin;
    y = aleng * rsin + tpos * rcos;
    glabel(xorg + x, yorg + y, "* 10", size, theta, 1, NULL);
    x = -1.0 * size * 2.0 / 3.0 * rsin;
    y = size * 2.0 / 3.0 * rcos;
    val = (long)pval;
    xorg = gpmdev_cont.graph.xpre - gpmdev_cont.graph.xorg;
    yorg = gpmdev_cont.graph.ypre - gpmdev_cont.graph.yorg;
    sprintf(cval, "%ld", val);
    glabel(xorg + x, yorg + y, cval, size * 0.7, theta, 0, NULL);
  }

  libgpm_proto_set_dash_width_dmode((int)gpmdev_cont.pen.type, 
                                    (int)gpmdev_cont.pen.size, 
                                    (int)gpmdev_cont.pen.mode);
}

/*********************************************************
 *        gaxisl                                          *
 *********************************************************/
void gaxisl(double x, double y, double aleng, double theta, 
            double dmin, double dmax, int iformat, int imode, 
            char *ctitle, double  size, int igrid, double gleng){
  double  delt, count, count2, point, xorg, yorg, xx, yy;
  double  sleng, rcos, rsin, rad, vpos, tpos;
  double  base, dstart, dend;
  int     val;
  char    cval[33];
  
  xorg = x;
  yorg = y;
  
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE){
    libgpm_proto_set_dash_width_dmode(0, (int)gpmdev_cont.pen.size,
                                      (int)gpmdev_cont.pen.mode);
    libgpm_proto_set_color((int)gpmdev_cont.pen.color, -1);
  }
  
  rad = theta;
  theta = theta * 180.0 / M_PI;
  rcos = cos(rad);
  rsin = sin(rad);
  
  x = aleng * rcos;
  y = aleng * rsin;
  gplot(xorg, yorg, 0);
  gplot(xorg + x, yorg + y, 2);
  
  /* Symbol Position */
  if(imode == 0){  /* On The Axis */
    vpos = 2.0;
    tpos = vpos + size * 7.0 / 6.0 + 2.0;
    sleng = 1.5;
  }else{           /* Under The Axis */
    vpos = -2.0 - size * 7.0 / 6.0;
    tpos = vpos - 2.0 - size;
    sleng = -1.5;
  }
  
  /* Axis Draw */
  delt = (dmax - dmin) / aleng;
  
  if(igrid != 0){
    sleng = gleng * (-(imode == 0) + (imode == 1));
    if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
       gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
      libgpm_proto_set_dash_width_dmode(igrid-1, (int)gpmdev_cont.pen.size, 
                                        (int)gpmdev_cont.pen.mode);
  }
  for(count = dmin; count <= dmax + 1.0; count += 1.0){
    base = floor(count);
    if(count == dmin){
      dstart = 1.0;
      /*
        dstart = floor(count);
        if(dstart == count)
          dstart = 1.0;
        else
        dstart = fabs(floor((dstart - count) * 10.0));
      */

    }else{
      dstart = 1.0;
    }
    if(count == dmax){
      dend = ceil(count);
      if(dend == count)
        dend = 1.0;
      else
        dend = fabs(floor((dend - count) * 10.0));
    } else
      dend = 9.0;

    for(count2 = dstart; count2 <= dend; count2 += 1.0){
      point  = log10(pow(10.0, base) * count2);
      x = (point - dmin) / delt * rcos;
      y = (point - dmin) / delt * rsin;
      if(x > aleng || y > aleng)
        break;
      if((point - dmin) > -0.00001){
        gplot(xorg + x, yorg + y, 0);
        xx = -1.0 * sleng * rsin;
        yy = sleng * rcos;
        gplot(xorg + x + xx, yorg + y + yy, 2);
      }else{
        continue;
      }
      if(count2 != 1.0)
        continue;

      /* Scale Value */
      if(iformat != 1)
        continue;
      xx = -1.0 * vpos * rsin;
      yy = vpos * rcos;
      strcpy(cval, "10");
      glabel(xorg + x + xx, yorg + y + yy, cval, size, theta, 2, NULL);

      xx = gpmdev_cont.graph.xpre - 
        gpmdev_cont.graph.xorg - 2.0 * size / 3.0 * rsin;
      yy = gpmdev_cont.graph.ypre -
        gpmdev_cont.graph.yorg + 2.0 * size / 3.0 * rcos;

      val = (int)base;
      sprintf(cval, "%d", val);
      glabel(xx, yy, cval, size * 0.5, theta, 0, NULL);
    }
  }
  
  /* Title */
  x = aleng * 0.5 * rcos - tpos * rsin;
  y = aleng * 0.5 * rsin + tpos * rcos;
  glabel(xorg + x, yorg + y, ctitle, size, theta, 2, NULL);
  
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_set_dash_width_dmode((int)gpmdev_cont.pen.type,
                                      (int)gpmdev_cont.pen.size, 
                                      (int)gpmdev_cont.pen.mode);
}

/*********************************************************
 *        ggradation                                      *
 *********************************************************/
void ggradation(int c, double level){
  int n;
  int div     = GPM_RAINBOW_DIVNUM;
  int basecol = GPM_RAINBOW_BASECOL;
  n = (int)(level * div * basecol);
  gpmdev_cont.pen.rainbow = (char) n + 1;
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_set_gradation(c, n, level);
}

/*********************************************************
 *        groll                                           *
 *********************************************************/
void groll(){
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_clear_window();
  libgpm_dvi_write_roll(&gpmdev_cont.dvi);
}

/*********************************************************
 *        gsymbol                                         *
 *********************************************************/
void gsymbol(double x, double y, char cdata){
  int    ichar;
  double x1, y1, rw, rh, rad, rcos, rsin;
  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_set_dash_width_dmode(0, (int)gpmdev_cont.pen.size,
                                      (int)gpmdev_cont.pen.mode);
  gpmdev_cont.symbol.cdata  = cdata;
  x1 = gpmdev_cont.graph.xorg + x;
  y1 = gpmdev_cont.graph.yorg + y;
  if((x1 <= gpmdev_cont.graph.xmax) && (y1 <= gpmdev_cont.graph.ymax) &&
     (x1 >= gpmdev_cont.graph.xmin) && (y1 >= gpmdev_cont.graph.ymin)){

    /* Charactor Check */
   ichar = (int)gpmdev_cont.symbol.cdata;
   if(ichar >= MAX_SYMBOL_NUM)
     return; /* if symbol number is greater than 0x1f, return */
   if(gpmdev_cont.common.device == GPM_DEVICE_DISPFILE ||
      gpmdev_cont.common.device == GPM_DEVICE_FILE)
     libgpm_dvi_write_csymbol(&gpmdev_cont.dvi, x, y, &gpmdev_cont.pen,
                              &gpmdev_cont.symbol);

   if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
      gpmdev_cont.common.device == GPM_DEVICE_DISPFILE){
      rad  = (double)gpmdev_cont.symbol.theta / 180.0 * M_PI;
      rcos = cos(rad);
      rsin = sin(rad);
      rw = rh = gpmdev_cont.symbol.height / SYMBOL_RANGE;
      plot_symbol(x1, y1, rw, rh, rcos, rsin, GPMsymbol[ichar]);
    }
  }

  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE)
    libgpm_proto_set_dash_width_dmode((int)gpmdev_cont.pen.type,
                                      (int)gpmdev_cont.pen.size,
                                      (int)gpmdev_cont.pen.mode);
}


static double gpm_aint(double x){
  double sgn, y;
  sgn = ( x < 0.0 ) ? -1.0 : 1.0;
  y = floor(fabs(x));
  if ( y != 0.0 ) y *= sgn;
  return y;
}

static double gpm_autoscale(double a1, double a2){
  static double t[9] =
    { 0.025, 0.05, 0.1, 0.2, 0.25, 0.5, 1.0, 2.0, 2.5 };
  static double xt[9] =
    { -0.93, -0.648, -0.347, -0.125, 0.07, 0.544, 0.657, 0.875 };
  double x, lx, y, z;
  int a = 0;
  if(a1 == a2){
    z= (a1/2.0);
  }else{
    x  = fabs(a1 - a2);
    lx = log10(x);
    y  = lx - floor(lx);
    if(y < xt[0])                    a = 0;
    else if(xt[0] <= y && y < xt[1]) a = 1;
    else if(xt[1] <= y && y < xt[2]) a = 2;
    else if(xt[2] <= y && y < xt[3]) a = 3;
    else if(xt[3] <= y && y < xt[4]) a = 4;
    else if(xt[4] <= y && y < xt[5]) a = 5;
    else if(xt[5] <= y && y < xt[6]) a = 6;
    else if(xt[6] <= y && y < xt[7]) a = 7;
    else if(xt[7] <= y)              a = 8;
    z = (double)(t[a]*pow(10.0,floor(lx)));
  }
  return z;
}

void libgpm_mm2pixel(int paper, int *dst_px, int *dst_py,
                     double src_mx, double src_my){
  double fff;
  switch(paper){
  case GPM_PAPER_A4: fff = FFF_A4; break;
  case GPM_PAPER_B4: fff = FFF_B4; break;
  case GPM_PAPER_FREE:
  default: fff = FFF_FREE; break;
  }
  *dst_px = (int)(src_mx * fff * FFX);
  *dst_py = (int)(src_my * fff * FFY);
}

void libgpm_pixel2mm(int paper, double *dst_mx, double *dst_my, 
                     int src_px, int src_py){
  double fff;
  switch(paper){
  case GPM_PAPER_A4: fff = FFF_A4; break;
  case GPM_PAPER_B4: fff = FFF_B4; break;
  case GPM_PAPER_FREE:
  default: fff = FFF_FREE; break;
  }
  *dst_mx = (double)src_px / FFX / fff;
  *dst_my = (double)src_py / FFX / fff;
  if(*dst_mx < 0.0) *dst_mx = 0.0;
  if(*dst_my < 0.0) *dst_my = 0.0;
}


static int symbol(double x, double y, double rcos, double rsin){
  double  rh, rw;
  double  fontwidth;
  int     ichar;

  /* Charactor Check */
  ichar = (int)gpmdev_cont.symbol.cdata;
  if(ichar < 0x20)  /* if charactor code less than 0x20, return */
    return 1;
  ichar -= 0x20;
  fontwidth = getfontwidth(gpmdev_cont.symbol.cdata,
                           gpmdev_cont.symbol.height,
                           gpmdev_cont.symbol.font);
  gpmdev_cont.graph.xpre = (float)(x + fontwidth * rcos);
  gpmdev_cont.graph.ypre = (float)(y + fontwidth * rsin);


  if(gpmdev_cont.common.device == GPM_DEVICE_DISPLAY ||
     gpmdev_cont.common.device == GPM_DEVICE_DISPFILE){
    rh = gpmdev_cont.symbol.height / FONT_RANGE;
    rw = fontwidth            / FONT_RANGE;
    plot_symbol(x, y, rw, rh, rcos, rsin, GPMFont[ichar]);
  }
  return 0;
}

#define MAX_FILLS 50
static void plot_symbol(double x, double y, double rw, double rh, 
                        double rcos, double rsin, int *SymBuf){
  double    fx, fy;
  int       icolor;
  int       i, j, k;
  int       ixpre = 0, iypre = 0, ixx, iyy, lx, ly, d1, d2;
  gpm_xpoint_t  fills[MAX_FILLS];

  icolor = (int)gpmdev_cont.pen.color;
  libgpm_proto_set_color(icolor,-1);

  i = 0;
  while((j = SymBuf[i++]) != EOD){
    lx = (int)SymBuf[i++];
    ly = (int)SymBuf[i++];
    xy_sub(x, y, lx, ly, &ixx, &iyy, rw, rh, rcos, rsin);
    switch(j){
    case PLT:
      fills[0].x = ixpre; fills[0].y = iypre;
      fills[1].x = ixx;   fills[1].y = iyy;
      libgpm_proto_draw_lines(fills, 2);
    case MOV:
      ixpre = ixx;
      iypre = iyy;
      break;
    case POL:
    case FIL:
    case FLB:
      fills[0].x = ixx;
      fills[0].y = iyy;
      k = 1;
      do{
        while(SymBuf[i] != EPL && k < MAX_FILLS){
          lx = (int)SymBuf[i++];
          ly = (int)SymBuf[i++];
          xy_sub(x, y, lx, ly, &ixx, &iyy, rw, rh, rcos, rsin);
          fills[k].x = ixx;
          fills[k].y = iyy;
          k++;
        }
        if(k > 2){
          switch(j){
          case FLB:
            libgpm_proto_set_color(GPM_COLOR_WHITE, -1);
          case FIL:
            libgpm_proto_fill_polygon(fills, k);
          case POL:
            libgpm_proto_draw_lines(fills, k);
            if(j==FLB)
              libgpm_proto_set_color(icolor, -1);
            break;
          }
          fills[0].x = fills[k-1].x;
          fills[0].y = fills[k-1].y;
          k = 1;
        }
      }while(SymBuf[i] != EPL);
      i++;
      break;
    case CIR:
    case FCR:
    case FCB:
      fx = gpmdev_cont.win.ffx * gpmdev_cont.graph.factor;
      fy = gpmdev_cont.win.ffy * gpmdev_cont.graph.factor;
      d1 = (int)(fx * rw * (double)SymBuf[i++]);
      d2 = (int)(fy * rh * (double)SymBuf[i++]);
      switch(j){
      case CIR:
        libgpm_proto_draw_arc(ixx-d1, iyy-d2, d1*2, d2*2, 0, 23040);
        break;
      case FCB:
        libgpm_proto_set_color(GPM_COLOR_WHITE, -1);
      case FCR:
        libgpm_proto_fill_arc(ixx-d1, iyy-d2, d1*2, d2*2, 0, 23040);
        if (j==FCB)
          libgpm_proto_set_color(icolor, -1);
        break;
      }
      break;
    }
  }
}

static void xy_sub(double x, double y, int iix, int iiy, int *ix, int *iy,
                   double rw, double rh, double rcos, double rsin){
  double rx, ry, xx, yy;
  rx = (double)iix;
  ry = (double)iiy;
  xx = (rw * rx * rcos - rh * ry * rsin) + x;
  yy = (rw * rx * rsin + rh * ry * rcos) + y;
  *ix = (int)(xx * gpmdev_cont.win.ffx * gpmdev_cont.graph.factor + 0.5);
  *iy = gpmdev_cont.win.iyleng - 
    (int)(yy * gpmdev_cont.win.ffy * gpmdev_cont.graph.factor + 0.5);
}

static void gplot_static_variables_init(){
  int i;
  for(i=0; i<GPM_ENV_MAX_PLOTSTACK; i++){
    gplot_vars.point_stack[i].x = -1;
    gplot_vars.point_stack[i].y = -1;
  }
  gplot_vars.n = 0;
  gplot_vars.before_level = -1;
  gplot_vars.before_col = -1;
}

static double trans_func(double x1, double y1, double x2, double y2, 
                         double x){
  double    val;
  if(x1-x2 != 0.0)
    val = (y1-y2) / (x1-x2) * (x-x2) + y2;
  else
    val = y2;
  return val;
}

#ifdef __cplusplus
}
#endif
