/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/lib/context.c,v 1.12 2005/03/29 12:16:57 orrisroot Exp $ */

/*******************************************************************
 **      File Name : context.c (original rwgpm.c)                 **
 **                                                               **
 **          System Common Area Read / Write Routine              **
 **                   for GPM Module                              **
 **                                                               **
 ******************************************************************/
#define _LIBGPM_GLOBAL_VARS_

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "libgpm.h"
#include "libgpmpriv.h"

#ifdef __cplusplus
extern "C" {
#endif

#ifdef WIN32
# define FOPEN_MODE_R  "rb"
# define FOPEN_MODE_W  "wb"
# define FOPEN_MODE_RP "rb+"
#else
# define FOPEN_MODE_R  "r"
# define FOPEN_MODE_W  "w"
# define FOPEN_MODE_RP "r+"
#endif

/* static int  libgpm_context_create_gpm(); */
static int  libgpm_context_create_gpm2();
static int  libgpm_context_create_win();
static void libgpm_context_init_gpm();
static void libgpm_context_init_win();
static int  libgpm_setenv(const char *name, const char *value);

static char tmpdir[GPM_ENV_MAX_FILEPATH] = "";

int regpm(){
  int debug;
  debug = GpmCont.winNum;
  if(regpm2(-1) != 0)
    return -1;
  if(debug != GpmCont.winNum)
    fprintf(stderr, "***** Warning : winNum != num in regpm()\n");
  return libgpm_context_read_win(GpmCont.winNum);
}

int wrgpm(){
  int debug;
  debug = GpmCont.winNum;
  if(wrgpm2(-1) != 0)
    return -1;
  if(debug != GpmCont.winNum)
    fprintf(stderr, "***** Warning : winNum != num in wrgpm()\n");
  return libgpm_context_write_win(GpmCont.winNum);
}

int chgpm(int num){
  wrgpm();
  if(regpm2(num) != 0)
    return -1;
  if(libgpm_context_read_win(num) != 0)
    return -1;
  if(wrgpm2(num) != 0)
    return -1;
  return libgpm_context_write_win(num);
}

/******************************************************
 * for GPM_CONTEXT (window parameters of all windows) *
 ******************************************************/
static int libgpm_context_create_gpm2(){
  static int num = 1;
  GpmContext  tmp[GPM_ENV_MAX_WIN];
  FILE *fp;
  char  fname[GPM_ENV_MAX_FILEPATH];
  int   i;
  if(*tmpdir == '\0')
    return -1;
  snprintf(fname, GPM_ENV_MAX_FILEPATH, "%s/%s", tmpdir, 
           GPM_CONTEXT_FILE_GPM);
  if((fp = fopen(fname, FOPEN_MODE_W)) == NULL){
    return -1;
  }
  for(i=0; i < GPM_ENV_MAX_WIN; i++){
    tmp[i] = GpmCont;
    tmp[i].winNum = i;
  }
  fwrite(&num, sizeof(int), 1, fp);
  fwrite(tmp, sizeof(GpmContext), GPM_ENV_MAX_WIN, fp);
  fclose(fp);
  return 0;
}

int read_current_window_number(){
  FILE *fp;
  char  fname[GPM_ENV_MAX_FILEPATH];
  int   num;
  if(*tmpdir == '\0')
    return -1;
  snprintf(fname, GPM_ENV_MAX_FILEPATH, "%s/%s", tmpdir, 
           GPM_CONTEXT_FILE_GPM);
  if((fp = fopen(fname, FOPEN_MODE_R)) == NULL)
    return -1;
  fread(&num, sizeof(GpmContext), 1, fp);
  fclose(fp);
  return num;
}

int regpm2(int num){
  FILE *fp;
  char  fname[GPM_ENV_MAX_FILEPATH];
  if(*tmpdir == '\0')
    return -1;
  snprintf(fname, GPM_ENV_MAX_FILEPATH, "%s/%s", tmpdir, 
           GPM_CONTEXT_FILE_GPM);
  if((fp = fopen(fname, FOPEN_MODE_R)) == NULL)
    return -1;
  if(num == -1)
    fread(&num, sizeof(int), 1, fp);
  fseek(fp, sizeof(int) + sizeof(GpmContext) * num, SEEK_SET);
  fread(&GpmCont, sizeof(GpmContext), 1, fp);
  fclose(fp);
  return 0;
}

int wrgpm2(int num){
  FILE *fp;
  char  fname[GPM_ENV_MAX_FILEPATH];
  if(*tmpdir == '\0')
    return -1;
  snprintf(fname, GPM_ENV_MAX_FILEPATH, "%s/%s", tmpdir, 
           GPM_CONTEXT_FILE_GPM);
  if((fp = fopen(fname, FOPEN_MODE_RP)) == NULL)
    return -1;
  if(num == -1)
    fread(&num, sizeof(int), 1, fp);
  fseek(fp, 0L, SEEK_SET);
  fwrite(&num, sizeof(int), 1, fp);
  fseek(fp, sizeof(int) + sizeof(GpmContext) * num, SEEK_SET);
  fwrite(&GpmCont, sizeof(GpmContext), 1, fp);
  fclose(fp);
  return 0;
}

/************************************************
 * for GPMWIN (window child process parameters )
 ************************************************/
static int libgpm_context_create_win(){
  gpm_window_context_t tmp[GPM_ENV_MAX_WIN];
  FILE *fp;
  char  fname[GPM_ENV_MAX_FILEPATH];
  int   i;
  if(*tmpdir == '\0')
    return -1;
  snprintf(fname, GPM_ENV_MAX_FILEPATH, "%s/%s", tmpdir, 
           GPM_CONTEXT_FILE_WIN);
  if((fp = fopen(fname, FOPEN_MODE_W)) == NULL){
    return -1;
  }
  for (i = 0; i < GPM_ENV_MAX_WIN; i++)
    tmp[i] = gpm_window_context;
  fwrite(tmp, sizeof(gpm_window_context_t), GPM_ENV_MAX_WIN, fp);
  fclose(fp);
  return 0;
}

int libgpm_context_read_win(int num){
  gpm_window_context_t tmp[GPM_ENV_MAX_WIN];
  FILE *fp;
  char  fname[GPM_ENV_MAX_FILEPATH];
  if(*tmpdir == '\0')
    return -1;
  snprintf(fname, GPM_ENV_MAX_FILEPATH, "%s/%s", tmpdir, 
           GPM_CONTEXT_FILE_WIN);
  if((fp = fopen(fname, FOPEN_MODE_R)) == NULL)
    return -1;
  fread(tmp, sizeof(gpm_window_context_t), GPM_ENV_MAX_WIN, fp);
  fclose(fp);
  gpm_window_context = tmp[num];
  return 0;
}

int libgpm_context_write_win(int num){
  gpm_window_context_t tmp[GPM_ENV_MAX_WIN];
  FILE *fp;
  char  fname[GPM_ENV_MAX_FILEPATH];
  if(*tmpdir == '\0')
    return -1;
  snprintf(fname, GPM_ENV_MAX_FILEPATH, "%s/%s", tmpdir, 
           GPM_CONTEXT_FILE_WIN);
  if((fp = fopen(fname, FOPEN_MODE_RP)) == NULL)
    return -1;
  fread(tmp, sizeof(gpm_window_context_t), GPM_ENV_MAX_WIN, fp);
  fseek(fp, 0L, SEEK_SET);
  tmp[num] = gpm_window_context;
  fwrite(tmp, sizeof(gpm_window_context_t), GPM_ENV_MAX_WIN, fp);
  fclose(fp);
  return 0;
}

int libgpm_context_close_win(int num){
#ifdef WIN32
  CloseHandle(gpm_window_context.fd[0]);
  CloseHandle(gpm_window_context.fd[1]);
#else
  close(gpm_window_context.fd[0]);
  close(gpm_window_context.fd[1]);
#endif
  libgpm_context_init_win();
  return 0;
}

/* context initializer */
int libgpm_context_settempdir(const char *src_tmpdir){
  if(src_tmpdir == NULL || *src_tmpdir == '\0'){ /* child */
    char *tmp;
    tmp = getenv(GPM_TEMPDIR_ENV);
    if(tmp == NULL)
      return -1;
    snprintf(tmpdir, GPM_ENV_MAX_FILEPATH, "%s", tmp);
  }else{
    snprintf(tmpdir, GPM_ENV_MAX_FILEPATH, "%s", src_tmpdir);
    libgpm_setenv(GPM_TEMPDIR_ENV, src_tmpdir);
  }
  return 0;
}

int libgpm_context_init(const char *src_tmpdir){
  int err;
  if(libgpm_context_settempdir(src_tmpdir) != 0)
    return -1;
  if(src_tmpdir == NULL || *src_tmpdir == '\0'){ /* child */
    /* initialize communication pipes */
    libgpm_comm_server_init();
  }else{ /* parent */
    libgpm_context_init_gpm();
    libgpm_context_init_win();
    err = libgpm_context_create_gpm2();
    if(err != 0)
      return -1;
    err = libgpm_context_create_win();
    if(err != 0)
      return -1;
  }
  libgpm_dvi_context_init(&gpmdev_cont.dvi);
  return 0;
}

int libgpm_context_clean(){
  int i;
  char fname[GPM_ENV_MAX_FILEPATH];
  if(*tmpdir == '\0')
    return -1;
  for(i=1; i<=GPM_ENV_MAX_WIN; i++)
    libgpm_window_close(i);
  snprintf(fname, GPM_ENV_MAX_FILEPATH, "%s/%s", tmpdir,
           GPM_CONTEXT_FILE_GPM);
  unlink(fname);
  snprintf(fname, GPM_ENV_MAX_FILEPATH, "%s/%s", tmpdir,
           GPM_CONTEXT_FILE_WIN);
  unlink(fname);
  libgpm_dvi_context_clean(&gpmdev_cont.dvi);
  return 0;
}

static void libgpm_context_init_gpm(){
  int i, j; 
  GpmCont.winNum       = 1;
  for ( i = 0; i < GPM_ENV_MAX_FILEPATH; i++ )
    GpmCont.dvifile[i] = ' ';
  GpmCont.dvifile[0]   = '\0';
  GpmCont.orientation  = GPM_ORIENT_PORTRAIT;
  GpmCont.paper        = GPM_PAPER_A4;
  GpmCont.paperWidth   = GPM_PAPER_A4_WIDTH;
  GpmCont.paperHeight  = GPM_PAPER_A4_HEIGHT;
  GpmCont.device       = GPM_DEVICE_NONE;

  GpmCont.factor       = 1.0;
  GpmCont.xOrigin      = 20.0;
  GpmCont.yOrigin      = 20.0;
  GpmCont.xSize        = 100.0;
  GpmCont.ySize        = 100.0;
  GpmCont.fontType     = GPM_FONT_TYPE_TR;
  GpmCont.fLineWidth   = 0;
  GpmCont.gLineWidth   = 0;
  GpmCont.fLineType    = GPM_LINE_TYPE_SOLID;
  GpmCont.gLineType    = GPM_LINE_TYPE_SOLID;
  GpmCont.fColor       = GPM_COLOR_BLACK;
  GpmCont.gColor       = GPM_COLOR_BLACK;

  GpmCont.axisType     = GPM_AXIS_TYPE_RR;
  GpmCont.axisDraw     = GPM_AXIS_DRAW_LT;

  GpmCont.xType        = GPM_SCALE_TYPE_LINEAR;
  GpmCont.yType        = GPM_SCALE_TYPE_LINEAR;
  GpmCont.zType        = GPM_SCALE_TYPE_LINEAR;

  GpmCont.xMode        = GPM_SCALE_MODE_AUTO;
  GpmCont.yMode        = GPM_SCALE_MODE_AUTO;
  GpmCont.zMode        = GPM_SCALE_MODE_AUTO;
  
  GpmCont.xMin         = 0.0;
  GpmCont.xMax         = 1.0;
  GpmCont.yMin         = 0.0;
  GpmCont.yMax         = 1.0;
  GpmCont.zMin         = 0.0;
  GpmCont.zMax         = 1.0;
  
  for (i = 0; i < GPM_ENV_MAX_TITLE; i++) {
    for (j = 0; j < GPM_ENV_MAX_TITLELEN; j++) {
      GpmCont.title_x[i][j] = ' ';
      GpmCont.title_y[i][j] = ' ';
      GpmCont.title_z[i][j] = ' ';
    }
    GpmCont.title_x[i][0] = '\0';
    GpmCont.title_y[i][0] = '\0';
    GpmCont.title_z[i][0] = '\0';
  }
}

static int libgpm_setenv(const char *name, const char *value){
#ifdef HAVE_SETENV
  return setenv(name, value, 1);
#else
  static char env[GPM_MAX_CHARBUF];
  /*
    warning :
    maybe, if putenv() specification is 4.3BSD like then it dose
    not work.. because, the environment variable setted by putenv()
    is not influence for child process.
  */
  if(strlen(name) + strlen(value) + 2 > GPM_MAX_CHARBUF)
    return -1;
  sprintf(env, "%s=%s", name, value); /* safe */
  return putenv(env);
#endif
}

static void libgpm_context_init_win(){
  gpm_window_context.pid   = -1;
#ifdef WIN32
  gpm_window_context.fd[0] = INVALID_HANDLE_VALUE;
  gpm_window_context.fd[1] = INVALID_HANDLE_VALUE;
#else
  gpm_window_context.fd[0] = -1;
  gpm_window_context.fd[1] = -1;
#endif
}

#ifdef __cplusplus
}
#endif
