/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/lib/comm-unix.c,v 1.6 2004/08/03 16:04:53 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifdef HAVE_SYS_TIME_H
# include <sys/time.h>
#endif
#ifdef HAVE_SYS_RESOURCE_H
# include <sys/resource.h>
#endif

#include "libgpm.h"
#include "libgpmpriv.h"

#ifdef __cplusplus
extern "C" {
#endif

/* #define DEBUG_COMM_UNIX_C 1 */

#ifndef WIN32

static int libgpm_comm_pipe_write(const void *buf, size_t nbytes);
static int libgpm_comm_send(int type, const void *ptr, size_t len);

void libgpm_comm_server_init(){
  int rfd,wfd;
  rfd = 0; wfd = 1;
  gpm_window_context.fd[0] = rfd;
  gpm_window_context.fd[1] = wfd;
}

int libgpm_comm_send_int(int i){
  return libgpm_comm_send(LIBGPM_COMM_TYPE_INT, &i, sizeof(int));
}

int libgpm_comm_send_float(double d){
  return libgpm_comm_send(LIBGPM_COMM_TYPE_FLOAT, &d, sizeof(double));
}

int libgpm_comm_send_str(const char *s, size_t len){
  return libgpm_comm_send(LIBGPM_COMM_TYPE_STRING, s, len);
}

int libgpm_comm_recive_int(int *i){
  int rtype;
#ifdef DEBUG_COMM_UNIX_C
  static char *str[]={"int","double","str",NULL};
  char *tstr;
#endif
  if(read(gpm_window_context.fd[0], &rtype, sizeof(int)) <= 0)
    return -1;
#ifdef DEBUG_COMM_UNIX_C
  tstr = str[rtype];
  fprintf(stderr,"recive <fd:%d, type:%s, ", gpm_window_context.fd[0], tstr);
#endif
  if(rtype != LIBGPM_COMM_TYPE_INT)
    return -1;
  if(read(gpm_window_context.fd[0], i, sizeof(int)) <= 0)
    return -1;
#ifdef DEBUG_COMM_UNIX_C
  fprintf(stderr,"%d>\n", *i);
#endif
  return 0;
}

int libgpm_comm_recive_float(double *d){
  int rtype;
#ifdef DEBUG_COMM_UNIX_C
  static char *str[]={"int","double","str",NULL};
  char *tstr;
#endif
  if(read(gpm_window_context.fd[0], &rtype, sizeof(int)) <= 0)
    return -1;
  if(rtype != LIBGPM_COMM_TYPE_FLOAT)
    return -1;
#ifdef DEBUG_COMM_UNIX_C
  tstr = str[rtype];
  fprintf(stderr,"recive <fd:%d, type:%s, ", gpm_window_context.fd[0], tstr);
#endif
  if(read(gpm_window_context.fd[0], d, sizeof(double)) <= 0)
    return -1;
#ifdef DEBUG_COMM_UNIX_C
  fprintf(stderr,"%lf>\n", *d);
#endif
  return 0;
}

int libgpm_comm_recive_str(char *s, size_t blen, size_t *rlen){
  int rtype;
  size_t i;
  char c;
  if(read(gpm_window_context.fd[0], &rtype, sizeof(int)) <= 0)
    return -1;
  if(rtype != LIBGPM_COMM_TYPE_STRING)
    return -1;
  if(read(gpm_window_context.fd[0], rlen, sizeof(size_t)) <= 0)
    return -1;
  for(i=0; i<*rlen; i++){
    if(read(gpm_window_context.fd[0], &c, 1) <= 0)
      return -1;
    if(i < blen){
      *s = c; s++;
    }
  }
  return 0;
}

static int libgpm_comm_pipe_write(const void *buf, size_t nbytes){
  int ret = 0; /* success */
  sig_t sigpipe;
  /* disable SIGPIPE */
  sigpipe = signal(SIGPIPE, SIG_IGN);
  if(write(gpm_window_context.fd[1], buf, nbytes) <= 0)
    ret = -1;
  signal(SIGPIPE, sigpipe);
  return ret;
}

static int libgpm_comm_send(int type, const void *ptr, size_t len){
#ifdef DEBUG_COMM_UNIX_C
  static char *str[]={"int","double","str",NULL};
  char *tstr = str[type];
  fprintf(stderr,"send <fd:%d, type:%s>\n", gpm_window_context.fd[1], tstr);
#endif
  if(libgpm_comm_pipe_write(&type, sizeof(int)) != 0)
    return -1;
  if(type == LIBGPM_COMM_TYPE_STRING)
    if(libgpm_comm_pipe_write(&len, sizeof(size_t)) != 0)
      return -1;
  if(libgpm_comm_pipe_write(ptr, len) != 0)
    return -1;
  return 0;
}

#endif /* WIN32 */

#ifdef __cplusplus
}
#endif
