/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/gpmwin/x11.c,v 1.20 2004/08/11 13:58:02 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifndef WIN32

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <math.h>
#include <locale.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Xlocale.h>

/* athena wiget */
#ifdef HAVE_XAW3D
# include <X11/Xaw3d/Cardinals.h>
# include <X11/Xaw3d/Form.h>
# include <X11/Xaw3d/Label.h>
# include <X11/Xaw3d/Simple.h>
# include <X11/Xaw3d/Viewport.h>
#else
# include <X11/Xaw/Cardinals.h>
# include <X11/Xaw/Form.h>
# include <X11/Xaw/Label.h>
# include <X11/Xaw/Viewport.h>
# include <X11/Xaw/Simple.h>
#endif

#include "libgpm.h"
#include "gpmwinpriv.h"

#ifdef __cplusplus
extern "C" {
#endif

extern int    paper,  orient;
extern double height, width;

typedef RETSIGTYPE(*sigfunc_t)(int);

/* program termination handler */
static void program_exit();

/* signal handers */
static void sig_destroy(int n);

/* translate table actions */
static void quit_action(Widget w, XEvent *ev, String *prms, Cardinal nprm);
static void erase_action(Widget w, XEvent *ev, String *prms, Cardinal nprm);
static void expose_action(Widget w, XEvent *ev, String *prms, Cardinal nprm);

/* event callback handlers */
static void readinput_event(XtPointer pt, int *p, XtInputId * id);
static void exceptinput_event(XtPointer pt, int *p, XtInputId * id);
static void dispcoord_event(Widget w, XtPointer cil_data, XEvent *ev);
static void getcoord_event(Widget w, XtPointer cil_data, XEvent *ev);

/* selection call back handlers */
static Bool sel_convert(Widget w, Atom *selection, Atom *target, Atom *type, 
                        XtPointer *value, unsigned long *length, int *format);
static void sel_lose();
static void sel_done();

/* color allocation */
static void color_allocation();
static void color_alloc_rainbow();
static void color_alloc_gradation();
static void color_alloc_setcols_rainbow(int n, double r, double g, double b);
static void color_alloc_setcols_gradation(int n, double x, int c);

static void set_tile_pattern(Drawable d, GC gc, double level);

/* default application and widgets resources */
static String x11_resources[] = {
  "Wopen*international: True",
  "Wopen*fontSet: -*-fixed-medium-r-normal--14-*,r14",
  "Wopen*font: r14",
#ifdef HAVE_XAW3D
  "Wopen*background: gray80",
  "Wopen*shapeStyle: Rectangle",
  "Wopen*beNiceToColormap: False",
#endif
  NULL
};

static XtActionsRec x11_actions[] = {
  {"Quit",   (XtActionProc)quit_action},
  {"Erase",  (XtActionProc)erase_action},
  {"Expose", (XtActionProc)expose_action},
  {NULL, (XtActionProc)NULL}
};

static Arg form_args[] = {
  {XtNforeground,      (XtArgVal)NULL},
  {XtNbackground,      (XtArgVal)NULL},
  {XtNdefaultDistance, (XtArgVal)0},
};

static Arg label_args[] = {
  {XtNforeground,      (XtArgVal)NULL},
  {XtNbackground,      (XtArgVal)NULL},
  {XtNlabel,           (XtArgVal)"(000.0, 000.0)"},
  {XtNborderWidth,     (XtArgVal)0},
  {XtNjustify,         (XtArgVal)XtJustifyLeft},
  {XtNtop,             (XtArgVal)XawChainTop},  /* fixed size */
  {XtNbottom,          (XtArgVal)XawChainTop},  /* fixed size */
  {XtNleft,            (XtArgVal)XawChainLeft}, /* fixed size */
  {XtNright,           (XtArgVal)XawChainLeft}, /* fixed size */
};

static Arg view_args[] = {
  {XtNwidth,           (XtArgVal)NULL},
  {XtNheight,          (XtArgVal)NULL},
  {XtNallowHoriz,      (XtArgVal)True},
  {XtNallowVert,       (XtArgVal)True},
  {XtNuseBottom,       (XtArgVal)True},
  {XtNuseRight,        (XtArgVal)True},
};

static Arg frame_args[] = {
  {XtNforeground,      (XtArgVal)NULL},
  {XtNbackground,      (XtArgVal)NULL},
  {XtNwidth,           (XtArgVal)NULL},
  {XtNheight,          (XtArgVal)NULL},
  {XtNborderWidth,     (XtArgVal)0},
};

static String toplebel_trans_table_str = 
  "<Message>WM_PROTOCOLS: Quit()";

static String frame_trans_table_str = 
  "Ctrl<Key>Q: Quit()\n"
  "Ctrl<Key>W: Erase()\n"
  "<Expose>: Expose()";


typedef struct _gpm_x11_t {
  Display      *display;
  int           screen, depth;
  char          clipbord[256];
  Atom          wm_delete_window; /* for "WM_DELETE_WINDOW" */
  Colormap      cmap;
  GC            gc;
  Pixmap        pixmap;
  unsigned long fgcolor;
  unsigned long bgcolor;
  unsigned long gray_color;
  unsigned long color[GPM_COLOR_SIZE];
  unsigned long rainbow[GPM_RAINBOW_BASECOL*GPM_RAINBOW_DIVNUM+2];
  unsigned long gradation[7][GPM_RAINBOW_BASECOL*GPM_RAINBOW_DIVNUM+2];
  Window        window;
  Widget        toplevel, form, label, view, frame;
  XtAppContext  app_con;
  unsigned int  wx, wy;
} gpm_x11_t;


/* global variable */
static gpm_x11_t x11cont;

/* public functions */

/* window initialize */
int gpmwin_init(int argc, char *argv[]){
  /* set locale */
  XtSetLanguageProc(NULL, NULL, NULL);
  setlocale(LC_CTYPE, "");
  /* open display */
  /* initialize and create top level widget */
  x11cont.toplevel = XtAppInitialize(&x11cont.app_con, "Wopen", NULL, 
                                     0, &argc, argv, x11_resources, 
                                     NULL, 0);
  if(x11cont.app_con == NULL)
    return 101; /* can't open display */
  return 0;
}

int create_frame_pixmap(int x, int y){
  x11cont.wx = x;
  x11cont.wy = y;

  /* get window conditions */
  x11cont.window  = XtWindow(x11cont.frame);
  x11cont.pixmap  = XCreatePixmap(x11cont.display, x11cont.window,
                                  x11cont.wx, x11cont.wy, x11cont.depth);
  x11cont.gc      = XCreateGC(x11cont.display, x11cont.window, 0, NULL);

  /* initialize pixmap buffer */
  XSetForeground(x11cont.display, x11cont.gc, 
                 WhitePixel(x11cont.display, x11cont.screen));
  XFillRectangle(x11cont.display, x11cont.pixmap, x11cont.gc, 0, 0, 
                 x11cont.wx, x11cont.wy);
  return 0;
}

/* window creation */
int gpmwin_create(int x, int y, const char *title){
  XtTranslations trans_table;
  XColor c,cc;

  x11cont.wx = (unsigned int)x;
  x11cont.wy = (unsigned int)y;

  /* get display informations */
  x11cont.display = XtDisplay(x11cont.toplevel);
  x11cont.screen  = DefaultScreen(x11cont.display);
  x11cont.cmap    = DefaultColormap(x11cont.display, x11cont.screen);
  x11cont.depth   = DefaultDepth(x11cont.display, x11cont.screen);

  /* add application actions */
  XtAppAddActions(x11cont.app_con, x11_actions, XtNumber(x11_actions));

  /* add application input event */
  XtAppAddInput(x11cont.app_con, 0, (XtPointer)XtInputReadMask, 
                readinput_event, NULL);
  XtAppAddInput(x11cont.app_con, 0, (XtPointer)XtInputExceptMask, 
                exceptinput_event, NULL);

  /* set colors */
  x11cont.fgcolor = BlackPixel(x11cont.display, x11cont.screen);
  x11cont.bgcolor = WhitePixel(x11cont.display, x11cont.screen);
  x11cont.gray_color = x11cont.bgcolor;
  if(XAllocNamedColor(x11cont.display, x11cont.cmap, "gray50", &c, &cc))
     x11cont.gray_color = c.pixel;
  
  /* create child widgets */
  view_args[0].value = (XtArgVal)x11cont.wx;
  view_args[1].value = (XtArgVal)x11cont.wy;
  x11cont.view = XtCreateManagedWidget("view", viewportWidgetClass, 
                                       x11cont.toplevel, view_args, 
                                       XtNumber(view_args));

  form_args[0].value = (XtArgVal)x11cont.fgcolor;
  form_args[1].value = (XtArgVal)x11cont.gray_color;
  x11cont.form = XtCreateManagedWidget("form", formWidgetClass, 
                                       x11cont.view, form_args, 
                                       XtNumber(form_args));

  label_args[0].value = (XtArgVal)x11cont.fgcolor;
  label_args[1].value = (XtArgVal)x11cont.bgcolor;
  x11cont.label = XtCreateManagedWidget("label", labelWidgetClass,
                                        x11cont.form, label_args, 
                                        XtNumber(label_args));

  frame_args[0].value = (XtArgVal)x11cont.fgcolor;
  frame_args[1].value = (XtArgVal)x11cont.gray_color;
  frame_args[2].value = (XtArgVal)x11cont.wx;
  frame_args[3].value = (XtArgVal)x11cont.wy;
  x11cont.frame = XtCreateManagedWidget("frame", simpleWidgetClass, 
                                        x11cont.form, frame_args, 
                                        XtNumber(frame_args));
  trans_table = XtParseTranslationTable(frame_trans_table_str);
  XtOverrideTranslations(x11cont.frame, trans_table);

  /* add transrations */
  trans_table = XtParseTranslationTable(toplebel_trans_table_str);
  XtOverrideTranslations(x11cont.toplevel, trans_table);
  trans_table = XtParseTranslationTable(frame_trans_table_str);
  XtOverrideTranslations(x11cont.frame, trans_table);

  /* realize widget */
  XtRealizeWidget(x11cont.toplevel);

  /* for window manager */
  x11cont.wm_delete_window = XInternAtom(x11cont.display, "WM_DELETE_WINDOW", 
                                         False);
  XSetWMProtocols(x11cont.display, XtWindow(x11cont.toplevel),
                  &x11cont.wm_delete_window, 1);

  /* create window pixmap buffer */
  x11cont.window  = XtWindow(x11cont.frame);
  x11cont.pixmap  = XCreatePixmap(x11cont.display, x11cont.window,
                                  x11cont.wx, x11cont.wy, x11cont.depth);
  x11cont.gc      = XCreateGC(x11cont.display, x11cont.window, 0, NULL);

  /* initialize pixmap buffer */
  XSetForeground(x11cont.display, x11cont.gc, 
                 WhitePixel(x11cont.display, x11cont.screen));
  XFillRectangle(x11cont.display, x11cont.pixmap, x11cont.gc, 0, 0, 
                 x11cont.wx, x11cont.wy);

  /* color allocation */
  color_allocation();

  /* set window title */
  XStoreName(x11cont.display, XtWindow(x11cont.toplevel), title);

  /* add coordinate label update handler */
  XtAddEventHandler(x11cont.label, PointerMotionMask, FALSE, 
                    (XtEventHandler)dispcoord_event, (XtPointer)NULL);
  XtAddEventHandler(x11cont.frame, PointerMotionMask, FALSE, 
                    (XtEventHandler)dispcoord_event, (XtPointer)NULL);
  /* add coordinate coping to clipboard handler */
  XtAddEventHandler(x11cont.label, ButtonPressMask, FALSE,
                    (XtEventHandler)getcoord_event, (XtPointer)NULL);
  XtAddEventHandler(x11cont.frame, ButtonPressMask, FALSE,
                    (XtEventHandler)getcoord_event, (XtPointer)NULL);
  return 0;
}

/* application main loop */
int gpmwin_mainloop(){
  signal(SIGTERM, (sigfunc_t)sig_destroy);
  signal(SIGFPE,  (sigfunc_t)sig_destroy);
  signal(SIGBUS,  (sigfunc_t)sig_destroy);
  signal(SIGHUP,  (sigfunc_t)sig_destroy);
  signal(SIGUSR1, SIG_IGN);
  signal(SIGQUIT, SIG_IGN);
  signal(SIGINT,  SIG_IGN);
  XtAppMainLoop(x11cont.app_con);
  return 0;
}

/* window application termination */
int gpmwin_quit(){
  program_exit();
  return 0;
}

/* get window attributes */
int gpmwin_getwattr(int *w, int *h){
  XWindowAttributes attr;
  XGetWindowAttributes(x11cont.display, x11cont.window, &attr);
  *w = (int)x11cont.wx;
  *h = (int)x11cont.wy;
  return 0;
}

int gpmwin_flush(){
  unsigned int width, height, ww, wh;
  Window r; int x, y; unsigned int bw,d;
  XGetGeometry(x11cont.display, x11cont.window, &r, &x, &y, &ww, &wh, &bw, &d);
  width  = (ww < x11cont.wx) ? ww: x11cont.wx;
  height = (wh < x11cont.wy) ? wh: x11cont.wy;
  XCopyArea(x11cont.display, x11cont.pixmap, x11cont.window, x11cont.gc,
            0, 0, width, height, 0, 0);
  return 0;
}

int gpmwin_erase(){
  XSetForeground(x11cont.display, x11cont.gc, x11cont.bgcolor);
  XFillRectangle(x11cont.display, x11cont.pixmap, x11cont.gc, 0, 0, 
                 x11cont.wx, x11cont.wy);
  gpmwin_flush();
  return 0;
}

int gpmwin_resize(const char *title, int w, int h){
  XSetForeground(x11cont.display, x11cont.gc, x11cont.gray_color);
  XFillRectangle(x11cont.display, x11cont.window, x11cont.gc, 0, 0, 
                 x11cont.wx, x11cont.wy);
  XFreePixmap(x11cont.display, x11cont.pixmap);
  create_frame_pixmap(w, h);
  /* TODO: resize window */
  XStoreName(x11cont.display, XtWindow(x11cont.toplevel), title);
  gpmwin_flush();
  return 0;
}

int gpmwin_setcolor(int color, int rainbow){
  unsigned long c;
  static   int  RainbowMax = RAINBOW_DIVNUM * RAINBOW_BASECOL+2;
  double level;
  if(x11cont.depth > 1 ){
    if(color < 0 || rainbow >= 0){
      if(rainbow >= 0 && rainbow < RainbowMax){
        if(x11cont.depth >= 8){
          c = x11cont.rainbow[rainbow];
        }else{
          rainbow /= (RAINBOW_DIVNUM);
          c = x11cont.rainbow[rainbow];
        }
      }else{
        c = (rainbow < 0) ? x11cont.rainbow[0] : x11cont.rainbow[RainbowMax-1];
      }
    }else{
      c = x11cont.color[color%MAX_COLOR];
    }
    XSetForeground(x11cont.display, x11cont.gc, c);
    XSetBackground(x11cont.display, x11cont.gc, x11cont.bgcolor);
    XSetFillStyle(x11cont.display, x11cont.gc, FillSolid);
  }else{
    if(color < 0){
      level = (double)rainbow / (double)RainbowMax;
      set_tile_pattern(x11cont.window, x11cont.gc, level);
    }else{
      XSetForeground(x11cont.display, x11cont.gc,
                     (color!=0) ? x11cont.bgcolor: x11cont.fgcolor);
      XSetBackground(x11cont.display, x11cont.gc, x11cont.bgcolor);
    }
  }
  return 0;
}

int gpmwin_gradation(int color, int n, double level){
  if(x11cont.depth >= 8){
    XSetForeground(x11cont.display, x11cont.gc, x11cont.gradation[color][n]);
  }else if(x11cont.depth == 4){
    n /= RAINBOW_DIVNUM;
    XSetForeground(x11cont.display, x11cont.gc, x11cont.gradation[color][n]);
  }else{
    set_tile_pattern(x11cont.window, x11cont.gc, level);
  }
  return 0;
}

int gpmwin_setdash(int itype, int isize, int imode){
  static char dot1[] = { 4, 4 };
  static char dot2[] = { 2, 2, 6, 2 };
  static char dot3[] = { 2, 2, 2, 2, 6, 2 };
  static char dot4[] = { 2, 2 };
  static char dot5[] = { 1, 3 };
  static char dot6[] = { 1, 7 };
  static char dot7[] = { 1, 1 };
  static char *dashtbl[] = { dot1, dot2, dot3, dot4, dot5, dot6, dot7 };
  static int   dashlen[] = { 2, 4, 6, 2, 2, 2, 2};
  if(itype > 0 && itype < 8)
    XSetDashes(x11cont.display, x11cont.gc, 0, dashtbl[itype-1], 
               dashlen[itype-1]);
  if(isize >= 0)
    isize *= 2;
  XSetLineAttributes(x11cont.display, x11cont.gc, isize,
                     ((itype==0) ? LineSolid : LineDoubleDash),
                     CapButt, JoinMiter);
  switch(imode){
  case 1:
    XSetFunction(x11cont.display, x11cont.gc, GXxor);
    break;
  case 2:
    XSetFunction(x11cont.display, x11cont.gc, GXclear);
    break;
  case 0:
  default:
    XSetFunction(x11cont.display, x11cont.gc, GXcopy);
    break;
  }
  return 0;
}

int gpmwin_drawrect(int x, int y, int w, int h){
  XDrawRectangle(x11cont.display, x11cont.pixmap, x11cont.gc, x, y, w, h);
  return 0;
}

int gpmwin_fillrect(int x, int y, int w, int h){
  XFillRectangle(x11cont.display, x11cont.pixmap, x11cont.gc, x, y, w, h);
  return 0;
}

int gpmwin_drawpoint(gpm_xpoint_t *ps, int npts){
  int i;
  XPoint pstack[GPM_ENV_MAX_PLOTSTACK];
  for(i=0; i<npts; i++){
    pstack[i].x = ps[i].x;
    pstack[i].y = ps[i].y;
  }
  XDrawPoints(x11cont.display, x11cont.pixmap, x11cont.gc, pstack, npts,
              CoordModeOrigin);
  return 0;
}

int gpmwin_drawline(gpm_xpoint_t *ps, int npts){
  int i;
  XPoint pstack[GPM_ENV_MAX_PLOTSTACK];
  for(i=0; i<npts; i++){
    pstack[i].x = ps[i].x;
    pstack[i].y = ps[i].y;
  }
  XDrawLines(x11cont.display, x11cont.pixmap, x11cont.gc, pstack, npts,
             CoordModeOrigin);
  return 0;
}

int gpmwin_fillpoly(gpm_xpoint_t *ps, int npts){
  int i;
  XPoint pstack[GPM_ENV_MAX_PLOTSTACK];
  for(i=0; i<npts; i++){
    pstack[i].x = ps[i].x;
    pstack[i].y = ps[i].y;
  }
  XFillPolygon(x11cont.display, x11cont.pixmap, x11cont.gc, pstack, npts,
               Complex, CoordModeOrigin);
  return 0;
}

int gpmwin_drawarc(int x, int y, int w, int h, int a1, int a2){
  XDrawArc(x11cont.display, x11cont.pixmap, x11cont.gc, x, y, 
           (unsigned int)w, (unsigned int)h, a1, a2);
  return 0;
}

int gpmwin_fillarc(int x, int y, int w, int h, int a1, int a2){
  XSetArcMode(x11cont.display, x11cont.gc, ArcPieSlice);
  XFillArc(x11cont.display, x11cont.pixmap, x11cont.gc, x, y, 
           (unsigned int)w, (unsigned int)h, a1, a2);
  return 0;
}

/* private functions */

/* program termination */
static void program_exit(){
  exit(0);
}

/* signal handers */
static void sig_destroy(int n){
  program_exit();
}

/* translate table actions */
static void quit_action(Widget w, XEvent *ev, String *prms, Cardinal nprm){
  if((ev->type == ClientMessage) &&
     (ev->xclient.data.l[0] != x11cont.wm_delete_window)){
    XBell(XtDisplay(w), 0);
  }else{
    program_exit();
  }
}

static void erase_action(Widget w, XEvent *ev, String *prms, Cardinal nprm){
  gpmwin_erase();
}

static void expose_action(Widget w, XEvent *ev, String *prms, Cardinal nprm){
  unsigned int width, height, ww, wh;
  Window r; int x, y; unsigned int bw,d;
  /* TODO: get expose window size */
  XGetGeometry(x11cont.display, x11cont.window, &r, &x, &y, &ww, &wh, &bw, &d);
  width  = (ww < x11cont.wx) ? ww: x11cont.wx;
  height = (wh < x11cont.wy) ? wh: x11cont.wy;
  XCopyArea(x11cont.display, x11cont.pixmap, x11cont.window, x11cont.gc,
            0, 0, width, height, 0, 0);
  /* gpmwin_flush(); */
}

/* event callback handlers */
static void readinput_event(XtPointer pt, int *p, XtInputId * id){
  if(gpmwin_comm_command() == 0)
    program_exit();
}

static void exceptinput_event(XtPointer pt, int *p, XtInputId * id){
  fprintf(stderr, "connection closed.\n");
  program_exit();
}

static void dispcoord_event(Widget w, XtPointer cli_data, XEvent *ev){
  char    buf[256];
  double  positionX, positionY;
  libgpm_pixel2mm(paper, &positionX, &positionY, ev->xmotion.x, ev->xmotion.y);
  positionY = height - positionY;
  if(positionY < 0.0) positionY = 0.0;
  sprintf(buf, "(%5.1f, %5.1f)\n", positionX, positionY); /* safe */
  XtVaSetValues(x11cont.label, "label", buf, NULL);
}

static void getcoord_event(Widget w, XtPointer cli_data, XEvent *ev){
  double  positionX, positionY;
  libgpm_pixel2mm(paper, &positionX, &positionY, ev->xmotion.x, ev->xmotion.y);
  positionY = height - positionY;
  if(positionY < 0.0) positionY = 0.0;
  sprintf(x11cont.clipbord, "(%5.1f, %5.1f)", positionX, positionY); /* safe */
  XtOwnSelection(w, XA_PRIMARY, ev->xbutton.time,
                 (XtConvertSelectionProc) sel_convert,
                 (XtLoseSelectionProc)    sel_lose,
                 (XtSelectionDoneProc)    sel_done);
}

/* select callback handlers */
static Bool sel_convert(Widget w, Atom *selection, Atom *target, Atom *type, 
                        XtPointer *value, unsigned long *length, int *format){
  *type   = XA_STRING;
  *value  = x11cont.clipbord;
  *length = strlen(x11cont.clipbord);
  *format = 8;
  return True;
}
static void sel_lose(){}
static void sel_done(){}

/* color allocation */
static void color_allocation(){
  int i;
  unsigned long white, black;
  XColor c,cc;
  static struct {
    int num;  const char *name;
  } named_color[] = {
    {GPM_COLOR_BLUE,   "Blue"},    {GPM_COLOR_RED,    "Red"},
    {GPM_COLOR_MAGENTA,"Magenta"}, {GPM_COLOR_GREEN,  "Green"},
    {GPM_COLOR_CYAN,   "Cyan"},    {GPM_COLOR_YELLOW, "Yellow"},
    {GPM_COLOR_GRAY1,  "Gray80"},  {GPM_COLOR_GRAY2,  "Gray60"},
    {GPM_COLOR_GRAY3,  "Gray30"},  {-1, NULL}
  };

  black = BlackPixel(x11cont.display, x11cont.screen);
  white = WhitePixel(x11cont.display, x11cont.screen);

  /* named color allocation */
  if(x11cont.depth == 1){
    for(i=1; i<GPM_COLOR_SIZE; i++)
      x11cont.color[i] = black;
    x11cont.color[GPM_COLOR_WHITE] = white;
  }else{
    x11cont.color[GPM_COLOR_BLACK] = black;
    x11cont.color[GPM_COLOR_WHITE] = white;
    for(i=0; named_color[i].name != NULL; i++){
      if(XAllocNamedColor(x11cont.display, x11cont.cmap, 
                          named_color[i].name, &c, &cc)){
        x11cont.color[named_color[i].num] = c.pixel;
      }else{
        if(XAllocColorCells(x11cont.display, x11cont.cmap, FALSE, NULL, 0, 
                            &x11cont.color[named_color[i].num], 1)){
          XStoreNamedColor(x11cont.display, x11cont.cmap, named_color[i].name, 
                           x11cont.color[named_color[i].num], 
                           DoRed | DoGreen | DoBlue);
        }else{
          x11cont.color[i] = BlackPixel(x11cont.display, x11cont.screen);
        }
      }
    }
  }
  /* rainbow color allocation */
  color_alloc_rainbow();
  /* gradation color allocation */
  color_alloc_gradation();
}

static void color_alloc_rainbow(){
  double  r, g, b, level;
  int     i, n, maxcols;
  int     div = GPM_RAINBOW_DIVNUM;
  int     basecol = GPM_RAINBOW_BASECOL;
  double  rbasecol = 1.0 / basecol;
  maxcols = div * basecol;
  n = 1;
  if(x11cont.depth >= 8){
    for(i=0; i < div; i++, n++){
      level = (double)n / (double)maxcols;
      r = 0.0;  g = 0.0; b = level/rbasecol;
      color_alloc_setcols_rainbow(n-1, r, g, b);
    }
    for(i=0; i < div; i++, n++){
      level = (double)n / (double)maxcols;
      r = 0.0;  g = (level-rbasecol)/rbasecol; b = 1.0;
      color_alloc_setcols_rainbow(n-1, r, g, b);
    }
    for(i=0; i < div; i++, n++){
      level = (double)n / (double)maxcols;
      r = 0.0; g = 1.0; b = 1.0 - (level-rbasecol*2.0)/rbasecol;
      color_alloc_setcols_rainbow(n-1, r, g, b);
    }
    for(i=0; i < div; i++, n++){
      level = (double)n / (double)maxcols;
      r = (level-rbasecol*3.0)/rbasecol; g = 1.0;  b = 0.0;
      color_alloc_setcols_rainbow(n-1, r, g, b);
    }
    for(i=0; i <= div; i++, n++){
      level = (double)n / (double)maxcols;
      r = 1.0;  g = 1.0 - (level-rbasecol*4.0)/rbasecol;  b = 0.0;
      color_alloc_setcols_rainbow(n-1, r, g, b);
    }
    r = g = b = 1.0;
    color_alloc_setcols_rainbow(n-1, r, g, b);
  }else{
    x11cont.rainbow[0] = x11cont.color[GPM_COLOR_BLUE];
    x11cont.rainbow[1] = x11cont.color[GPM_COLOR_CYAN];
    x11cont.rainbow[2] = x11cont.color[GPM_COLOR_GREEN];
    x11cont.rainbow[3] = x11cont.color[GPM_COLOR_YELLOW];
    x11cont.rainbow[4] = x11cont.color[GPM_COLOR_RED];
    x11cont.rainbow[5] = x11cont.color[GPM_COLOR_RED];
    for(i=6; i<GPM_RAINBOW_BASECOL * GPM_RAINBOW_DIVNUM + 2; i++){
      x11cont.rainbow[i] = x11cont.color[GPM_COLOR_WHITE];
    }
  }
}

static void color_alloc_gradation(){
  int    c, i, n, maxcols;
  double level;
  int    div = GPM_RAINBOW_DIVNUM;
  int    basecol  = GPM_RAINBOW_BASECOL;
  for(c = 0; c < 7; c++){
    if(x11cont.depth > 8){
      maxcols = div * basecol;
      n = 1;
      for(i = 0; i < maxcols; i++, n++){
        level = (double)n / (double)maxcols;
        color_alloc_setcols_gradation(n-1, level, c);
      }
      color_alloc_setcols_gradation(n-1, 1.0, c);
    }else if(x11cont.depth == 8){
      /* TODO: limit number of color allocation */
      maxcols = div * basecol;
      n = 1;
      for(i = 0; i < maxcols; i++, n++){
        level = (double)n / (double)maxcols;
        color_alloc_setcols_gradation(n-1, level, c);
      }
      color_alloc_setcols_gradation(n-1, 1.0, c);
    }else if((x11cont.depth == 4) || (x11cont.depth == 1)){
      x11cont.gradation[c][0] = x11cont.color[1];
      x11cont.gradation[c][1] = x11cont.color[5];
      x11cont.gradation[c][2] = x11cont.color[4];
      x11cont.gradation[c][3] = x11cont.color[6];
      x11cont.gradation[c][4] = x11cont.color[2];
      x11cont.gradation[c][5] = x11cont.color[2];
    }
  }
}


static void color_alloc_setcols_rainbow(int n, double r, double g, double b){
  XColor rainbow;
  rainbow.red   = ( r > 0.0 ) ? 65535 * sqrt(r) : 0;
  rainbow.green = ( g > 0.0 ) ? 65535 * sqrt(g) : 0;
  rainbow.blue  = ( b > 0.0 ) ? 65535 * sqrt(b) : 0;
  rainbow.flags = DoRed | DoGreen | DoBlue;
  XAllocColor(x11cont.display, x11cont.cmap, &rainbow);
  x11cont.rainbow[n] = rainbow.pixel;
}

static void color_alloc_setcols_gradation(int n, double x, int c){
  int    status;
  XColor rainbow;
  static int t[3][7] = {
    { 1, 0, 1, 0, 1, 0, 1 },
    { 0, 1, 1, 0, 0, 1, 1 },
    { 0, 0, 0, 1, 1, 1, 1 }};
  x = (x > 0.0) ?  65535 * sqrt(x) : 0;
  rainbow.blue  = t[0][c] ? x : 0 ;
  rainbow.red   = t[1][c] ? x : 0 ;
  rainbow.green = t[2][c] ? x : 0 ;
  rainbow.flags = DoRed | DoGreen | DoBlue;
  status = XAllocColor(x11cont.display, x11cont.cmap, &rainbow);
  x11cont.gradation[c][n] = rainbow.pixel;
}

static void set_tile_pattern(Drawable d, GC gc, double level){
  Pixmap bitmap;
  char  *bits;
  int    level_num;
  static char col_0[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  static char col_1[] = {0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00};
  static char col_2[] = {0x11, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00};
  static char col_3[] = {0x11, 0x00, 0x44, 0x00, 0x11, 0x00, 0x44, 0x00};
  static char col_4[] = {0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44};
  static char col_5[] = {0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa};
  static char col_6[] = {0xee, 0xbb, 0xee, 0xbb, 0xee, 0xbb, 0xee, 0xbb};
  static char col_7[] = {0xee, 0xff, 0xbb, 0xff, 0xee, 0xff, 0xbb, 0xff};
  static char col_8[] = {0xee, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff, 0xff};
  static char col_9[] = {0xfe, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff};
  static char col_a[] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
  static char * col_table[] = {
    col_0, col_1, col_2, col_3, col_4, col_5, 
    col_6, col_7, col_8, col_9, col_a
  };
  static Pixmap pixmap_table[16] = {
    (Pixmap)0,(Pixmap)0,(Pixmap)0,(Pixmap)0,
    (Pixmap)0,(Pixmap)0,(Pixmap)0,(Pixmap)0,
    (Pixmap)0,(Pixmap)0,(Pixmap)0,(Pixmap)0,
    (Pixmap)0,(Pixmap)0,(Pixmap)0,(Pixmap)0
  };

  if(level < 0.0){
    level_num =  0;
  }else if(level > 1.0){
    level_num = 10;
  }else{
    level_num = (int) (level * 11);
    if (level_num == 11) level_num = 10;
  }

  if(pixmap_table[level_num] == (Pixmap)0){
    bits = *(col_table + level_num);
    bitmap = XCreateBitmapFromData(x11cont.display, d, bits, 8, 8);
    pixmap_table [level_num] = bitmap;
  }else{
    bitmap = pixmap_table[level_num];
  }

  XSetStipple(x11cont.display, gc, bitmap);
  XSetFillStyle(x11cont.display, gc, FillOpaqueStippled);
  XSetForeground(x11cont.display, gc, x11cont.fgcolor);
  XSetBackground(x11cont.display, gc, x11cont.bgcolor);
}

#ifdef __cplusplus
}
#endif

#endif /* WIN32 */
