/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/gpmwin/main.c,v 1.14 2004/08/11 13:58:02 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_FCNTL_T
# include <fcntl.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif

#include "libgpm.h"
#include "gpmwinpriv.h"

#define GPMWIN_TITLE_BSIZE 128

#ifdef __cplusplus
extern "C" {
#endif

int    wnum, paper, orient;
double width, height;

static const char *paper_name[]  = {"A4", "B4", "FREE"};
static const char *orient_name[] = {"Portrait", "Landscape"};
static const char  title_fmt[] = "GPM Window No. %d (%s %s - %5.1fx%5.1f)";

static int  gpmwin_window_properties_get();

#ifdef WIN32
int main(int argc, char *argv[]);
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine, int nCmdShow){
  return main(1, NULL);
}
#endif

int main(int argc, char *argv[]){
  int    code;
  int    px, py;
  char   title[GPMWIN_TITLE_BSIZE];

  if(libgpm_context_init(NULL) != 0)
    return 999; /* TODO errono assignment */

  /* get window size and window number */
  code = gpmwin_window_properties_get();
  if(code != 0){
    libgpm_comm_send_int(code); return code;
  }
  /* create widnow title */
  snprintf(title, GPMWIN_TITLE_BSIZE, title_fmt, wnum, paper_name[paper],
           orient_name[orient], width, height);

  /* initialize window */
  code = gpmwin_init(argc, argv);
  if(code != 0){
    libgpm_comm_send_int(code); return code;
  }

  /* create window */
  libgpm_mm2pixel(paper, &px, &py, width, height);

  code = gpmwin_create(px, py, title);
  if(code != 0){
    libgpm_comm_send_int(code); return code;
  }
  
  /* success */
  code = 0;
  libgpm_comm_send_int(code);

  return gpmwin_mainloop();
}

static int gpmwin_window_properties_get(){
  if(libgpm_comm_recive_int(&wnum) != 0)
    return -1;
  if(libgpm_comm_recive_int(&paper) != 0)
    return -1;
  if(libgpm_comm_recive_int(&orient) != 0)
    return -1;
  if(libgpm_comm_recive_float(&height) != 0)
    return -1;
  if(libgpm_comm_recive_float(&width) != 0)
    return -1;
  return 0;
}

int gpmwin_comm_command(){
  int c;
  int color, rainbow;      /* for set color */
  int itype, isize, imode; /* for set dash with dmode */
  int x1, x2, y1, y2;      /* for draw rectangle and fill rectangle */
  gpm_xpoint_t pstack[GPM_ENV_MAX_PLOTSTACK];
  int i, npts, x, y;  /* for draw points, draw lines, draw polygon */
  int w, h, a1, a2; /* for draw arc, fill arc */
  int n;            /* for gradation */
  double level;
  char buf[GPMWIN_TITLE_BSIZE];
  if(libgpm_comm_recive_int(&c) != 0)
    return 0;
  switch(c){
  case GPM_PROTO_CLOSE_WINDOW:
    gpmwin_quit();
    return 0;
    break;
  case GPM_PROTO_GET_WINDOW_ATTRIBUTES:
    gpmwin_getwattr(&x, &y);
    if(libgpm_comm_send_int(x) != 0) return 0;
    if(libgpm_comm_send_int(y) != 0) return 0;
    break;
  case GPM_PROTO_FLUSH_WINDOW:
    gpmwin_flush();
    if(libgpm_comm_send_int(0) != 0) return 0;
    break;
  case GPM_PROTO_RESIZE_WINDOW:
    if(libgpm_comm_recive_int(&paper) != 0) return 0;
    if(libgpm_comm_recive_int(&orient) != 0) return 0;
    if(libgpm_comm_recive_float(&width) != 0) return 0;
    if(libgpm_comm_recive_float(&height) != 0) return 0;
    libgpm_mm2pixel(paper, &x, &y, width, height);
    snprintf(buf, GPMWIN_TITLE_BSIZE, title_fmt, wnum, paper_name[paper],
             orient_name[orient], width, height);
    gpmwin_resize(buf, x, y);
    if(libgpm_comm_send_int(0) != 0) return 0;
    break;
  case GPM_PROTO_CLEAR_WINDOW:
    gpmwin_erase();
    break;
  case GPM_PROTO_SET_COLOR:
    if(libgpm_comm_recive_int(&color) != 0) return 0;
    if(libgpm_comm_recive_int(&rainbow) != 0) return 0;
    gpmwin_setcolor(color, rainbow);
    break;
  case GPM_PROTO_SET_GRADATION:
    if(libgpm_comm_recive_int(&color) != 0) return 0;
    if(libgpm_comm_recive_int(&n) != 0) return 0;
    if(libgpm_comm_recive_float(&level) != 0) return 0;
    gpmwin_gradation(color, n, level);
    break;
  case GPM_PROTO_SET_DASH_WIDTH_DMODE:
    if(libgpm_comm_recive_int(&itype) != 0) return 0;
    if(libgpm_comm_recive_int(&isize) != 0) return 0;
    if(libgpm_comm_recive_int(&imode) != 0) return 0;
    gpmwin_setdash(itype, isize, imode);
    break;
  case GPM_PROTO_DRAW_POINTS:
    if(libgpm_comm_recive_int(&npts) != 0) return 0;
    for(i=0; i<npts; i++){
      if(libgpm_comm_recive_int(&x) != 0) return 0;
      if(libgpm_comm_recive_int(&y) != 0) return 0;
      pstack[i].x = x; pstack[i].y = y;
    }
    gpmwin_drawpoint(pstack, npts);
    break;
  case GPM_PROTO_DRAW_LINES:
    if(libgpm_comm_recive_int(&npts) != 0) return 0;
    for(i=0; i<npts; i++){
      if(libgpm_comm_recive_int(&x) != 0) return 0;
      if(libgpm_comm_recive_int(&y) != 0) return 0;
      pstack[i].x = x; pstack[i].y = y;
    }
    gpmwin_drawline(pstack, npts);
    break;
  case GPM_PROTO_DRAW_RECTANGLE:
    if(libgpm_comm_recive_int(&x1) != 0) return 0;
    if(libgpm_comm_recive_int(&y1) != 0) return 0;
    if(libgpm_comm_recive_int(&x2) != 0) return 0;
    if(libgpm_comm_recive_int(&y2) != 0) return 0;
    gpmwin_drawrect(x1, y1, x2, y2);
    break;
  case GPM_PROTO_FILL_RECTANGLE:
    if(libgpm_comm_recive_int(&x1) != 0) return 0;
    if(libgpm_comm_recive_int(&y1) != 0) return 0;
    if(libgpm_comm_recive_int(&x2) != 0) return 0;
    if(libgpm_comm_recive_int(&y2) != 0) return 0;
    gpmwin_fillrect(x1, y1, x2, y2);
    break;
  case GPM_PROTO_FILL_POLYGON:
    if(libgpm_comm_recive_int(&npts) != 0) return 0;
    for(i=0; i<npts; i++){
      if(libgpm_comm_recive_int(&x) != 0) return 0;
      if(libgpm_comm_recive_int(&y) != 0) return 0;
      pstack[i].x = x; pstack[i].y = y;
    }
    gpmwin_fillpoly(pstack, npts);
    break;
  case GPM_PROTO_DRAW_ARC:
    if(libgpm_comm_recive_int(&x) != 0) return 0;
    if(libgpm_comm_recive_int(&y) != 0) return 0;
    if(libgpm_comm_recive_int(&w) != 0) return 0;
    if(libgpm_comm_recive_int(&h) != 0) return 0;
    if(libgpm_comm_recive_int(&a1) != 0) return 0;
    if(libgpm_comm_recive_int(&a2) != 0) return 0;
    gpmwin_drawarc(x, y, w, h, a1, a2);
    break;
  case GPM_PROTO_FILL_ARC:
    if(libgpm_comm_recive_int(&x) != 0) return 0;
    if(libgpm_comm_recive_int(&y) != 0) return 0;
    if(libgpm_comm_recive_int(&w) != 0) return 0;
    if(libgpm_comm_recive_int(&h) != 0) return 0;
    if(libgpm_comm_recive_int(&a1) != 0) return 0;
    if(libgpm_comm_recive_int(&a2) != 0) return 0;
    gpmwin_fillarc(x, y, w, h, a1, a2);
    break;
  default:
    fprintf(stderr, "unknown command no:%d\r\n", c);
  }
  return 1;
}

static int create_window_title(char *buf, size_t bsize){
  return 0;
}

#ifdef __cplusplus
}
#endif
