/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/command/map.cpp,v 1.6 2004/09/09 10:45:13 orrisroot Exp $ */

/**************************************************************
        MAP GRAPHIC COMMAND  -New Version-
***************************************************************
       map(buf, dir, mode, set, vp)

       buf         : BUFFER
       dir         : DRAWING DIRECTION
       mode        : MAP STYLEE
       set         : TILE SIZE
       vp          : VIEWPOINT
***************************************************************
        T.KOBAYASHI (Buffer Version      ) 90/09/05
        M.Shirakawa (PS Version          ) 92/01/31
        S.Hitomi    (on SATELLITE Language) 92/11/17
        K.Takebe    (Extended Version    ) 94/03/03
**************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "libgpm.h"
#include "interval.h"

#define GFLUSH_INTERVAL 5

#ifdef __cplusplus
extern "C" {
#endif

/* in gpmx1.c */
extern void set_tile_pattern (/* display, screen, window, gc, level */);

/* private functions */
static double norm(double data, double amin, double amax);
static int    draw_box_(int ix, int iy, double size, double hx, double hy, 
                        int fillmode, double level);
static void   gradation(int c, double level);

DLLEXPORT int mod_gpm_map(){
  Buffer **data, *indat, *tmp;
  int   wait_time = 500, startp = 0, endp = 0, step = 1, dviflag = 0;
  double a, rmin, rmax, min, max, hx, hy, level, size;
  int   dim, index[MAX_INDEX];
  int   tnum, dnum, viewpoint, flag, dmode, xyset;
  int   nx, ny, subsize, fillmode;
  int   j, k, p1, p2;
  register int i, ix, iy;
  char  *timeaxis, rest_device;
  
  /* LOAD SYSTEM PARAMETER */
  regpm();

  indat     = (Buffer*)GetSeries(0,&dim,index);
  timeaxis  =       GetString(1);
  dmode     = (int) GetScalar(2);
  xyset     = (int) GetScalar(3);
  viewpoint = ((int)GetScalar(4) > 0) ? 1 : -1;
  
  if(GetArgNum() > 5){
    wait_time = (int) GetScalar(5);
    startp    = (int) GetScalar(6);
    endp      = (int) GetScalar(7);
    step      = (int) GetScalar(8);
  }

  /* Load Data */
  if(indat == NULL || dim < 2 || dim > 3)
    return 14; /* data dimension mismatch */ 

  if(timeaxis == NULL)
    return 9; /* illegal axis type */

  dnum = IndexSize(dim, index);
  tnum = (dim == 3) ? index[0] : 1;
  data = (Buffer**)emalloc(tnum*sizeof(Buffer*));
  if(data == NULL)
    return 2; /* out of memory */

  if(*timeaxis == 'X' || *timeaxis == 'x'){
    nx = index[dim-2];
    ny = index[dim-1];
    p1 = ny;
    p2 = 1;
  }else{
    nx = index[dim-1];
    ny = index[dim-2];
    p1 = 1;
    p2 = nx;
  }

  subsize = index[dim-2]*index[dim-1];
  for(tmp = indat, i = 0; i < tnum; i++, tmp += subsize)
    data[i] = tmp;

  if(dim == 3){
    if(startp >= tnum){
      startp = tnum-1;
    }else{
      startp = (startp < 0) ? 0 : startp;
    }
    if(endp >= tnum){
      endp   = tnum-1;
    }else{
      endp = (endp <= 0) ? tnum : endp;
      endp = (startp >= endp) ? startp+1 : endp;
    }
    if(step <= 0){
      step = 1;
    }else{
      step = (step > endp-startp) ? endp-startp : step;
    }
  }else{
    startp = 0;
    endp   = 1;
    step   = 1;
  }
  
  dviflag = checktodvi();

  rest_device = GpmCont.device;
  if(dim != 2)
    GpmCont.device = 0;
  
  if(GpmCont.xMode == 0){
    GpmCont.xMin = 0.0;
    GpmCont.xMax = (float)(nx-1);
  }

  if(GpmCont.yMode == 0){
    GpmCont.yMin = 0.0;
    GpmCont.yMax = (float)(ny-1);
  }
  GpmCont.axisType = 0;

  min = indat[0];
  max = indat[0];
  for(i = 0; i < dnum; i++){
    if(indat[i] > max) max = indat[i];
    if(indat[i] < min) min = indat[i];
  }

  if(GpmCont.zMode == 0){
    rmin = min;
    GpmCont.zMin = (float)rmin;
    rmax = max;
    GpmCont.zMax = (float)rmax;
  }else{
    rmin = GpmCont.zMin;
    rmax = GpmCont.zMax;
/*
    if(GpmCont.zType == 1){
      rmin = (rmin > 0.0)? log10(rmin) : 0.0;
      rmax = (rmax > 0.0)? log10(rmax) : 0.0;
    }
*/
  }
  
  hx = GpmCont.xSize / (double)nx;
  hy = GpmCont.ySize / (double)ny;
  
  if(xyset == 1){
    if(hx > hy)
      hx = hy;
    else
      hy = hx;
  }

  if(rmin == rmax){
    rmin -= 0.5;
    rmax += 0.5;
  }

  printf("  Matrix Size  %d x %d\n", nx, ny);
  printf("  (scale: min = %g, max = %g, data: min = %g, max = %g)\n",
         rmin, rmax, min, max);
  
  if(gopen(GpmCont.paper, GpmCont.orientation, GpmCont.device, 
           GpmCont.paperWidth, GpmCont.paperHeight) != 0){
    efree(data);
    return 6; /* window does not exist */
  }
  if(dviflag != 0)
    libgpm_dvi_write_param(&gpmdev_cont.dvi, GpmCont.factor, 
                           GpmCont.xOrigin, GpmCont.yOrigin);
  gpen(GpmCont.gLineWidth, GpmCont.gLineType, 0);
  gnewpen(GpmCont.gColor);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfactor(GpmCont.factor);

  for(j = startp; j < endp; j+=step){
    tmp = data[j];
    if(dim == 3){
      if(j + step > endp)
        GpmCont.device = rest_device;
      gpmcmd_interval_store_start_time();
    }
    if(dmode > 0 && dmode < 8){
      gnewpen(GPM_COLOR_WHITE);
      gbox(0, 0, GpmCont.xSize, GpmCont.ySize, 3, -1.0);
      gnewpen(GpmCont.gColor);
    }

    switch(dmode){
    case 1:/* size */
      for(ix = 0; ix < nx; ix++){
        for(iy = 0; iy < ny; iy++){
          i = ix*p1 + iy*p2;
          a = tmp[i];
          if(a < 0.0){
            gnewpen(GpmCont.fColor);
            size = norm(a,0.0,rmin);
          }else{
            gnewpen(GpmCont.gColor);
            size = norm(a,0.0, rmax);
          }
          if(size <= 0.0) continue;
          if(size >  1.0) size = 1.0;
          k = (viewpoint > 0) ? iy : (ny - iy -1);
          level = size;
          fillmode = 3;
          draw_box_(ix, k, size, hx, hy, fillmode, level);
        }
        if((ix % GFLUSH_INTERVAL) == 0) gflush();
      }
      break;

    case 2:/* size & color */
      for(ix = 0; ix < nx; ix++){
        for(iy = 0; iy < ny; iy++){
          i = ix*p1 + iy*p2;
          a = tmp[i];
          if(a < 0.0){
            size = norm(a,0.0, rmin);
          }else{
            size = norm(a,0.0, rmax);
          }
          if(size <= 0.0) continue;
          if(size >  1.0) size = 1.0;
          k = (viewpoint > 0) ? iy : ny - iy -1;
          level = norm(a, rmin, rmax);
          grainbow(level);
          fillmode = 2;
          draw_box_(ix, k, size, hx, hy, fillmode, level);
        }
        if((ix % GFLUSH_INTERVAL) == 0) gflush();
      }
      break;
        
    case 3:/* normalized size */
    case 4:/* normalized size (inverse) */
      for(ix = 0; ix < nx; ix++){
        for(iy = 0; iy < ny; iy++){
          i = ix*p1 + iy*p2;
          a = tmp[i];
          size = level = norm(a, rmin, rmax);
          if(size <= 0.0) continue;
          if(size >  1.0) size = 1.0;
          if(dmode == 3) fillmode = 2;
          else             fillmode = 1;
          k = (viewpoint > 0) ? iy : ny - iy -1;
          level = size;
          draw_box_(ix, k, size, hx, hy, fillmode, level);
        }
        if((ix % GFLUSH_INTERVAL) == 0) gflush();
      }
      break;
        
    case 5:/* normalized size & color */
      for(ix = 0; ix < nx; ix++){
        for(iy = 0; iy < ny; iy++){
          i = ix*p1 + iy*p2;
          a = tmp[i];
          size = level = norm(a, rmin, rmax);
          if(size <= 0.0) continue;
          if(size >  1.0) size = 1.0;
          grainbow(level);
          fillmode = 2;
          k = (viewpoint > 0) ? iy : ny - iy -1;
          level = size;
          draw_box_(ix, k, size, hx, hy, fillmode, level);
        }
        if((ix % GFLUSH_INTERVAL) == 0) gflush();
      }
      break;
        
    case 6: /* data>0: open,   data<0: fill */
    case 7:/* data>0: fill,   data<0: open */
      for(ix = 0; ix < nx; ix++){
        for(iy = 0; iy < ny; iy++){
          i = ix*p1 + iy*p2;
          a = tmp[i];
          if(a < 0.0){
            flag = 1;
            size = norm(a, 0.0, rmin);
          }else{
            flag = 0;
            size = norm(a, 0.0, rmax);
          }
          if(size <= 0.0) continue;
          if(size >  1.0) size = 1.0;
          level = size;
          
          gnewpen(GpmCont.gColor);
          fillmode = 1;
          if(dmode == 6){
            if(flag == 1) fillmode = 3;
          }else{
            if(flag == 0) fillmode = 3;
          }
          k = (viewpoint > 0) ? iy : (ny - iy -1);
          draw_box_(ix, k, size, hx, hy, fillmode, level);
        }
        if((ix % GFLUSH_INTERVAL) == 0) gflush();
      }
      break;
      
    case 0: /* Rainbow Color */
      for(ix = 0; ix < nx; ix++){
        for(iy = 0; iy < ny; iy++){
          i = ix*p1 + iy*p2;
          a = tmp[i];
          level = norm(a, rmin, rmax);
          grainbow(level);
          size = 1.0;
          fillmode = 2;
          k = (viewpoint > 0) ? iy : ny - iy -1;
          draw_box_(ix, k, size, hx, hy, fillmode, level);
        }
        if((ix % GFLUSH_INTERVAL) == 0) gflush();
      }
      break;
      
    default: /* Gradation */
      for(ix = 0; ix < nx; ix++){
        for(iy = 0; iy < ny; iy++){
          i = ix*p1 + iy*p2;
          a = tmp[i];
          level = norm(a, rmin, rmax);
          ggradation(dmode-8, level);
          size = 1.0;
          fillmode = 2;
          k = (viewpoint > 0) ? iy : ny - iy -1;
          draw_box_(ix, k, size, hx, hy, fillmode, level);
        }
        if((ix % GFLUSH_INTERVAL) == 0) gflush();
      }
      break;
    }
    if(dim == 3){
      printf("\r[%6d]",j);
      gpmcmd_interval_wait_msec(wait_time);
    }
  }

  if(dim == 3)
    printf("\n");

  efree(data);

  gflush();
  gclose();
  GpmCont.device = rest_device;
  wrgpm();
  return 0;
}

static double norm(double data, double amin, double amax){
  double level;
  level = (data - amin) / (amax - amin);
  if(level > 1.0)
    level = 1.0;
  else if(level < 0.0)
    level = 0.0;
  return level;
}

static int draw_box_(int ix, int iy, double size, double hx, double hy, 
                     int fillmode, double level){
  double x, y, lx, ly;
  x = ((double)ix + 0.5 - size*0.5) * hx;
  y = ((double)iy + 0.5 - size*0.5) * hy;
  lx = hx * size;
  ly = hy * size;
  gbox(x, y, x + lx, y + ly, fillmode, level);
  return 0;
}

#ifdef __cplusplus
}
#endif
