/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/command/graph_sub.cpp,v 1.5 2004/09/09 10:45:13 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "libgpm.h"
#include "graph_sub.h"

#ifdef __cplusplus
extern "C" {
#endif

static int  _gpmcmd_convert_to_log(float **data, int n, int m, int is);
static void _gpmcmd_scale(float **data, float *amin, float *amax,
                          int n, int m);
static void _gpmcmd_scalg(float **data, float *amin, float *amax,
                          int n, int m, int is);

/* zero at 0 data point : return value 0 or 1 */
int gpmcmd_zerop(int n_series, float **data){
  int          is = 0;
  register int i;
  for(i = 0; i < n_series; i++){
    if(data[i][0] == 0.0){
      is = 1;
      break;
    }
  }
  return is;
}

void gpmcmd_set_xscale(float **xdata, int n, int m, int is){
  /*********** Set X-Scale ***********/
  if(GpmCont.axisType == GPM_AXIS_TYPE_RR || 
     GpmCont.axisType == GPM_AXIS_TYPE_RI){
    if(GpmCont.xMode == GPM_SCALE_MODE_AUTO)
      _gpmcmd_scale(xdata, &GpmCont.xMin, &GpmCont.xMax, n, m);
  }else{ /* axis_type 1 or 3 */
    _gpmcmd_convert_to_log(xdata, n, m, is);
    if(GpmCont.xMode == GPM_SCALE_MODE_AUTO)
      _gpmcmd_scalg(xdata, &GpmCont.xMin, &GpmCont.xMax, n, m, is);
  }
}

void gpmcmd_set_yscale(float **ydata, int n, int m, int is){
  /*********** Set Y-Scale ***********/
  if(GpmCont.axisType == 0 || GpmCont.axisType == 1){
    if(GpmCont.yMode == 0) /* AUTO MODE */
      _gpmcmd_scale(ydata, &GpmCont.yMin, &GpmCont.yMax, n, m);
  }else{
    _gpmcmd_convert_to_log(ydata, n, m, is);
    if(GpmCont.yMode == 0 ) /* AUTO MODE */
      _gpmcmd_scalg(ydata, &GpmCont.yMin, &GpmCont.yMax, n, m, is);
  }
}

void gpmcmd_set_scale(){
  if(GpmCont.axisType == GPM_AXIS_TYPE_IR || 
     GpmCont.axisType == GPM_AXIS_TYPE_II){
    if(GpmCont.xMin <= 0.0) GpmCont.xMin = (float)1.0e-37;
    if(GpmCont.xMax <= 0.0) GpmCont.xMax = (float)1.0e-37;
  }
  if(GpmCont.axisType == GPM_AXIS_TYPE_RI || 
     GpmCont.axisType == GPM_AXIS_TYPE_II){
      if(GpmCont.yMin <= 0.0) GpmCont.yMin = (float)1.0e-37;
    if(GpmCont.yMax <= 0.0) GpmCont.yMax = (float)1.0e-37;
  }

  switch(GpmCont.axisType){
  case GPM_AXIS_TYPE_RR:
    gscale(GpmCont.xMin, GpmCont.yMin, GpmCont.xMax, GpmCont.yMax);
    break;
  case GPM_AXIS_TYPE_IR:
    gscale(log10((double)fabs(GpmCont.xMin)), GpmCont.yMin,
           log10((double)fabs(GpmCont.xMax)), GpmCont.yMax);
    break;
  case GPM_AXIS_TYPE_RI:
    gscale(GpmCont.xMin, log10(fabs((double)GpmCont.yMin)),
           GpmCont.xMax, log10(fabs((double)GpmCont.yMax)));
    break;
  case GPM_AXIS_TYPE_II:
    gscale(log10(fabs((double)GpmCont.xMin)), 
           log10(fabs((double)GpmCont.yMin)),
           log10(fabs((double)GpmCont.xMax)),
           log10(fabs((double)GpmCont.yMax)));
    break;
  }
}

void gpmcmd_get_draw_scale(float *xMin, float *xMax, float *yMin, float *yMax){
  switch(GpmCont.axisType){
  case GPM_AXIS_TYPE_RR:
  case GPM_AXIS_TYPE_RI:
    *xMin = GpmCont.xMin;
    *xMax = GpmCont.xMax;
    break;
  case GPM_AXIS_TYPE_IR:
  case GPM_AXIS_TYPE_II:
    *xMin = (float)log10((double)fabs(GpmCont.xMin));
    *xMax = (float)log10((double)fabs(GpmCont.xMax));
    break;
  }
  switch(GpmCont.axisType){
  case GPM_AXIS_TYPE_RR:
  case GPM_AXIS_TYPE_IR:
    *yMin = GpmCont.yMin;
    *yMax = GpmCont.yMax;
    break;
  case GPM_AXIS_TYPE_RI:
  case GPM_AXIS_TYPE_II:
    *yMin = (float)log10(fabs((double)GpmCont.yMin));
    *yMax = (float)log10(fabs((double)GpmCont.yMax));
    break;
  }
}


void gpmcmd_set_data(int type, int dim, int *index, Buffer *buf){
  register int   i;
  double         p, delta;
  int            index2[MAX_INDEX];
  int            n = index[0];
  int            siz = IndexSize(dim, index);
  
  switch(type){
  case 'T':
    printf("(Time scale)");
    for(i = 0; i < siz; i++){
      rIndex(i, index2, dim, index);
      p = (double)index2[0];
      buf[i] = 1000.0 / (double)get_sampling() * p;
    }
    break;
  case 'F':
    printf("(Frequency scale)");
    delta = (double)get_sampling() / (double)((n - 1) * 2.0);
    for(i = 0; i < siz; i++){
      rIndex(i, index2, dim, index);
      p = (double)index2[0];
      buf[i] = delta * p;
    }
    break;
  case 'D':
    printf("(Data point scale)");
    for(i = 0; i < siz; i++){
      rIndex(i, index2, dim, index);
      p = (double)index2[0];
      buf[i] = p;
    }
    break;
  default:
    break;
  }
}

void gpmcmd_btof(int dim, int *index, Buffer *buf, float **data){
  register int i, j, k;
  int          index2[MAX_INDEX];
  int          siz = IndexSize(dim, index);
  
  for(i = 0; i < siz; i++){
    rIndex(i, index2, dim, index);
    k = Index(SubIndex(index2), dim - 1, SubIndex(index));
    j = index2[0];
    data[k][j] = (float)buf[i];
  }
}


void gpmcmd_disp_scale_mode(int mode1, int mode2){
  /***** Scale Auto or Manual *****/
  int mode = mode1+2*mode2;
  switch(mode){
  case 1:
    printf(" Fixed Scale[ x ] ... ");
    break;
  case 2:
    printf(" Fixed Scale[ y ] ... ");
    break;
  case 3:
    printf(" Fixed Scale[ x & y ] ... ");
    break;
  case 0:
    printf(" Auto Scaling ... ");
    break;
  }
}

int gpmcmd_open_window(){
  if(gopen(GpmCont.paper, GpmCont.orientation, GpmCont.device, 
           GpmCont.paperWidth, GpmCont.paperHeight) != 0)
    return -1;
  if(checktodvi() != 0)
    libgpm_dvi_write_param(&gpmdev_cont.dvi, GpmCont.factor, 
                           GpmCont.xOrigin, GpmCont.yOrigin);
  gnewpen(GpmCont.gColor);
  gpen(GpmCont.gLineWidth, GpmCont.gLineType, 0);
  gorigin(GpmCont.xOrigin, GpmCont.yOrigin);
  gfsize(GpmCont.xSize, GpmCont.ySize);
  gfactor(GpmCont.factor);
  glimit(0.0, 0.0, GpmCont.xSize, GpmCont.ySize);
  return 0;
}

void gpmcmd_close_window()
{
  glimit(0.0, 0.0, 0.0, 0.0);
  gflush();
  gclose();
}


static int _gpmcmd_convert_to_log(float **data, int n, int m, int is){
  int i, j, erflag = 0;

  for(i = 0; i < n; i++)
    for(j = is; j < m; j++){
      if(data[i][j] > 0.0){
        data[i][j] = (float)log10((double)data[i][j]);
      }else{
        data[i][j] = -38.0;
        erflag = -1;
      }
    }
  if(erflag != 0){
    printf("graph: Warning ... used negative values in log scale mode\n");
  }
  return erflag;
}


static void _gpmcmd_scale(float **data, float *amin, float *amax, 
                          int n, int m){
  register int i, j;
  double dmin, dmax;
  
  dmax = dmin = data[0][0];
  for(i = 0; i < n; i++){
    for(j = 0; j < m; j++){
      if(data[i][j] < dmin)
        dmin = data[i][j];
      else if(data[i][j] > dmax)
        dmax = data[i][j];
    }
  }
  if(dmax == dmin){
    if(dmax == 0.0){
      dmax = 1.0;
      dmin = -1.0;
    }else{
      dmax += 0.05 * fabs(dmax);
      dmin -= 0.05 * fabs(dmin);
    }
  }else{
    dmax += 0.05 * fabs(dmax);
  }
  *amin = (float)dmin;
  *amax = (float)dmax;
}


static void _gpmcmd_scalg(float **data, float *amin, float *amax, 
                          int n, int m, int is){
  register int i, j;
  double       dmin, dmax;
  
  dmax = dmin = data[0][is];
  for(i = 0; i < n; i++)
    for(j = is; j < m; j++){
      if(data[i][j] > dmax)
        dmax = data[i][j];
      else if(data[i][j] < dmin)
        dmin = data[i][j];
    }

  if(dmin < -37.0)
    *amin = (float)1.0e-37;
  else if(dmin >  38.0)
    *amin = (float)1.0e38;
  else
    *amin = (float)pow(10.0,dmin);

  if(dmax < -37.0)
    *amax = (float)1.0e-37;
  else if(dmax >  38.0)
    *amax = (float)1.0e38;
  else
    *amax = (float)pow(10.0,dmax);
}

#ifdef __cplusplus
}
#endif
