/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/command/graph.cpp,v 1.6 2004/08/24 07:03:48 orrisroot Exp $ */

/****************************************************************
*       DRAW GRAPH     For EWS                                  *
*---------------------------------------------------------------*
*     graph( IC1,IC2,I3,I4,I5,I6,F7 )                           *
*       IC1 : y axis data ------------buf_num[0]                *
*       IC2 : x axis data ------------time_freq[1],buf_num[1]   *
*       I3  : xy axis display type----axis_type                 *
*       I4  : display line type-------line_type                 *
*       I5  : I6-1 ten oki hyouji-----increment                 *
*       I6  : center symbol type------symbol_type               *
*       F7  : size  of center symbol--center_symbol             *
*---------------------------------------------------------------*
*            S.Hitomi           DATE : 1992, 12, 1              *
*            K.Takebe           DATE : 1994,  3, 5              *
****************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "libgpm.h"
#include "graph_sub.h"

#ifdef __cplusplus
extern "C" {
#endif

DLLEXPORT int mod_gpm_graph(){
  float **xdata, **ydata;
  int     dimx, indexx[MAX_INDEX], dimy, indexy[MAX_INDEX];
  int     axis_type, line_type, symbol_type, increment;
  int     n_series, nx_series, ny_series;
  char   *str, time_freq_x = '?', time_freq_y = '?';
  float   center_symbol, rleng;
  Buffer *bufx, *bufy;
  int     length, is;
  register int i;
  
  /********* Load system parameter **********/
  regpm();

  /********* Get X-Y Data from SATELLITE Language *********/
  /* Get Y-Data */
  if(strcmp(GetArgType(0), "string") == 0){
    str = (char*)GetString(0);
    if(strlen(str) == 1)
      time_freq_y = *str;
    bufy = NULL;
  }else{
    bufy = (Buffer *)GetSeries(0, &dimy, indexy);
  }
  
  /* Get X-Data */
  if(strcmp(GetArgType(1), "string") == 0){
    str = (char*)GetString(1);
    if(strlen(str) == 1)
      time_freq_x = *str;
    bufx = NULL;
  }else{
    bufx = (Buffer*)GetSeries(1, &dimx, indexx);
  }
  
  if(bufx == NULL && bufy == NULL)
    return 8; /* illegal series object */
  
  /********* Get Parameter from SATELLITE Language *********/
  axis_type     = (int)GetScalar(2);
  line_type     = (int)GetScalar(3);
  increment     = (int)GetScalar(4);
  symbol_type   = (int)GetScalar(5);
  center_symbol = (float)GetScalar(6);
  
  /********** Parameters Check *********/
  if(axis_type < GPM_AXIS_TYPE_RR || axis_type > GPM_AXIS_TYPE_SIZE)
    /* Log or Linear Axis */
    return 9; /* illegal axis type */

  if(axis_type != 0){
    /* if type greater than 4, put on 0 */
    GpmCont.axisType = (char)(axis_type % GPM_AXIS_TYPE_SIZE);
  }
  /* if axis_type = 0, graph will be draw within scale command setting */

  /* Scale Auto or Manual */
  gpmcmd_disp_scale_mode(GpmCont.xMode, GpmCont.yMode);
  
  if(symbol_type < 0 || symbol_type > 16)  /* Set Center Symbol */
    return 10; /* illegal symbol number */

  /* Increment */
  increment = (increment <= 0) ? 1 : (short)increment;
  
  /* Length of Center Symbol */
  rleng = (center_symbol <= 0.0) ? (float)0.25 : center_symbol;
  
  /********** Data Check *********/
  if(bufx == NULL) {
    dimx = dimy;
    CopyIndex(indexx, indexy, dimy);
    bufx = (Buffer *)CAllocBuffer(IndexSize(dimx, indexx));
  }
  if(bufy == NULL) {
    dimy = dimx;
    CopyIndex(indexy, indexx, dimx);
    bufy = (Buffer *)CAllocBuffer(IndexSize(dimy, indexy));
  }
  nx_series = IndexSize(dimx - 1, SubIndex(indexx));
  ny_series = IndexSize(dimy - 1, SubIndex(indexy));
  nx_series = (nx_series == 0) ? 1 : nx_series;
  ny_series = (ny_series == 0) ? 1 : ny_series;
  
  /********* Memory Allocation for FLOAT DATA *********/
  xdata = (float **)emalloc(nx_series * sizeof(float *));
  for(i = 0; i < nx_series; i++)
    xdata[i] = (float *)emalloc(indexx[0] * sizeof(float));
  
  ydata = (float **)emalloc(ny_series * sizeof(float *));
  for(i = 0; i < ny_series; i++)
    ydata[i] = (float *)emalloc(indexy[0] * sizeof(float));
  
  /*********** Set Y-Scale ***********/
  gpmcmd_set_data(time_freq_x, dimx, indexx, bufx);
  gpmcmd_set_data(time_freq_y, dimy, indexy, bufy);
  
  gpmcmd_btof(dimx, indexx, bufx, xdata);
  gpmcmd_btof(dimy, indexy, bufy, ydata);
  
  is = (gpmcmd_zerop(nx_series, xdata) &&
        ((GpmCont.axisType == GPM_AXIS_TYPE_IR) ||
         (GpmCont.axisType == GPM_AXIS_TYPE_II)))
    || (gpmcmd_zerop(ny_series, ydata) &&
        ((GpmCont.axisType == GPM_AXIS_TYPE_RI) ||
         (GpmCont.axisType == GPM_AXIS_TYPE_RR)));
  gpmcmd_set_xscale(xdata, nx_series, indexx[0], is);
  gpmcmd_set_yscale(ydata, ny_series, indexy[0], is);
  
  /**********  GPM Window OPEN ***********/
  if(gpmcmd_open_window() != 0)
    return 6; /* window does not exist */

  gpmcmd_set_scale();
  
  /**********   Drawing Loop   ********/
  n_series = Min(nx_series, ny_series);
  length   = Min(indexx[0], indexy[0]);
  for(i = 0; i < n_series; i++){
    /***** DRAW GRAPH *****/
    ggraph(&xdata[i][is], &ydata[i][is], length - is, (int)increment,
           (int)line_type, (int)symbol_type, rleng);
  }

  /* free allocated memory */
  for(i = 0; i< nx_series; i++)
    efree(xdata[i]);
  for(i = 0; i< ny_series; i++)
    efree(ydata[i]);
  efree(xdata);
  efree(ydata);

  printf("\n");
  gpmcmd_close_window();
  wrgpm();
  return 0;
}


#ifdef __cplusplus
}
#endif
